//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX size value to string or TxPrint
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  24-07-2005 Initial version, split off from TXUTIL

#include <txlib.h>


/*****************************************************************************/
// Format (file) size in 7.1 value + Byte/KiB/MiB/GiB (11); to TxPrint output
/*****************************************************************************/
void txPrtSize64
(
   char               *text,                    // IN    leading string
   LLONG               data,                    // IN    size data
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", txStrSize64( text, string, data, trail));
}                                               // end 'txPrtSize64'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (file) size in 7.1 value + Byte/KiB/MiB/GiB (11); append to string
/*****************************************************************************/
char *txStrSize64                               // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   LLONG               data,                    // IN    size data
   char               *trail                    // IN    trailing text
)
{
   TXTS                form;

   strcat( str, text);
   if (data <= 99999)                           // express as a byte-value
   {
      sprintf( form, "%5lld bytes", data);
   }
   else                                         // use KiB/MiB/GiB/TiB
   {
      double value = ((double) data) / 1024.0;  // start with KiB

      if      (value <= 4999.9)                 // arbitrary, but 7.8 GiB
      {                                         // works out OK then :-)
         sprintf( form, "%7.1lf KiB", value);   // consistent with DFSee
      }
      else
      {
         value /= 1024.0;                       // convert to MiB
         if (value <= 49999.9)
         {
            sprintf(form, "%7.1lf MiB", value);
         }
         else
         {
            value /= 1024.0;                    // convert to GiB
            if (value <= 49999.9)
            {
               sprintf(form, "%7.1lf GiB", value);
            }
            else
            {
               value /= 1024.0;                 // convert to TiB
               sprintf(form, "%7.1lf TiB", value);
            }
         }
      }
   }
   strcat( str, form);
   strcat( str, trail);
   return( str);
}                                               // end 'txStrSize64'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format time in seconds in an HHH:MM:SS string (Elapsed/ETA)
/*****************************************************************************/
char *txStrSec2hms                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               sec,                     // IN    seconds
   char               *trail                    // IN    trailing text
)
{
   TXTS                form;
   ULONG               hours   = (sec / 3600);
   ULONG               minutes = (sec % 3600) / 60;
   ULONG               seconds = (sec % 60);

   strcat( str, text);

   if (hours)
   {
      sprintf( form, "%3u:", hours);
   }
   else
   {
      strcpy(  form, "    ");
   }
   strcat( str, form);

   sprintf( form, "%u:%02u", minutes, seconds);
   if (minutes < 10)
   {
      strcat( str, (hours) ? "0" : " ");
   }
   strcat( str, form);
   strcat( str, trail);
   return( str);
}                                               // end 'txStrSec2hms'
/*---------------------------------------------------------------------------*/

