//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// Trace functions for module entry/exit
//
// Author: J. van Wijk
//
// JvW  26-12-1996 Split-off from ansilog
// JvW  11-10-1999 No timestamp when tracelevel >= 900
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#define DUMP 1                                  // tracecode always available

#include <txlib.h>
#include <txtpriv.h>

        ULONG     TxTrLevel      = 0;           // trace level
        ULONG     TxTrSlowDown   = 0;           // xx ms pause per traceline
        BOOL      TxTrLogOnly    = TRUE;        // default to log only
        BOOL      TxTrTstamp     = FALSE;       // default no timestamp

        ULONG     TxTrIndent[TXTHREADS] = {0};  // indent per thread

static  ULONG     TxTrMainThread = 1;           // TID of main() thread


/*****************************************************************************/
// Init trace level and destinations at startup, to be called from main()
/*****************************************************************************/
void TxTraceInitMain
(
   int                *pargc,                   // INOUT argument count
   char              **pargv[],                 // INOUT array of arg values
   char               *envname,                 // IN    trace env-var name
   char               *prefix                   // IN    tracefile prefix
)
{
   char               *tr;                      // trace specification
   int                 argc = *pargc;
   char              **argv = *pargv;
   char               *s;

   TxTrMainThread = TXTHREADID;                 // remember TID of main()

   if ((          argc     >  1 ) &&            // there is a parameter
       (strlen(   argv[1]) >  2 ) &&            // 3 or more characters
       (         *argv[1] == '-') &&            // and it is a switch
       (isdigit(*(argv[1] +1) ) ) &&            // and starts numerical
       (isdigit(*(argv[1] +2) ) )  )            // with at least two digits
   {
      tr = argv[1] +1;                          // start trace specification

      *pargc = argc -1;                         // shift rest of arguments
      *pargv = argv +1;
   }
   else
   {
      if ((tr = getenv( envname)) != NULL)      // trace in environment ?
      {
         int           i;

         for (i = 1; i < argc; i++)
         {
            if (strncmp(argv[i], "-l", 2) == 0) // there is a -l switch!
            {
               tr = NULL;                       // NO trace now, to avoid
               break;                           // clobbering real logfile
            }
         }
      }
   }
   if (tr != NULL)
   {
      TxTrLevel = atol( tr);                    // numeric trace level
      if (strchr( tr, 's') != NULL)             // screen output too
      {
         TxTrLogOnly  = FALSE;
      }
      if (strchr( tr, 'r') != NULL)             // close/reopen logfile
      {
         TxSetLogReOpen( TRUE);
      }
      if ((s = strchr( tr, 'd')) != NULL)       // 100ms pause per traceline
      {
         if ((TxTrSlowDown = atol( s+1)) == 0)  // or specific value
         {
            TxTrSlowDown = 100;
         }
      }
      if (strchr( tr, 't') != NULL)             // timestamps included
      {
         TxTrTstamp   = TRUE;
      }
      if (strchr( tr, 'l') == NULL)             // auto-log allowed
      {
         TXTM           trcfile;

         sprintf( trcfile, "%s-%u", prefix, TxTrLevel);
         TxAppendToLogFile( trcfile, TRUE);
      }
   }
}                                               // end 'TxTraceInitMain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle entry for a routine
/*****************************************************************************/
void TxTraceEnter
(
   char              *mod                       // IN    module/function name
)
{
   TxTraceLeader();
   TxPrint( "%sEnter func%s : %s%s%s\n", CNM, CNN, CNC, mod, CNN);
   TxTrIndent[ TXTHREADID]++;
}                                               // end 'TxTraceEnter'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle return of a routine, default, 32-bit ULONG value
/*****************************************************************************/
void TxTraceReturn
(
   char              *mod,                      // IN    module/function name
   ULONG              rc                        // IN    return value
)
{
   if (TxTrIndent[TXTHREADID])
   {
      TxTrIndent[TXTHREADID]--;
   }
   TxTraceLeader();
   TxPrint( "%sRet%s:%s% 8.8X %s%s%s\n", CNM, CNN, ((rc) ? CBR : CBG), rc, CNC, mod, CNN);
}                                               // end 'TxTraceReturn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle return of a routine, large, 64-bit ULN64 value
/*****************************************************************************/
void TxTraceRetn64
(
   char              *mod,                      // IN    module/function name
   ULN64              rc                        // IN    return value
)
{
   if (TxTrIndent[TXTHREADID])
   {
      TxTrIndent[TXTHREADID]--;
   }
   TxTraceLeader();
   TxPrint( "%sRet%s:0x%llX %s%s%s\n", CNM, CNN, rc, CNC, mod, CNN);
}                                               // end 'TxTraceRetn64'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle return of a routine, with a POINTER value
/*****************************************************************************/
void TxTraceRetPtr
(
   char              *mod,                      // IN    module/function name
   void              *rc                        // IN    return value
)
{
   if (TxTrIndent[TXTHREADID])
   {
      TxTrIndent[TXTHREADID]--;
   }
   TxTraceLeader();
   TxPrint( "%sRet%s:0x%p %s%s%s\n", CNM, CNN, rc, CNC, mod, CNN);
}                                               // end 'TxTraceRetPtr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle boolean return of a routine
/*****************************************************************************/
void TxTraceRetBool
(
   char              *mod,                      // IN    module/function name
   BOOL               rc                        // IN    return value
)
{
   if (TxTrIndent[TXTHREADID])
   {
      TxTrIndent[TXTHREADID]--;
   }
   TxTraceLeader();
   TxPrint( "%sRet%s:%s%s %s%s%s\n", CNM, CNN, ((rc) ?  CBG       :  CBR),
                                               ((rc) ? "True    " : "False   "),
                                                        CNC,   mod,  CNN);
}                                               // end 'TxTraceRetBool'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle return of a void routine
/*****************************************************************************/
void TxTraceRetVoid
(
   char              *mod                       // IN    module/function name
)
{
   if (TxTrIndent[TXTHREADID])
   {
      TxTrIndent[ TXTHREADID]--;
   }
   TxTraceLeader();
   TxPrint( "%sReturn void%s: %s%s%s\n", CNM, CNN, CNC, mod, CNN);
}                                               // end 'TxTraceRetVoid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Timestamp, thread-id and indent for trace-line
/*****************************************************************************/
void TxTraceLeader
(
   void
)
{
   TXTS                lti = {0};
   TXTT                ltm = {0};
   ULONG               tid = TXTHREADID;

   if (TxTrSlowDown)
   {
      TxSleep( TxTrSlowDown);
   }
   if (TxTrTstamp)
   {
      sprintf( ltm, "%s%12.6lf ", CBZ, TxTmrGetSecondsFromStart());
   }
   if ((tid != TxTrMainThread) || (TxTrTstamp))
   {
      sprintf( lti, "%2.2u ", tid);
   }
   TxPrint( "%s%s%s%*.*s",  ltm, lti, CNN, (int) TxTrIndent[tid], (int) TxTrIndent[tid], "");
}                                               // end 'TxTraceLeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine if tracing is enabled for specified (function) name
/*****************************************************************************/
BOOL TxTrFnEnabled
(
   char               *fn                       // IN    Function name
)
{
   BOOL                rc = TRUE;               // function return

   if      (((TxTrLevel % 2) == 0) &&          // only if ODD level
            ((strncasecmp( fn, "txw",       3) == 0) ||
             (strncasecmp( fn, "TxSetMenu", 9) == 0) ||
             (strncasecmp( fn, "TxSetList", 9) == 0) ||
             (strncasecmp( fn, "txaParseN", 9) == 0) ||
             (strncasecmp( fn, "txaReadAn", 9) == 0) ||
             (strncasecmp( fn, "txaOption", 9) == 0) ||
             (strncasecmp( fn, "txTextSea", 9) == 0) ||
             (strncasecmp( fn, "txFileTre", 9) == 0) ||
             (strncasecmp( fn, "txFileDir", 9) == 0) ||
             (strncasecmp( fn, "txFindNex", 9) == 0) ||
             (strncasecmp( fn, "txRecursi", 9) == 0) ||
             (strncasecmp( fn, "txTrueNam", 9) == 0) ||
             (strncasecmp( fn, "TxSelSort", 9) == 0) ||
             (strncasecmp( fn, "TxSelComp", 9) == 0) ||
             (strncasecmp( fn, "TxSelSetI", 9) == 0) ||
             (strncasecmp( fn, "TxSelGetM", 9) == 0) ||
             (strncasecmp( fn, "TxSelItem", 9) == 0) ||
             (strncasecmp( fn, "TxFreeMem", 9) == 0) ))
   {
      rc = FALSE;
   }

   //- to be refined, could implement a (small) configurable function-name database
   //- for excluding (or including, or perhaps both :)

   return (rc);
}                                               // end 'TxTrFnEnabled'
/*---------------------------------------------------------------------------*/

