/*******************************************************************
 *
 *  gwin_x11.c  graphics utility X-Window driver.              1.0
 *
 *  This is the driver for windowed display under X11, used by the
 *  graphics utility of the FreeType test suite.
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************/

#include "gdriver.h"
#include "gmain.h"
#include "gevents.h"

#include "tttypes.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define USE_XIMAGE

  /* Translator added to ease changes to control keys */

  typedef struct  _Translator
  {
    char    key;
    GEvent  event_class;
    int     event_info;
  } Translator;

#define NUM_Translators  15

  static const Translator  trans[NUM_Translators] =
  {
    { (char)27, event_Quit,            0 },

    { 'x',      event_Rotate_Glyph,   -1 },
    { 'c',      event_Rotate_Glyph,    1 },
    { 'v',      event_Rotate_Glyph,  -16 },
    { 'b',      event_Rotate_Glyph,   16 },

    { '9',      event_Change_Glyph, -100 },
    { '0',      event_Change_Glyph,  100 },
    { 'i',      event_Change_Glyph,  -10 },
    { 'o',      event_Change_Glyph,   10 },
    { 'k',      event_Change_Glyph,   -1 },
    { 'l',      event_Change_Glyph,    1 },

    { '+',      event_Scale_Glyph,    10 },
    { '-',      event_Scale_Glyph,   -10 },
    { 'u',      event_Scale_Glyph,     1 },
    { 'j',      event_Scale_Glyph,    -1 }
  };

  /* End of translators addition. See Get_Event() below */

  extern char      Header[];    /* defined in the test programs */

  static Window    win;
  static GC        gcblack;
  static GC        gcwhite;

#ifndef USE_XIMAGE
  static GC        gccol[5];
#endif

  static XColor    color[5];

  Display*         display;
  static char*     displayname = "";

#ifdef USE_XIMAGE
  static XImage*   image;
#endif

  static Visual*   visual;
  static Colormap  colormap;
  static int       depth;
  static Bool      gray;

  /* window size */

  static int       x, y, w, h;

  long             vioBufOfs;


  /* restore acreen to its original state */

  int  Driver_Restore_Mode()
  {
    XUnmapWindow( display, win );
    XCloseDisplay( display );

    return 1;       /* success */
  }


  /* set graphics mode */

  void  x11init()
  {
    int                   screen_num, i;
    XTextProperty         xtp;
    XSizeHints            xsh;
    XSetWindowAttributes  xswa;

    unsigned short colors[5] = { 0, 16, 32, 48, 63 }; 


    XrmInitialize();

    if( !( display = XOpenDisplay( displayname ) ) )
    {
      printf( "Cannot open Display\n" );
      exit( 1 );
    }

    screen_num = DefaultScreen  ( display );
    colormap   = DefaultColormap( display, screen_num );
    depth      = DefaultDepth   ( display, screen_num );
    visual     = DefaultVisual  ( display, screen_num );

#ifdef USE_XIMAGE
    if ( gray )
    {
      int                   count;

      XPixmapFormatValues*  formats;


      formats           = XListPixmapFormats( display, &count );
      vio_ScanLineWidth = 0;

      while ( count > 0 )
      {
        --count;
        if ( formats[count].depth == depth )
        {
          int  bits;

          bits = w * formats[count].bits_per_pixel;
          if ( bits % formats[count].scanline_pad )
          {
            bits -= bits % formats[count].scanline_pad;
            bits += formats[count].scanline_pad;
          }

          vio_ScanLineWidth = bits / 8;
          break;
        }
      }
      if ( !vio_ScanLineWidth )
      {
        printf( "The display doesn't offer a suitable pixmap format\n" );
        exit( 1 );
      }

      XFree( formats );
    }
#endif

    Vio = (PByte)malloc( h * vio_ScanLineWidth );

    if ( !Vio )
    {
      printf( "Cannot malloc Display mem\n" );
      exit( 1 );
    }

    xswa.border_pixel     = BlackPixel( display, screen_num );
    xswa.background_pixel = WhitePixel( display, screen_num );

#ifndef USE_XIMAGE
    xswa.event_mask = KeyPressMask;
#else
    xswa.event_mask = KeyPressMask | ExposureMask;
#endif
    
    win = XCreateWindow( display,
                         RootWindow( display, screen_num ),
                         x,
                         y,
                         w,
                         h,
                         10,
                         depth,
                         InputOutput, 
                         visual,
                         CWBackPixel | CWBorderPixel | CWEventMask,
                         &xswa );

    XMapWindow( display, win );
 
    gcblack = XCreateGC( display, RootWindow( display, screen_num ),
                         0L, NULL );

    XSetForeground( display, gcblack, BlackPixel( display, screen_num ) );
    XSetBackground( display, gcblack, WhitePixel( display, screen_num ) );

    gcwhite = XCreateGC( display, RootWindow( display, screen_num ),
                         0L, NULL );

    XSetForeground( display, gcwhite, WhitePixel( display, screen_num ) );
    XSetBackground( display, gcwhite, BlackPixel( display, screen_num ) );

    /* allocate colors */

    if ( gray )
      for ( i = 0; i < 5; i++ )
      {

#ifndef USE_XIMAGE
        XGCValues  values;
#endif
        gray_palette[i] = i;
  
        color[i].red   =
        color[i].green =
        color[i].blue  = 65535 - ( colors[i] * 65535 ) / 63;
  
        if ( !XAllocColor( display, colormap, &color[i] ) )
        {
          printf( "Cannot allocate Color\n" );
          exit( 1 );
        }
  
#ifndef USE_XIMAGE
        values.foreground = color[i].pixel;
  
        gccol[i] = XCreateGC( display, 
                              RootWindow( display,screen_num ),
                              GCForeground, 
                              &values );
#endif
      }

#ifdef USE_XIMAGE
    image = XCreateImage( display, 
                          visual,
                          gray ? depth   : 1, 
                          gray ? ZPixmap : XYBitmap,
                          0, 
                          Vio, 
                          w, 
                          h, 
                          8, 
                          0 );
    if ( !image )
    {
      printf( "Cannot create image\n" );
      exit( 1 );
    }

    if ( !gray )
    {
      /* this IS portable, according to my Xlib manual -- peak */
      image->byte_order       = MSBFirst;
      image->bitmap_bit_order = MSBFirst;
    }
#endif /* USE_XIMAGE */

    /* Make Window Manager happy :-) */
    xtp.value    = "FreeType";
    xtp.encoding = 31;
    xtp.format   = 8;
    xtp.nitems   = strlen( xtp.value );

    xsh.x = x;
    xsh.y = y;

    xsh.width  = w;
    xsh.height = h;
    xsh.flags  = (PPosition | PSize);
    xsh.flags  = 0;

    XSetWMProperties( display, win, &xtp, &xtp, NULL, 0, &xsh, NULL, NULL );
  }


  int  Driver_Set_Graphics( int  mode )
  {
    if ( mode == Graphics_Mode_Gray )
    {
      gray = 1;
      vio_ScanLineWidth = 320;

      x = 0;
      y = 0;
      w = 320;
      h = 200;
    }
    else if ( mode == Graphics_Mode_Mono )
    {
      gray = 0;
      vio_ScanLineWidth = 80;

      x = 0;
      y = 0;
      w = 640;
      h = 450;
    }
    else
    {
      printf( "mode %d not supported\n", mode );
      exit( 1 );
    }

    vio_Width  = w;
    vio_Height = h;

    x11init();

    return 1;       /* success */
  }


#ifdef USE_XIMAGE
  void  Put_Image( int  x, int  y, int  w, int  h )
  {
    XPutImage( display, win, gcblack, image, x, y, x, y, w, h );
  }
#endif


  int  Driver_Display_Bitmap( char*  buffer, int  line, int  col )
  {
#ifndef USE_XIMAGE
    int    b;
#endif
    int    z, y, used_col;
    char*  target;

    XClearWindow( display, win );

    if ( gray )
      memset( Vio, color[0].pixel, h * vio_ScanLineWidth );
    else
      memset( Vio, 0, h * vio_ScanLineWidth );

    /* this displays the Header string in the window title */
    XStoreName( display, win, Header );

    if ( line > h )
      line = h;
    if ( col > vio_ScanLineWidth )
      used_col = vio_ScanLineWidth;
    else
      used_col = col;

#ifndef USE_XIMAGE
    target = Vio + ( line - 1 ) * vio_ScanLineWidth;

    for ( y = 0; y < line; y++ )
    {
      memcpy( target, buffer, used_col );
      target -= vio_ScanLineWidth;
      buffer += col;
    }

    for ( y = 0; y < line; y++ )
    {
      for ( z = 0; z < used_col; z++ )
      {
        int  c = Vio[y * vio_ScanLineWidth + z];


        if ( gray )
        { 
          if ( c < 0 || c >= 5 )    /* security check */
          {
            printf( "weird grayshade: %d\n", c );
            c = 0;
          }
          XDrawPoint( display, win, gccol[c], z, y );
        }
        else
          for ( b = 7; b >= 0 ; b-- )
          {
            if ( (1 << b) & c )
              XDrawPoint( display, win, gcblack, z * 8 + 7 - b, y );
          }
      }
    }

#else /* USE_XIMAGE */

    if ( !gray )
    {
      target = Vio + ( line - 1 ) * vio_ScanLineWidth;

      for ( y = 0; y < line; y++ )
      {
        memcpy( target, buffer, used_col );
        target -= vio_ScanLineWidth;
        buffer += col;
      }
    }
    else
    {
      for ( y = line - 1; y >= 0; y-- )
      {
        char*  bufp;


        bufp = buffer;

        for ( z = 0; z < used_col; z++ )
        {
          int  c;


          c = *bufp++;

          if ( c < 0 || c >= 5 ) /* security check */
          {
            printf( "weird grayshade: %d\n", c );
            c = 0;
          }
          XPutPixel( image, z, y, color[c].pixel );
        }

        buffer += col;
      }
    }

    Put_Image( 0, 0, vio_Width, vio_Height );

#endif

    return 1;
  }


  /* This function maps X keystrokes into GEvents. Note that */
  /* currently only keystrokes events exit this function.    */

  void  Get_Event( TEvent*  event )
  {
    static char     key_buffer[10];
    static int      key_cursor = 0;
    static int      key_number = 0;
    static XEvent   x_event;
           KeySym   key;

    int             i, bool_exit;
    char            c;

    XComposeStatus  compose;


    bool_exit = key_cursor < key_number;

    while ( !bool_exit )
    {
      XNextEvent( display, &x_event );

      switch ( x_event.type )
      {
      case KeyPress:
        key_number = XLookupString( &x_event.xkey,
                                    key_buffer,
                                    sizeof ( key_buffer ),
                                    &key,
                                    &compose );
        key_cursor = 0;

        if ( key_number > 0 )
          bool_exit = 1;
        break;

      case MappingNotify:
        XRefreshKeyboardMapping( &x_event.xmapping );
        break;
      
#ifdef USE_XIMAGE
        /* XPutImage is fast enough to handle exposures */
      case Expose:
        Put_Image( x_event.xexpose.x, 
                   x_event.xexpose.y,
                   x_event.xexpose.width, 
                   x_event.xexpose.height );
        break;

      /* You should add more cases to handle mouse events, etc. */
#endif

      }
    }

    c = key_buffer[key_cursor++];

    for ( i = 0; i < NUM_Translators; i++ )
    {
      if ( c == trans[i].key )
      {
        event->what = trans[i].event_class;
        event->info = trans[i].event_info;
        return;
      }
    }

    event->what = event_Keyboard;
    event->info = (int)c;
  }


/* End */
