/*******************************************************************
 *
 *  ttfile.c                                                     1.2
 *
 *    File I/O Component (body).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTES :
 *
 *   This implementation relies on the ANSI libc. You may wish to
 *   modify it to get rid of the libc and go straight to the your
 *   platform's stream routines.
 *
 *   The same source code can be used for thread-safe and re-entrant
 *   builds of the library.
 *
 ******************************************************************/

static const char  _to_satisfy_ANSI_C_[] = "";


#include "ttconfig.h"

#ifndef HAVE_MMAP

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "freetype.h"
#include "tttypes.h"
#include "tterror.h"
#include "ttfile.h"     /* our prototypes */
#include "ttmutex.h"

#define MAX_FONT_PATHNAME 128

  typedef struct _TStream_Rec
  {
    FILE*   file;    /* file handle */
    long    base;    /* stream base in file */
    long    size;    /* stream size in file */
    int*    error;   /* error variable */

#ifdef TT_CONFIG_REENTRANT
    char    path[ MAX_FONT_PATHNAME ];  /* current pathname */
#endif

  } TStream_Rec;
  /* we support embedded TrueType files by allowing them to be   */
  /* inside any file, at any location, hence the 'base' argument */

  /* note however that the current implementation does not allow you */
  /* to specify a 'base' index when opening a file.                  */
  /* (will come later)                                               */

  /* I still don't know if this will turn out useful ??   - DavidT */

  /* note also that the only reason to keep the stream's pathname is   */
  /* the fact that the ANSI libc seems to lack a function to duplicate */
  /* FILE structures. We need several streams to access the same font  */
  /* file in a re-entrant build, and we're obliged to re-open the same */
  /* font file to get a new valid FILE* !! Ugly but portable..         */

  typedef TStream_Rec*  PStream_Rec;

#define STREAM2REC(x)  ((TStream_Rec*)(x))
#define REC2STREAM(x)  ((TT_Stream)(x))

#define MAX_TT_FILES 8  /* maximum number of opened files */

 static TStream_Rec  stream_table[ MAX_TT_FILES ]; /* table of stream recs */
 static TMutex       table_mutex;                  /* table's mutex        */


#ifndef TT_CONFIG_REENTRANT     /* thread-safe implementation only */

 static TFileFrame   cur_frame;         /* current frame      */
 static PStream_Rec  cur_stream = NULL; /* current stream rec */
 static TMutex       file_mutex;        /* the file mutex     */

#endif /* !REENTRANT */


#ifdef TT_CONFIG_REENTRANT

#define CUR_Stream  STREAM2REC(stream)      /* re-entrant macros */
#define CUR_Frame  (*frame)
#define STREAM_ARGS stream,
#define STREAM_ARG  stream
#define ERROR      (*error)

#else

#define CUR_Stream  cur_stream              /* thread-safe macros */
#define CUR_Frame   cur_frame
#define STREAM_ARGS /* void */
#define STREAM_ARG  /* void */
#define ERROR       Error       /* the global Error variable */

#endif /* REENTRANT */

  /* The macro CUR_Stream denotes the current input file */
  /* Note that for the re-entrant version, the 'stream' name has been */
  /* chosen according to the macro FILE_OPS                           */

  /* The macro CUR_Frame denotes the current file frame */
  /* Note that for the re-entrant version, the 'frame' name has been */
  /* chosen according to the macro FRAME_OPS                         */

  /* The macro STREAM_ARG is used when calling public functions */
  /* without others                                             */

/*******************************************************************
 *
 *  Function    :  TTFile_Init
 *
 *  Description :  Init the File component
 *
 ******************************************************************/

 void TTFile_Init()
 {
   int i;

   Mutex_Create( &table_mutex );

   /* Clear the stream table */

   for ( i = 0; i < MAX_TT_FILES; i++ )
   {
     stream_table[i].file = NULL;
     stream_table[i].base = 0;
     stream_table[i].size = 0;

#ifdef TT_CONFIG_REENTRANT
     stream_table[i].path[0] = 0;
#endif
   }

#ifndef TT_CONFIG_REENTRANT
   Mutex_Create( &file_mutex );
#endif
 }

/*******************************************************************
 *
 *  Function    :  TTFile_Done
 *
 *  Description :  Finalize the File component
 *
 ******************************************************************/

 void TTFile_Done()
 {

#ifndef TT_CONFIG_REENTRANT
   Mutex_Delete( &file_mutex);
#endif

   Mutex_Delete( &table_mutex );
 }

/*******************************************************************
 *
 *  Function    :  Allocate     
 *
 *  Description :  Allocate a new stream record from the table       
 *
 *  Input  :  file     input file. Must be opened by caller.
 *
 *  Output :  pointer to new stream rec. NULL in case of failure
 *
 *  Note : called by TT_Open_File
 *
 ******************************************************************/

  static PStream_Rec  Allocate( FILE* file, long base, int* error )
  {
    int i;

    PStream_Rec  result;

    Mutex_Lock( &table_mutex );

    result = NULL;

    i = 0;
    while ( i < MAX_TT_FILES )
    {
      if ( stream_table[i].file == NULL )
      {
        stream_table[i].file = file;

        fseek( file, 0, SEEK_END );
        stream_table[i].size  = ftell( file ) - base;
        stream_table[i].base  = base;
        stream_table[i].error = error;

        DebugTrace2( "(Open_Font_File) file size is = %ld\n", 
                     stream_table[i].size );

        fseek( file, base, SEEK_SET );

        result = (PStream_Rec)&stream_table[i];
        i      = MAX_TT_FILES;
      }

      i++;
    }

    Mutex_Release( &table_mutex );

    return result;
  }

/*******************************************************************
 *
 *  Function    :  Release
 *
 *  Description :  Release a used stream record from the table       
 *
 *  Input  :  streamRec
 *
 *  Output :  None.
 *
 *  Note : Called by TT_Close_File
 *
 ******************************************************************/

  static void  Release( PStream_Rec streamRec )
  {
    Mutex_Lock( &table_mutex );

    /* XXX : No check yet */

    streamRec->file  = NULL;
    streamRec->base  = 0;
    streamRec->size  = 0;
    streamRec->error = NULL;

    Mutex_Release( &table_mutex );
  }

/*******************************************************************
 *
 *  Function    :  TT_Open_File
 *
 *  Description :  opens the font file and saves the total file size.
 *
 *  Input  :  error          address of stream's error variable
 *                           ( re-entrant build only ).
 *
 *            filepathname   pathname of the file to open
 *            stream         address of target TT_Stream structure
 *
 *
 *  Output :  SUCCESS on sucess, FAILURE on error.
 *            The target stream is set to -1 in case of failure.
 *
 *
 ******************************************************************/

  Bool  TT_Open_File( ERROR_OPS char* filepathname, TT_Stream*  stream )
  {
    FILE*        font_file;
    PStream_Rec  stream_rec;

    *stream = NULL;

    if ( !( font_file = fopen( filepathname, "rb" ) ) )
    {
      ERROR = TT_Err_File_Error;
      /* We cannot return an error message in a global variable anymore */
      return FAILURE;
    }

    stream_rec = Allocate( font_file, 0, &ERROR );

#ifdef TT_CONFIG_REENTRANT
    if ( stream_rec )
    {
      strncpy( stream_rec->path, filepathname, MAX_FONT_PATHNAME-1 );
      stream_rec->path[ MAX_FONT_PATHNAME ] = '\0';
    }
#endif

    *stream = REC2STREAM( stream_rec );

    return (*stream != NULL);
  }



#ifdef TT_CONFIG_REENTRANT

/*******************************************************************
 *
 *  Function    :  TT_Duplicate_File
 *
 *  Description :  Duplicate a stream for a new instance record
 *
 *  Input  :  input_stream   source stream to duplicate
 *            copy           address of target duplicate stream
 *            error          error of duplicate's error variable
 *
 *  Output :  SUCCESS on sucess, FAILURE on error.
 *            The target stream is set to -1 in case of failure.
 *
 *  Note : This function is only available to re-entrant builds of
 *         the engine. Note that we kept a font pathname in the
 *         stream structure because we like an ANSI function to
 *         duplicate FILE structures conveniently.
 *
 ******************************************************************/

  Bool  TT_Duplicate_File( TT_Stream  input_stream,
                           TT_Stream* copy,
                           int*       error )
  {
    FILE*        font_file;
    PStream_Rec  stream_rec;

    stream_rec = STREAM2REC(input_stream);
    *copy      = NULL;

    if ( !( font_file = fopen( stream_rec->path, "rb" ) ) )
    {
      *error = TT_Err_File_Error;
      return FAILURE;
    }

    *copy = REC2STREAM( Allocate( font_file, stream_rec->base, error ) );

    strncpy( STREAM2REC(*copy)->path, stream_rec->path, MAX_FONT_PATHNAME );

    return ( *copy != NULL);
  }

#endif /* REENTRANT */

/*******************************************************************
 *
 *  Function    : TT_Close_File
 *
 *  Description : Closes the font file and releases memory buffer.
 *
 *  Input  :  
 *
 *  Output :  SUCCESS (always)
 *
 ******************************************************************/

  Bool  TT_Close_File( TT_Stream  stream )
  {
    /* XXX : No check yet */
    Release( STREAM2REC(stream) );
    return SUCCESS;
  }


#ifndef TT_CONFIG_REENTRANT    /* single-threaded implementation only */

/*******************************************************************
 *
 *  Function    : TT_Lock_Access
 *
 *  Description : Acquires the file mutex (blocking call).
 *
 *  Input  :  None
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  Bool  TT_Lock_Access( TT_Stream stream )
  {
    Mutex_Lock( &file_mutex );

    /* XXX : No check yet */
    cur_stream = STREAM2REC(stream);

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    : TT_Release_Access
 *
 *  Description : Releases the file mutex.
 *
 *  Input  :  None
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  Bool  TT_Release_Access( TT_Stream  aStream )
  {
    /* XXX : No check yet */
    cur_stream = NULL;

    Mutex_Release( &file_mutex );

    return SUCCESS;
  }

#endif /* !REENTRANT */

/*******************************************************************
 *
 *  Function    : TT_Seek_File
 *
 *  Description : Seeks the file cursor to a different position.
 *
 *  Input  :  position     new position in file
 *
 *  Output :  SUCCESS on success. FAILURE if out of range.
 *
 ******************************************************************/

  Bool  TT_Seek_File( FILE_OPS long position )
  {
    position += CUR_Stream->base;

    if ( fseek( CUR_Stream->file, position, SEEK_SET ) )
    {                               /* zero return from fseek is SUCCESS */
      *CUR_Stream->error = TT_Err_File_Error;
      return FAILURE;
    }

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    : TT_Skip_File
 *
 *  Description : Skips forward the file cursor.
 *
 *  Input  :  distance    number of bytes to skip
 *
 *  Output :  see TT_Seek_File
 *
 ******************************************************************/

  Bool  TT_Skip_File( FILE_OPS long distance )
  {
    return TT_Seek_File( STREAM_ARGS ftell( CUR_Stream->file ) -
                                     CUR_Stream->base + distance );
  }

/*******************************************************************
 *
 *  Function    : TT_Read_File
 *
 *  Description : Reads a chunk of the file and copies it to memory.
 *
 *  Input  :  buffer    target buffer
 *            count     length in bytes to read
 *
 *  Output :  SUCCESS on success. FAILURE if out of range.
 *
 ******************************************************************/

  Bool  TT_Read_File( FILE_OPS void* buffer, long count )
  {
    if ( fread( buffer, 1, count, CUR_Stream->file ) != (unsigned long)count )
    {
      *CUR_Stream->error = TT_Err_File_Error;
      return FAILURE;
    }

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    : TT_Read_At_File
 *
 *  Description : Reads file at a specified position.
 *
 *  Input  :  position  position to seek to before read
 *            buffer    target buffer
 *            count     number of bytes to read
 *
 *  Output :  SUCCESS on success. FAILURE if error.
 *
 ******************************************************************/

  Bool  TT_Read_At_File( FILE_OPS long position, void* buffer, long count )
  {
    if ( !TT_Seek_File( STREAM_ARGS position ) || 
         !TT_Read_File( STREAM_ARGS buffer, count ) )
      return FAILURE;

    return SUCCESS;
  }

/*******************************************************************
 *
 *  Function    :  TT_File_Pos
 *
 *  Description :  Returns current file seek pointer.
 *
 *  Input  :  none
 *
 *  Output :  current file position
 *
 ******************************************************************/

  Long  TT_File_Pos( FILE_OP )
  {
    return (ftell( CUR_Stream->file ) - CUR_Stream->base);
  }

/*******************************************************************
 *
 *  Function    :  TT_Access_Frame
 *
 *  Description :  Notifies the component that we're going to read
 *                 'size' bytes from the current file position.
 *                 This function should load/cache/map these bytes
 *                 so that they will be addressed by the GET_xxx
 *                 functions easily.
 *
 *  Input  :  size   number of bytes to access.
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes:    The function fails if the byte range is not within the
 *            the file, or if there is not enough memory to cache
 *            the bytes properly (which usually means that aSize is
 *            too big in both cases).
 *
 ******************************************************************/

  Bool  TT_Access_Frame( FILE_OPS FRAME_OPS int size )
  {
    if ( CUR_Frame.address != NULL )
    {
      *CUR_Stream->error = TT_Err_Nested_Frame_Access;
      return FAILURE;
    }

    CUR_Frame.size   = size;

    CUR_Frame.address = (unsigned char*)malloc( size );

    if ( CUR_Frame.address == NULL ) 
      return FAILURE;

    if ( !TT_Read_File( STREAM_ARGS (void*)CUR_Frame.address, size ) )
    {
      CUR_Frame.size = 0;
      free( CUR_Frame.address );
      return FAILURE;
    }

    CUR_Frame.cursor = CUR_Frame.address;

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  TT_Check_And_Access_Frame
 *
 *  Description :  Notifies the component that we're going to read
 *                 aSize bytes from the current file position.
 *                 This function should load/cache/map these bytes
 *                 so that they will be addressed by the GET_xxx
 *                 functions easily.
 *
 *  Input  :  size   number of bytes to access.
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes:    The function truncates aSize if the byte range is not 
 *            within the file.
 *
 *            It will fail if there is not enough memory to cache
 *            the bytes properly (which usually means that aSize is
 *            too big).
 *
 *            It will fail if you make two consecutive calls
 *            to TT_Access_Frame(), without a TT_Forget_Frame() between
 *            them.
 *
 *            The only difference with TT_Access_Frame() is that we
 *            check that the frame is within the current file.  We
 *            otherwise truncate it.
 *
 ******************************************************************/

  Bool  TT_Check_And_Access_Frame( FILE_OPS FRAME_OPS int size )
  {
    long  readBytes;

    if ( CUR_Frame.address != NULL ) 
    {
      *CUR_Stream->error = TT_Err_Nested_Frame_Access;
      return FAILURE;
    }

    CUR_Frame.size   = size;

    CUR_Frame.address = (unsigned char*)malloc( size );

    if ( CUR_Frame.address == NULL ) 
      return FAILURE;

    readBytes = CUR_Stream->size - TT_File_Pos( STREAM_ARG );
    if ( size > readBytes ) size = readBytes;

    if ( !TT_Read_File( STREAM_ARGS (void*)CUR_Frame.address, size ) )
    {
      CUR_Frame.size = 0;
      free( CUR_Frame.address );
      return FAILURE;
    }

    CUR_Frame.cursor = CUR_Frame.address;

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  TT_Forget_Frame
 *
 *  Description :  Releases a cached frame after reading.
 *
 *  Input  :  None
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  Bool  TT_Forget_Frame( FRAME_OP )
  {
    if ( CUR_Frame.address == NULL )
      return FAILURE;

    free( CUR_Frame.address );

    CUR_Frame.address = NULL;
    CUR_Frame.size    = 0;
    CUR_Frame.cursor  = NULL;

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  GET_Byte
 *
 *  Description :  Extracts a byte from the current file frame.
 *
 *  Input  :  None or current frame
 *
 *  Output :  Extracted Byte
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call.)
 *
 ******************************************************************/

  unsigned char Get_Byte( FRAME_OP )
  {
    return (unsigned char)(*CUR_Frame.cursor++);
  }


/*******************************************************************
 *
 *  Function    :  GET_Char
 *
 *  Description :  Extracts a signed byte from the current file frame.
 *
 *  Input  :  None or current frame
 *
 *  Output :  Extracted char
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call.)
 *
 ******************************************************************/

  char Get_Char( FRAME_OP )
  {
    return (char)(*CUR_Frame.cursor++);
  }

/*******************************************************************
 *
 *  Function    :  GET_Short
 *
 *  Description :  Extracts a short from the current file frame.
 *
 *  Input  :  None or current frame
 *
 *  Output :  Extracted short
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call.)
 *
 ******************************************************************/

  short  Get_Short( FRAME_OP )
  {
    short  getshort;

    getshort = ((short)CUR_Frame.cursor[0] << 8) |
                (short)CUR_Frame.cursor[1];

    CUR_Frame.cursor += 2;

    return getshort;
  }


/*******************************************************************
 *
 *  Function    :  GET_UShort
 *
 *  Description :  Extracts an unsigned short from the frame.
 *
 *  Input  :  None or current frame
 *
 *  Output :  Extracted ushort
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call.)
 *
 ******************************************************************/

  unsigned short  Get_UShort( FRAME_OP )
  {
    unsigned short  getshort;

    getshort = ((unsigned short)CUR_Frame.cursor[0] << 8) |
                (unsigned short)CUR_Frame.cursor[1];

    CUR_Frame.cursor += 2;

    return getshort;
  }


/*******************************************************************
 *
 *  Function    :  GET_Long
 *
 *  Description :  Extracts a long from the frame.
 *
 *  Input  :  None or current frame
 *
 *  Output :  Extracted long
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call.)
 *
 ******************************************************************/

  long Get_Long( FRAME_OP )
  {
    long  getlong;

    getlong = ((long)CUR_Frame.cursor[0] << 24) |
              ((long)CUR_Frame.cursor[1] << 16) |
              ((long)CUR_Frame.cursor[2] << 8 ) |
               (long)CUR_Frame.cursor[3];

    CUR_Frame.cursor += 4;

    return getlong;
  }


/*******************************************************************
 *
 *  Function    :  GET_ULong
 *
 *  Description :  Extracts an unsigned long from the frame.
 *
 *  Input  :  None
 *
 *  Output :  Extracted ulong
 *
 *  NOTES : We consider that the programmer is intelligent enough
 *          not to try to get a byte that is out of the frame. Hence,
 *          we provide no bounds check here. (A misbehaving client
 *          could easily page fault using this call.)
 *
 ******************************************************************/

  unsigned long Get_ULong( FRAME_OP )
  {
    unsigned long  getlong;

    getlong = ( ((unsigned long)CUR_Frame.cursor[0] << 24) |
                ((unsigned long)CUR_Frame.cursor[1] << 16) |
                ((unsigned long)CUR_Frame.cursor[2] << 8 ) |
                 (unsigned long)CUR_Frame.cursor[3] );

    CUR_Frame.cursor += 4;

    return getlong;
  }

#endif /* HAVE_MMAP */


/* End */
