/****************************************************************************/
/*                                                                          */
/*  The FreeType project - a free and portable quality TrueType renderer.   */
/*                                                                          */
/*  Copyright 1996, 1997 by                                                 */
/*  D. Turner, R.Wilhelm, and W. Lemberg                                    */
/*                                                                          */
/*  zoom : A simple glyph viewer.  Now supports graylevel rendering         */
/*         with the '-g' option.                                            */
/*                                                                          */
/*         use -p <platformID> together with -e <encodingID> to select      */
/*         a cmap;                                                          */
/*                                                                          */
/*                                                                          */
/*  Keys :                                                                  */
/*                                                                          */
/*  x :   fine counter_clockwise rotation                                   */
/*  c :   fine clockwise rotation                                           */
/*                                                                          */
/*  v :   fast counter_clockwise rotation                                   */
/*  b :   fast clockwise rotation                                           */
/*                                                                          */
/*  + :   fast scale up                                                     */
/*  - :   fast scale down                                                   */
/*  u :   fine scale down                                                   */
/*  j :   fine scale up                                                     */
/*                                                                          */
/*  l :   go to next glyph                                                  */
/*  k :   go to previous glyph                                              */
/*                                                                          */
/*  o :   go to tenth next glyph                                            */
/*  i :   go to tenth previous glyph                                        */
/*                                                                          */
/*  0 :   go to hundredth next glyph                                        */
/*  9 :   go to hundredth previous glyph                                    */
/*                                                                          */
/*  ESC :   exit                                                            */
/*                                                                          */
/*                                                                          */
/*  NOTE : This is just a test program that is used to show off and         */
/*         debug the current engine; which is still in alpha. In no         */
/*         way does it shows the final high-level interface that            */
/*         client applications will use. Wait for at least a beta for       */
/*         this.                                                            */
/*                                                                          */
/****************************************************************************/

#ifdef ARM
#include "std.h"
#include "graflink.h"
#endif

#include "tttypes.h"
#include "tterror.h"
#include "ttcalc.h"
#include "tttables.h"
#include "ttmemory.h"
#include "ttraster.h"
#include "ttindex.h"
#include "ttins.h"
#include "ttfile.h"
#include "ttexec.h"
#include "tttables.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>       /* libc ANSI */

/* getopt() should be in either getopt.h, stdlib.h, or unistd.h */

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif

#include "gmain.h"
#include "gevents.h"
#include "gdriver.h"


#ifdef ARM
#include "armsup.c" /* pull in our routines */
#endif


#define  Pi         3.1415926535

#define  MAXPTSIZE  500                /* dtp */
#define  Center_X   ( Bit.width / 2 )  /* dtp */
#define  Center_Y   ( Bit.rows  / 2 )  /* dtp */

#define  Profile_Buff_Size  64000      /* Size of the render pool   */
                                       /* Minimum is around 4 Kb    */
                                       /* experiment to see how its */
                                       /* size impacts on raster    */
                                       /* performance...            */

#define  Font_Buff_Size     256000     /* this buffer holds all     */
                                       /* font specific data.       */

  char              Header[128];

  TT_Stream         stream;
  PResident_Record  resident;
  PInstance_Record  instance;

  PStorage          Font_Buffer;

  int  num_glyphs;
  int  num_pts;
  int  num_ctr;

  int  glyfArray;

  PShort  epts_ctr;

  /* DTP important font metrics */
  Long   cur_LeftSB,
         cur_Ascent,
         cur_Descent,
         cur_AdvanceWidth; 
           
  Int    ptsize;

  float  ymin,
         ymax,
         xmax,
         xmin,
         xsize;    /* DTP testing precision */

  float  resR;

  TGlyphRecord   gRec;

  TRasterBlock   Bit;

  Int            Rotation;
  Int            Fail;
  Int            Num;
  Int            Code;
  unsigned char  autorun;

  char           GrayLines[1024];
  int            gray_render;

  TT_PCMapTable  cmap;
  int            use_cmap = 0;


  static void  ClearData( void );


  void  Set_Raster_Area( PInstance_Record  instance,
                         Int               ptsize )
  {
    Int  ppem;


    ppem = ptsize * 96 / 72;

    Bit.rows  = 1.5 * ppem;
    Bit.width = 1.5 * ppem;

    if ( gray_render )
    {
      Bit.cols  = Bit.width;
      Bit.flow  = TT_Flow_Up;
      Bit.size  = Bit.rows * Bit.cols;
    }
    else
    {
      Bit.cols  = ( Bit.width + 7 ) / 8;    /* convert to # of bytes */
      Bit.flow  = TT_Flow_Down;
      Bit.size  = Bit.rows * Bit.cols;      /* number of bytes in buffer */
    }
  }


  int  Init_Engine( PInstance_Record  instance,
                    Int               maxres ) 
  {
    PByte  p;

  
    Set_Raster_Area( instance, maxres );   /* setup maximum raster size */
  
    Bit.bitmap = (void*)malloc( (int)Bit.size );
    if ( !Bit.bitmap ) 
    {
      printf( "ERROR: Not enough memory to allocate bitmap!\n" );
      exit( 1 );
    }
  
    p = (PByte)malloc( Profile_Buff_Size );
    if ( !p )
    {
      printf( "ERROR: Not enough memory to allocate render pool!\n" );
      exit( 1 );
    }

    if ( gray_render )
      InitRasterizer( (long*)p, Profile_Buff_Size, GrayLines, 512 );
    else
      InitRasterizer( (long*)p, Profile_Buff_Size, NULL, 0 );

    ClearData();

    return 0;
  }
  
  
  static void  ClearData()
  {
    if ( gray_render )
      memset( Bit.bitmap, gray_palette[0], Bit.size );
    else
      memset( Bit.bitmap, 0, Bit.size );
    
    if ( gRec.outStarts ) free( gRec.outStarts );
    if ( gRec.xCoord )    free( gRec.xCoord );
    if ( gRec.yCoord )    free( gRec.yCoord );
    if ( gRec.flag )      free( gRec.flag );
  }
  

  Bool  LoadTrueTypeChar( Int  idx )
  {
    PGlyph  pG;
    Short   numP, numC;
    Int     j;


    /* Reset_Context( instance->exec );  */

    if ( idx < 0 || idx > num_glyphs )
      return FAILURE;

    if ( !Load_TrueType_Glyph( instance, idx, TRUE ) )
      return FAILURE;
  
    pG   = instance->glyph;
    numP = instance->pts.n;
    numC = instance->glyph->numberOfContours;
  
    if ( numP <= 0 || numC <= 0 )
      return FAILURE;
  
    gRec.outlines  = numC;
    gRec.points    = numP;

    /* Get storage for rasterizer */

    gRec.outStarts = (PUShort) malloc( sizeof ( UShort )     * numC );
    gRec.xCoord    = (PStorage)malloc( sizeof ( TT_F26Dot6 ) * numP );
    gRec.yCoord    = (PStorage)malloc( sizeof ( TT_F26Dot6 ) * numP );
    gRec.flag      = (PByte)   malloc( numP );

    if ( gRec.outStarts == NULL || gRec.xCoord == NULL ||
         gRec.yCoord == NULL || gRec.flag == NULL )
      return FAILURE;

    TT_Transform_Glyph( instance );     /* This call scales the glyph */
  
    numP -= 2;  /* remove phantom points */

    for ( j = 0; j < numP; ++j )
    {
      gRec.xCoord[j] = instance->pts.cur_x[j];
      gRec.yCoord[j] = instance->pts.cur_y[j];
      gRec.flag[j]   = instance->pts.touch[j] & TT_Flag_On_Curve;
    }
  
    for ( j = 0; j < numC; ++j )
      gRec.outStarts[j] = instance->glyph->contours[j].finish;
  
    return SUCCESS;
  }


  Bool  ConvertRaster()
  {
    if ( gray_render )
      return Render_Gray_Glyph( &gRec, &Bit, 2, gray_palette );
    else
      return Render_Glyph( &gRec, &Bit, 2 );
  }


  int  Process_Event( TEvent*  event )
  {
    switch ( event->what )
    {
    case event_Quit:            /* ESC */
      return 0;
      
    case event_Rotate_Glyph:
      Rotation = ( Rotation + event->info ) & 1023;
      break;

    case event_Scale_Glyph:
      ptsize += event->info;
      if ( ptsize < 1 )         ptsize = 1;
      if ( ptsize > MAXPTSIZE ) ptsize = MAXPTSIZE;
      break;

    case event_Change_Glyph:
      if ( use_cmap )
      {
        if ( event->info < 0 )
        {
          if ( Code > -event->info )
            Code += event->info;
          else
            Code = 0;

          for ( ; Code >= 0; Code-- )
          {
            Num = TT_code_to_index( Code, cmap, instance );
            if ( Num > 0 )
              break;
          }
        }
        else
        {
          if ( Code < 65536 - event->info - 1 )
            Code += event->info;
          else
            Code = 65536 - 1;

          for ( ; Code < 65536; Code++ )
          {
            Num = TT_code_to_index( Code, cmap, instance );
            if ( Num > 0 )
              break;
          }
        }
      }
      else
      {
        if ( event->info < 0 )
          if ( Num > -event->info )
            Num += event->info;
          else
            Num = 0;
        else
          if ( Num < num_glyphs - event->info - 1 )
            Num += event->info;
          else
            Num = num_glyphs - 1;
      }
      break;
    }

    return 1;
  }


  void usage( char*  execname )
  {
    printf( "\n" );
    printf( "Zoom: simple TrueType glyph viewer - part of the FreeType project\n" );
    printf( "-----------------------------------------------------------------\n" );
    printf( "\n" );
    printf( "Usage: %s [-g] [-p platform] [-e encoding] fontname[.ttf]\n",
             execname );
    printf( "\n" );
    printf( "  where '-g' asks for gray-level rendering.\n" );
    printf( "  If either -p or -e is not set, no cmap will be used.\n" );
    printf( "\n" );

    exit( 1 );
  }


  /* stack check dtp */

  int  main( int  argc, char**  argv ) 
  {
    int         i, xpos, ypos;
    int         platform = -1, encoding = -1;
    TT_F26Dot6  x_off, y_off;
    char        filename[128 + 4];
    char*       execname;
    int         option;
  
    TEvent  event;
 

    Font_Buffer = (PStorage)malloc( Font_Buff_Size );
    if ( !Font_Buffer )
    {
      printf( "Error: Could not even allocate font pool!\n" );
      exit( 1 );
    }

    Init_FontPool( Font_Buffer, Font_Buff_Size );

    num_pts = 0;
    num_ctr = 0;

    execname    = argv[0];
    gray_render = 0;

    while ( 1 )
    {
      option = getopt( argc, argv, "gp:e:" );

      if ( option == -1 )
        break;

      switch ( option )
      {
      case 'g':
        gray_render = 1;
        break;

      case 'p':
        platform = atoi( optarg );
        break;

      case 'e':
        encoding = atoi( optarg );
        break;

      default:
        usage( execname );
        break;
      }
    }

    if ( optind == argc )
      usage( execname );

    i = strlen( argv[optind] );
    while ( i > 0 && argv[optind][i] != '\\' )
    {
      if ( argv[optind][i] == '.' )
        i = 0;
      i--;
    }

    filename[128] = 0;
    strncpy( filename, argv[optind], 128 );
    if ( i >= 0 )
      strncpy( filename + strlen( filename ), ".ttf", 4 );

    if ( platform >= 0 || encoding >= 0 )
      use_cmap = 1;

    if ( !TT_Open_File( filename , &stream ) )
    {
      printf( "Error: could not find/open %s\n", filename );
      exit( 1 );
    }

    if ( !TT_Load_Resident_Table( stream, &resident ) ) 
    {
      if ( Error == TT_Err_Out_Of_Memory )
      {
        printf( "Error: This font is too large to be loaded\n" );
        printf( "       increase the font pool then recompile\n" );
        exit( 1 );
      }

      printf( "ERROR: Could not load data from %s\n", filename );
      exit( 1 );
    }

    num_glyphs = resident->numGlyphs;
        
    if ( !TT_Load_Instance_Data( resident, &instance ) ) 
    {
      if ( Error == TT_Err_Out_Of_Memory )
      {
        printf( "Error: This font is too large to be loaded\n" );
        printf( "       increase the font pool then recompile\n" );
        exit( 1 );
      }

      printf( "ERROR: Could not open instance from %s\n", filename );
      exit( 1 );
    }

    if ( !Create_Context( instance ) )
      Panic1( "could not create execution context\n" );

    ptsize = 50;

    if ( !Reset_Context( instance, ptsize, 96L ) )
    {
      printf( "error = %d\n", instance->error );
      Panic1( "could not reset execution context\n" );
    }

    if ( use_cmap )
    {
      cmap = TT_select_CMap( platform, encoding, instance );
      if ( cmap == NULL )
      {
        printf( "Error: invalid platform and/or encoding ID\n" );
        exit( 1 );
      }
    }

    if ( Init_Engine( instance, MAXPTSIZE ) )
    {
      printf( "Init_Engine failed\n" );
      exit( 1 );
    }

    Set_Raster_Area( instance, ptsize ); 

    if ( gray_render ) 
    {
      if ( !SetGraphScreen( Graphics_Mode_Gray ) )
        Panic1( "could not set grayscale graphics mode\n" );
    }
    else
    {
      if ( !SetGraphScreen( Graphics_Mode_Mono ) )
        Panic1( "could not set mono graphics mode\n" );
    }

    if ( use_cmap )
    {
      /* we search for a first valid char code */
      for ( Code = 0; Code < 65636; Code++ )
      {
        Num = TT_code_to_index( Code, cmap, instance );
        if ( Num > 0 )
          break;
      }
    }
    else
    {
      Num  = 0;
      Code = 0;
    }
    
    autorun  = 0;
    Fail     = 0;
    Rotation = 0;
    xpos     = 0;
    ypos     = 0;
    x_off    = 0.2 * Bit.rows  * 64;
    y_off    = 0.3 * Bit.width * 64;

    for ( ;; )
    {
      ClearData();

      if ( ptsize != ((PExecution_Context)instance->exec)->pointSize/64 )
      {
         Reset_Context( instance, ptsize , 96L );
         Set_Raster_Area( instance, ptsize ); 
      }

      if ( LoadTrueTypeChar( Num ) )
      {
        for ( i = 0; i < gRec.points - 2; i++ )
        {
          gRec.xCoord[i] = gRec.xCoord[i] + x_off - 32;
          gRec.yCoord[i] = gRec.yCoord[i] + y_off - 32;
        }

        if ( ConvertRaster() )
        {
          sprintf( Header, "glyph index = %3d, char code = 0x%x",
                   Num, Code );
          Display_Bitmap_On_Screen( Bit.bitmap, Bit.rows, Bit.cols ); 
        }
        else
          Fail++;

#ifndef X11
#ifndef OS2
        Print_XY( 20, 0, Header );
#endif
#endif

      }
      else
        Fail++;

      if ( autorun )
      {
        xpos += cur_AdvanceWidth;
        if ( xpos + cur_AdvanceWidth > 640 )
        {
          xpos = 0;
          ypos += ( cur_Ascent - cur_Descent ) / 64;
          /* is == TRUETYPE DEFINED Precision */
        }

        Num = autorun++;
        if( !autorun )
        {
          xpos = 0;
          ypos += ( cur_Ascent - cur_Descent ) / 64;
          /* is == TRUETYPE DEFINED Precision */
        }
      }
      else
      {
        Get_Event( &event );
        if ( !Process_Event( &event ) ) goto Fin;
      }
    }

  Fin:

    RestoreScreen();
    TT_Close_File( stream );

    printf( "\n\nfails = %d\n", Fail );

    return 0;
}


/* End */
