{***************************************************************************}
{*                                                                         *}
{*  FreeType Glyph Viewer.                                                 *}
{*                                                                         *}
{*                                                                         *}
{*    This small program will load a TrueType font file and allow          *}
{*    you to view/scale/rotate its glyphs. Glyphs are in the order         *}
{*    found within the 'glyf' table.                                       *}
{*                                                                         *}
{*  NOTE : This version displays a magnified view of the glyph             *}
{*         along with the pixel grid.                                      *}
{*                                                                         *}
{*  This source code has been compiled and run under both Virtual Pascal   *}
{*  on OS/2 and Borland's BP7.                                             *}
{*                                                                         *}
{***************************************************************************}

program Abc;

uses Crt,

{$IFDEF OS2}
     Use32,
{$ENDIF}
     GMain,
     GEvents,
     GDriver,
     FreeType,
     TTTypes,
     TTMemory,
     TTFile,
     TTCalc,
     TTTables,
     TTRaster,
     TTIns,
     TTError;

{&PMTYPE NOVIO}

{$DEFINE DEBUG}

const
  Precis  = 64;
  Precis2 = Precis div 2;

  PrecisAux = 1024;

  Profile_Buff_Size = 64000;

var
  Font_Buffer : PStorage;

  num_pts : word;
  num_ctr : word;

  glyfArray : word;

  epts_ctr : PShort;

  xCoord : TT_PCoordinates;
  yCoord : TT_PCoordinates;
  Flag   : TT_PTouchTable;

  ymin, ymax, xmax, xmin, xsize : longint;
  res, old_res                  : int;

  numPoints, numContours : int;

  Bit : TRasterBlock;

  Rotation : int;  (* Angle modulo 1024 *)

  num_glyphs : int;

  stream     : TT_Stream;
  resident   : PResidentLeading;
  instance   : PInstanceRecord;
  gray_level : Boolean;

  display_outline : boolean;
  hint_glyph      : boolean;
  scan_type       : Byte;

  old_glyph : int;
  glyph     : int;

  scale_shift : Int;

  grayLines : array[0..2048] of Byte;

(*******************************************************************
 *
 *  Function    : InitRows
 *
 *  Description : Allocates the target bitmaps
 *
 *****************************************************************)

Procedure InitRows;
var
  i: integer;
  P: Pointer;
begin

  if gray_level then
  begin
    Bit.rows  := 200;
    Bit.cols  := 320;
    Bit.width := 320*2;
    Bit.flow  := TTFlowDown;
    Bit.size  := 320*200;
  end
  else
  begin
    Bit.rows  := 450;
    Bit.cols  := 80;
    Bit.width := 640;
    Bit.flow  := TTFlowDown;
    Bit.size  := 80*450;
  end;

  GetMem( Bit.buffer, Bit.size );
  if Bit.buffer = NIL then
   begin
    Writeln('ERREUR:InitRows:Not enough memory to allocate BitMap');
    halt(1);
   end;

  { Note that the render pool should be allocated from the font pool }
  { for vairous debugging reasons, and because we're still in alpha, }
  { we don't do it yet..                                             }

  GetMem( P, Profile_Buff_Size );
  if P=nil then
   begin
    writeln('ERREUR:InitRows:Not enough memory to allocate render pool');
    Halt(2);
   end;

  if gray_level then

    Init_Rasterizer( P,
                     Profile_Buff_Size,
                     @grayLines,
                     2048 )
  else
    Init_Rasterizer( P,
                     Profile_Buff_Size,
                     nil,
                     0 );

  FillChar( Bit.Buffer^, Bit.Size, 0 );
end;

(*******************************************************************
 *
 *  Function    :  ClearData
 *
 *  Description :  Clears the bitmaps
 *
 *****************************************************************)

Procedure ClearData;
var i: integer;
begin
  FillChar( Bit.Buffer^, Bit.Size, 0 );
end;

(*******************************************************************
 *
 *  Function    :  LoadTrueTypeChar
 *
 *  Description :  Loads a single glyph into the xcoord, ycoord and
 *                 flag arrays, from the instance data.
 *
 *****************************************************************)

Function LoadTrueTypeChar( index      : integer;
                           resolution : integer;
                           center_x   : integer;
                           center_y   : integer ) : boolean;

var
  off    : longint;
  x, y   : Longint;
  i      : integer;
  j      : word;
  EM     : Word;

begin
  LoadtrueTypeChar := FALSE;

  (* TT_Reset_Glyph( instance ); *)

  if not TT_Load_Glyph( instance, index ) then exit;

  numPoints   := instance^.pts.n;
  numContours := instance^.numContours;

  if (numPoints <= 0) or (numContours <= 0) then exit;

  for j := 0 to numPoints-1 do with instance^, pts do
  begin
    x := MulDiv_Round( org_x^[j], Scale1, Scale2 );
    y := MulDiv_Round( org_y^[j], Scale1, Scale2 );


    org_x^[j] := x;
    org_y^[j] := y;
    cur_x^[j] := x;
    cur_y^[j] := y;
    touch^[j] := touch^[j] and 1;
  end;

  if hint_glyph then
  begin

    Set_CodeRange( instance, 3, instance^.glyphIns, instance^.glyphSize );

    if not Goto_CodeRange( instance, TT_CodeRange_Glyph, 0 )
    then exit;

    TT_Glyph_Defaults( instance );

    with instance^, pts do
    begin
      cur_x^[n-2] := (org_x^[n-2]   ) and -64;
      cur_x^[n-1] := (org_x^[n-1]+32) and -64;
(*
      org_x^[n-2] := cur_x^[n-2];
      org_x^[n-1] := cur_x^[n-1];
*)
    (* Particularit de la fonte Georgia : si les deux lignes prcdentes *)
    (* sont commentes, le 'B' s'affiche correctement  10 pts, tandis    *)
    (* que le 'g' est trop troit d'un pixel. Si ces deux lignes ne sont  *)
    (* pas commentes, le 'g' est OK, le 'B' est trop troit !!           *)

    end;

    if not RunIns( instance ) then exit;

  end
  else
    for j := 0 to numPoints-1 do with instance^, pts do
    begin
      cur_x^[j] := (org_x^[j]+32) and -64;
      cur_y^[j] := (org_y^[j]+32) and -64;
    end;

  dec( numPoints, 2 );  (* remove phantom points *)

  for j := 0 to numPoints-1 do with instance^, pts do
  begin
    xCoord^[j] := (cur_x^[j] shl scale_shift) - 32;
    yCoord^[j] := (cur_y^[j] shl scale_shift) - 32;
    Flag^[j]   := touch^[j] and 1;
  end;

  LoadTrueTypeChar := TRUE;
end;


var
  Error_String : String;

function Render_ABC( res : integer ) : boolean;
var
  i, j : integer;
  G    : TGlyphRecord;

  x, y : longint;

  start_x,
  start_y,
  step_x,
  step_y  : longint;
begin

  Render_ABC := False;

  (* First, reset transform *)

  if not TT_Reset_Instance( instance, res, 96 ) then exit;

  start_x := 4;
  start_y := vio_Height - (50 shl (1-scale_shift));

  step_x  := instance^.ppem + 4;
  step_y  := instance^.ppem + 10;

  if not Goto_CodeRange( instance, TT_CodeRange_Cvt, 0 ) then
  begin
    Error_String := 'Could not go to CVT program';
    exit;
  end;

  if not RunIns( instance ) then
  begin
    Error_String := 'Error in CVT : ' + TT_ErrorStr;
    exit;
  end;

  TT_Set_Glyph_Defaults( instance );

  (* Then, begin render *)

  x := start_x;
  y := start_y;

  num_glyphs := resident^.numGlyphs;

  i := glyph;
  while i < num_glyphs do
  begin

    if LoadTrueTypeChar( i, res, 0, 0 ) then
    begin

      for j := 0 to numPoints-1 do
      begin
        inc( xcoord^[j], x*64 shl scale_shift );
        inc( ycoord^[j], y*64 shl scale_shift );
      end;

      G.numConts  := instance^.numContours;
      G.endPoints := instance^.endContours;
      G.Points    := numPoints;
      G.XCoord    := xCoord;
      G.YCoord    := yCoord;
      G.Flag      := Flag;

      if gray_level then
        Render_Gray_Glyph( G, @Bit, 2, nil )
      else
        Render_Glyph ( G, @Bit, 2 );

      with instance^, pts do inc( x, (cur_x^[n-1] div 64) + 1 );

      if x > vio_Width - 40 then
      begin
        x := start_x;
        dec( y, step_y );
        if y < 10 then
        begin
          Render_ABC := True;
          exit;
        end;
      end;

    end;

    inc(i);
  end;

  Render_ABC := True;

end;







procedure Usage;
begin
    Writeln('Simple TrueType Glyphs viewer - part of the FreeType project' );
    Writeln;
    Writeln('Usage : ',paramStr(0),' FontName[.TTF]');
    Halt(1);
end;



var i: integer;
    heure,
    min1,
    min2,
    sec1,
    sec2,
    cent1,
    cent2  :
{$IFDEF OS2}
    longint;
{$ELSE}
    word;
{$ENDIF}

    C : Char;

    Filename : String;

label Fin;

var
  Fail     : Int;
  glyphStr : String[4];
  event    : TEvent;

begin
  TextMode( co80+Font8x8 );

  GetMem       ( Font_Buffer, 64000 );
  Init_FontPool( Font_Buffer^, 64000 );

  num_pts   := 0;
  num_ctr   := 0;

  xCoord  := NIL;
  yCoord  := NIL;
  Flag    := NIL;

  gray_level := ParamStr(1)='-g';

  if gray_level then
    if ParamCount <> 2 then Usage else
  else
    if ParamCount <> 1 then Usage;

  if gray_level then Filename := ParamStr(2)
                else Filename := ParamStr(1);

  if Pos('.',FileName) = 0 then FileName:=FileName+'.TTF';

  if not TT_Open_File( FileName, stream ) then
    begin
      Writeln('ERROR: Could not open ', FileName );
      halt(1);
    end;

  if not TT_Load_Resident_Table( stream, resident ) then
   begin
    Writeln('ERROR: Could not load data from ', FileName );
    Halt(1);
   end;

  i := length(FileName);
  while (i > 1) and (FileName[i] <> '\') do dec(i);

  FileName := Copy( FileName, i+1, length(FileName) );

  i := resident^.MaxProfile.maxPoints;

  GetMem( XCoord, SizeOf(Fixed) * i );
  GetMem( YCoord, SizeOf(Fixed) * i );
  GetMem( Flag, i );

  if not TT_Load_Instance_Data( resident, instance ) then
   begin
    Writeln('ERROR: Could not open instance from ', FileName );
    Halt(1);
   end;

  Rotation  := 0;
  Fail      := 0;
  res       := 8;
  scan_type := 2;

  TT_Reset_Instance( instance, res, 96 );

  if resident^.fontPgmSize > 0 then
  begin

    if not Goto_CodeRange( instance, TT_CodeRange_Font, 0 ) then
    begin
      Writeln('Could not go to font program' );
      halt(1);
    end;

    if not RunIns( instance ) then
    begin
      Writeln('Error while running font program' );
      halt(1);
    end;

  end;

  if resident^.cvtPgmSize > 0 then
  begin

    if not Goto_CodeRange( instance, TT_CodeRange_Cvt, 0 ) then
    begin
      Writeln('Could not go to CVT program' );
      halt(1);
    end;

    if not RunIns( instance ) then
    begin
      Writeln('Error while running CVT program' );
      Writeln('> ', TT_ErrorStr, ' <' );
      halt(1);
    end;

  end;

  TT_Set_Glyph_Defaults( instance );

  InitRows;

  if ( gray_level ) then scale_shift := 1
                    else scale_shift := 0;


  if (gray_level) then
    Set_Graph_Screen( Graphics_Mode_Gray )
  else
    Set_Graph_Screen( Graphics_Mode_Mono );

  Set_High_Precision( True );

  display_outline := true;
  hint_glyph      := false;

  num_glyphs := instance^.fontres^.numGlyphs;

  old_glyph := -1;
  old_res   := res;
  glyph     := 24;

  Repeat

    if not Render_ABC( res ) then inc( Fail )
    else
      Display_Bitmap_On_Screen( Bit.Buffer^, Bit.rows, Bit.cols  );

    Print_XY( 0, 0, FileName );

    Print_Str('  point size = ');
    Str( instance^.pointSize div 64:3, glyphStr );
    Print_Str( glyphStr );

    Print_XY( 0, 1, 'Hinting  (''z'')  : ' );
    if hint_glyph then Print_Str('on ')
                  else Print_Str('off');

    Print_XY( 0, 2, 'scan type(''e'')  : ' );
    case scan_type of
      0 : Print_Str('none   ');
      1 : Print_Str('level 1');
      2 : Print_Str('level 2');
      4 : Print_Str('level 4');
      5 : Print_Str('level 5');
    end;

    ClearData;

    Get_Event(event);

    case event.what of

      event_Quit : goto Fin;

      event_Keyboard : case char(event.info) of

                        'z' : hint_glyph := not hint_glyph;

                        'e' : begin
                                inc( scan_type );
                                if scan_type  = 3 then scan_type := 4;
                                if scan_type >= 6 then scan_type := 0;
                              end;
                       end;

      event_Scale_Glyph : begin
                            inc( res, event.info );
                            if res < 1 then    res := 1;
                            if res > 1400 then res := 1400;
                          end;

      event_Change_Glyph : begin
                             inc( glyph, event.info );
                             if glyph < 0 then glyph := 0;
                             if glyph >= num_glyphs
                               then glyph := num_glyphs-1;
                           end;
    end;


  Until false;

 Fin:
  Restore_Screen;
  TT_Close_File(stream);

  Writeln('Echecs : ', Fail );
end.

