program Debugger;

uses
{$IFDEF OS2}
     Use32,
{$ENDIF}
     Drivers,
     Objects,
     Views,
     Menus,
     App,
     MsgBox,

     Crt,

     FreeType,
     TTIns,
     TTTypes,
     TTMemory,
     TTError,
     TTTables,
     TTFile,
     TTCalc,
     TTDebug,
     TTRaster,
     FullScr,

     CodeTv,
     StackTv,
     StateTv,
     ZoneTv;

{$I DEBUGGER.INC}

{ $DEFINE DEBUG_CVT}

const
  Precis = 64;

  Screen_Width  = 640;
  Screen_Height = 480;
  Screen_Cols   = Screen_Width div 8;
  Screen_Size   = Screen_Cols * Screen_Height;

  Grid_Width  = Screen_Width div 16;
  Grid_Height = Screen_Height div 16;
  Grid_Cols   = Grid_Width div 8;
  Grid_Size   = Grid_Cols * Grid_Height;

  Screen_Center_X = Screen_Width div 2;
  Screen_Center_Y = Screen_Height div 2;

  Grid_Center_X = Grid_Width div 2;
  Grid_Center_Y = Grid_Height div 2;

  Profile_Buff_Size = 64000;


type
  TDebug_Mode = ( debug_code, view_glyph );

  TMyApp = object( TApplication )
             constructor Init;
             procedure NewWindow; virtual;
             procedure InitMenuBar; virtual;
             procedure HandleEvent( var Event : TEvent ); virtual;

             procedure Single_Step;
             procedure Execute_Loop;
             procedure New_Execution;
             procedure ReFocus;
           end;

  TEtat = ( etat_Termine, etat_Arret, etat_Execution );

  TVolatileBreakPoint = record
                          range   : Int;
                          address : Int;
                        end;

var
  CW : PCodeWindow;
  SW : PStackWindow;
  GW : PStateWindow;
  ZW : PZoneWindow;

  Code_Range : array[1..3] of PCodeRange;

  Gen_Range : array[1..3] of TRangeRec;

  old_Range : Int;

  stream : TT_Stream;

  resident : PResidentLeading;
  instance : PInstanceRecord;

  Etat : TEtat;

  Volatiles : PBreakPoint;

  xCoord : TT_PCoordinates;
  yCoord : TT_PCoordinates;
  Flag   : TT_PTouchTable;

  Bitmap_small : TRasterBlock;
  Bitmap_big   : TRasterBlock;

  display_outline : boolean;
  hint_glyph      : boolean;

  debug_mode : TDebug_Mode;

  MyApp : TMyApp;

  Range       : Int;
  P           : PByteArray;
  FileName    : String;
  Font_Buffer : PStorage;
  Out_File    : Text;
  T, I        : int;


procedure Initialize;
var
  i : int;
begin
  for i := 1 to 3 do Code_Range[i] := Get_CodeRange(instance,i);
  for i := 1 to 3 do Generate_Range( Code_Range[i], i, Gen_Range[i] );

  Volatiles := nil;

  display_outline := true;
  Debug_Mode      := debug_code;
end;

(*******************************************************************
 *
 *  Function    : InitRows
 *
 *  Description : Allocates the target bitmaps
 *
 *****************************************************************)

Procedure InitRows;
var
  P: Pointer;
begin

  (* The big bitmap will contain the grid, the glyph contours and *)
  (* the magnified bitmap                                         *)

  Bitmap_big.rows  := Screen_Height;
  Bitmap_big.cols  := Screen_Cols;
  Bitmap_big.width := Screen_Width;
  Bitmap_big.flow  := TTFlowDown;
  Bitmap_big.size  := Screen_Size;

  GetMem( Bitmap_big.buffer, Bitmap_big.size );
  if Bitmap_big.buffer = NIL then
   begin
    Writeln('ERREUR:InitRows:Not enough memory to allocate big BitMap');
    halt(1);
   end;

  (* The small bitmap contains the rendered glyph, and is then later *)
  (* magnified into the big bitmap                                   *)

  Bitmap_small.rows  := Grid_Height;
  Bitmap_small.cols  := Grid_Cols;
  Bitmap_small.width := Grid_Width;
  Bitmap_small.flow  := TTFlowDown;
  Bitmap_small.size  := Grid_Size;

  GetMem( Bitmap_small.buffer, Bitmap_small.size );
  if Bitmap_small.buffer = NIL then
   begin
    Writeln('ERREUR:InitRows:Not enough memory to allocate big BitMap');
    halt(1);
   end;

  (* Note that the render pool should be allocated from the font pool *)
  (* for various debugging reasons, and because we're still in alpha, *)
  (* we don't do it yet..                                             *)

  GetMem( P, Profile_Buff_Size );
  if P=nil then
   begin
    writeln('ERREUR:InitRows:Not enough memory to allocate render pool');
    Halt(2);
   end;

  Init_Rasterizer( P,
                   Profile_Buff_Size,
                   nil,
                   0 );

  FillChar( Bitmap_big.Buffer^, Bitmap_big.Size, 0 );
  FIllChar( Bitmap_small.Buffer^, Bitmap_small.size, 0 );
end;

(*******************************************************************
 *
 *  Function    :  ClearData
 *
 *  Description :  Clears the bitmaps
 *
 *****************************************************************)

Procedure ClearData;
var i: integer;
begin
  FillChar( Bitmap_big.Buffer^, Bitmap_big.Size, 0 );

  FIllChar( Bitmap_small.Buffer^, Bitmap_small.size, 0 );
end;


function Render_Magnified : boolean;
label
  Exit_1;
type
  TBlock = array[0..7] of Byte;
  PBlock = ^TBlock;
const
{
  Grid_Empty : TBlock
             = ( $10, $10, $10, $FF, $10, $10, $10, $10 );
}
  Grid_Pixel2 : TBlock
              = ( $FE, $FE, $FE, $FE, $FE, $FE, $FE, $00 );

  Pixel_Center_X = 3;
  Pixel_Center_Y = 3;

  Grid_Empty : TBlock
             = ( $00, $00, $00, $10, $00, $00, $00, $00 );

  Grid_Pixel1 : TBlock
              = ( $00, $00, $38, $38, $38, $00, $00, $00 );

  Big_Center_X = Grid_Center_X*16 + Pixel_Center_X;
  Big_Center_Y = Grid_Center_Y*16 + Pixel_Center_Y;

var
  r, w, w2, u, v, b, c : integer;

  x, y : Long;

  block : PBlock;
  G     : TGlyphRecord;

  pixel,
  empty : PBlock;

  numPoints : integer;
begin
  Render_Magnified := False;

  numpoints := instance^.pts.n - 2; (* Remove phantom points *)

  for r := 0 to numPoints-1 do with instance^, pts do
  begin
    xCoord^[r] := cur_x^[r] + 31;
    yCoord^[r] := cur_y^[r] + 31;
    Flag^[r]   := touch^[r] and 1;
  end;

  (* We begin rendering the glyph within the small bitmap *)

  G.numConts  := instance^.numContours;
  G.endPoints := instance^.endContours;
  G.Points    := numPoints;
  G.XCoord    := xCoord;
  G.YCoord    := yCoord;
  G.Flag      := Flag;

  if not Render_Glyph ( G, @Bitmap_small, 2 ) then goto Exit_1;

  (* Then, we render the glyph outline in the bit bitmap *)

  for r := 0 to numPoints-1 do
  begin
    x := xcoord^[r]-31;
    y := ycoord^[r]-31;

    x := (x - Precis*Grid_Center_X)*16 + Big_Center_X*Precis;
    y := (y - Precis*Grid_Center_Y)*16 + Big_Center_Y*Precis;

    xcoord^[r] := x + 8*64;
    ycoord^[r] := y + 8*64;
  end;

   (* first compute the magnified coordinates *)

  G.numConts  := instance^.numContours;
  G.endPoints := instance^.endContours;
  G.Points    := numPoints;
  G.XCoord    := XCoord;
  G.YCoord    := YCoord;
  G.Flag      := Flag;

  if display_outline then
    if not Render_Glyph ( G, @Bitmap_big, 2 ) then goto Exit_1;

  (* Now, magnify the small bitmap, XORing it to the big bitmap *)

  r := 0;
  w := 0;
  b := 0;

  empty := @Grid_Empty;

  if display_outline then pixel := @Grid_Pixel1
                     else pixel := @Grid_Pixel2;

  for y := 0 to Grid_Height-1 do
  begin

    for x := 0 to Grid_Width-1 do
    begin

      w2 := w;
      b  := b shr 1;

      if b = 0 then
      begin
        c := PByte(Bitmap_small.Buffer)^[r];
        b := $80;
        inc( r );
      end;

      if c and b <> 0 then block := pixel
                      else block := empty;

      for v := 0 to 7 do
      begin
        PByte(Bitmap_Big.Buffer)^[w2] := PByte(Bitmap_Big.Buffer)^[w2]
                                         xor block^[v];
        inc( w2, Bitmap_Big.cols );
      end;

      inc( w, 2 );

    end;

    inc( w, 15*Screen_Cols );

  end;


  (* Display the resulting big bitmap *)

  Display( Bitmap_big.Buffer^, 450, 80  );

Exit_1:
  (* Clear the bitmaps *)

  ClearData;

  Render_Magnified := True;
end;


procedure Exit_Viewer;
begin
  RestoreScreen;
  debug_mode := debug_code;
  MyApp.SetScreenMode( smCo80 + smFont8x8 );
  MyApp.Show;
  MyApp.ReDraw;
end;


procedure Enter_Viewer;
begin
  SetGraphScreen( FS_Graphics_Mono );

  if not Render_Magnified then
    Exit_Viewer
  else
    debug_mode := view_glyph;
end;


procedure TMyApp.Execute_Loop;
var
  Out : Boolean;
  B   : PBreakPoint;

  Event : TEvent;
begin

  Out  := False;
  etat := etat_Execution;

  repeat

    Single_Step;

    B := Find_BreakPoint( Volatiles, instance^.curRange, instance^.IP );
    if B <> nil then
      begin
        Clear_Break( Volatiles, B );
        Out := True;
      end;

    if etat = etat_Execution then
      begin
        B := Find_BreakPoint( Gen_Range[instance^.curRange].Breaks,
                              instance^.curRange,
                              instance^.IP );
        if B <> nil then
          begin
            Out  := True;
            Etat := etat_Arret;
          end;
      end
    else
      Out := True;

  until Out;

end;


procedure TMyApp.New_Execution;
var
  Event : TEvent;
begin
  Event.What    := evWave;
  Event.Command := cmNewExecution;

  HandleEvent( Event );
end;


procedure TMyApp.Single_Step;
begin

  if not RunIns( instance ) then
  begin
    etat := etat_Termine;
    MessageBox( 'Error : '+TT_ErrorStr, nil, mfError+mfOkButton );
    exit;
  end;

  if instance^.IP >= instance^.codeSize then

    begin
      if (instance^.curRange <> TT_CodeRange_CVT) or
         not Goto_CodeRange( instance, TT_CodeRange_Glyph, 0 ) then

        begin
          etat := etat_Termine;
          MessageBox( 'Completed', nil, mfInformation+mfOkButton );
          exit;
        end;
    end
end;


procedure TMyApp.ReFocus;
var
  Event  : TEvent;
begin
  Event.What    := evCommand;

  if Old_Range <> instance^.curRange then
  begin
    Old_Range     := instance^.curRange;
    Event.Command := cmChangeRange;
    Event.InfoPtr := @Gen_Range[Old_Range];
    CW^.HandleEvent( Event );
  end;

  Event.What    := evWave;
  Event.Command := cmRefocus;

  if etat <> etat_Termine then
    Event.InfoInt := Get_Dis_Line( Gen_Range[Old_Range], instance^.IP )
  else
    Event.InfoInt := -1;

  HandleEvent( Event );
end;


procedure TMyApp.NewWindow;
var
  R  : TRect;
  RR : TRangeRec;
begin
  Desktop^.GetExtent(R);
  R.B.X := 32;

  Old_Range := instance^.curRange;

  New( CW, Init( R, @Gen_Range[Old_Range] ) );
  Desktop^.Insert(CW);

  Desktop^.GetExtent(R);
  R.A.X := 32;
  R.B.X := 50;
  R.B.Y := R.B.Y div 2;

  New( SW, Init( R, instance ) );
  Desktop^.Insert(SW);

  Desktop^.GetExtent(R);
  R.A.X := 50;
  R.B.Y := R.B.Y div 2;

  New( GW, Init( R, instance ) );
  Desktop^.Insert(GW);

  Desktop^.GetExtent(R);
  R.A.X := 32;
  R.A.Y := R.B.Y div 2;

{$IFDEF DEBUG_CVT}
  New( ZW, Init( R, @instance^.twilight ) );
{$ELSE}
  New( ZW, Init( R, @instance^.pts ) );
{$ENDIF}
  Desktop^.Insert(ZW);


  etat := etat_Arret;
end;


procedure TMyApp.InitMenuBar;
var
  R : TRect;
begin
  GetExtent(R);
  R.B.Y := R.A.Y + 1;
  MenuBar := New( PMenuBar, Init( R, NewMenu(
                  NewSubMenu( '~F~ile', hcNoContext, NewMenu(
                        NewItem( '~O~pen','F3', kbF3, cmFileOpen,
                                 hcNoContext,
                           nil )),
                   NewSubMenu( '~R~un', hcNoContext,
                       NewMenu(
                         NewItem( '~R~un','Ctrl-F9', kbCtrlF9,
                                  cmRun, hcNoContext,

                          NewItem( '~G~o to cursor','F4', kbF4,
                                   cmGoToCursor, hcNoContext,

                           NewItem( '~T~race into', 'F7', kbF7,
                                    cmTraceInto, hcNoContext,

                            NewItem( '~S~tep over', 'F8', kbF8,
                                     cmStepOver, hcNoContext,

                             NewItem( '~V~iew glyph', 'F9', kbF9,
                                       cmViewGlyph, hcNoContext,
                                       nil
                                    )
                                   )
                                  )
                                 )
                                )
                              ),
                  nil
                )))));
end;


procedure TMyApp.HandleEvent( var Event : TEvent );
var
  Adr : Long;
begin

  if debug_mode = view_glyph then
  begin

    case Event.What of

      evKeyDown : case Event.KeyCode of

                    kbF2  : begin
                              display_outline := not display_outline;

                              if not Render_Magnified then
                                Exit_Viewer;

                            end;

                    kbESC : Exit_Viewer;

                  end;
    end;

    ClearEvent( Event );
    exit;

  end;

  inherited HandleEvent(Event);

  case Event.What of

    evCommand : case Event.Command of

                  cmNewWin : NewWindow;

                  cmGoToCursor : begin
                                   if etat = etat_Termine then exit;

                                   Event.Command := cmQueryCursorAddr;
                                   Event.InfoPtr := @Adr;

                                   CW^.HandleEvent( Event );

                                   Set_Break( Volatiles,
                                              instance^.curRange,
                                              Adr );

                                   New_Execution;
                                   Execute_Loop;
                                   ReFocus;
                                 end;

                  cmTraceInto : begin
                                  if etat = etat_termine then exit;

                                  New_Execution;
                                  Single_Step;
                                  ReFocus;
                                end;

                  cmStepOver : begin
                                 if etat = etat_termine then exit;

                                 New_Execution;
                                 with instance^ do
                                 case code^[IP] of

                                   $2A,  (* LOOPCALL *)
                                   $2B : (* CALL     *)

                                   begin

                                     Set_Break( Volatiles,
                                                instance^.curRange,
                                                instance^.IP +
                                                Get_Length( instance^.Code,
                                                            instance^.IP ) );
                                     Execute_Loop;
                                   end;

                                 else

                                   Single_Step;
                                 end;

                                 ReFocus;
                               end;

                  cmViewGlyph :
                                Enter_Viewer;

                else
                  exit;
                end;

  else
    exit;
  end;

  ClearEvent(Event);
end;


constructor TMyApp.Init;
begin
  inherited Init;
  SetScreenMode( smCo80 + smFont8x8 );
  NewWindow;
end;



(*******************************************************************
 *
 *  Function    :  LoadTrueTypeChar
 *
 *  Description :
 *
 *  Notes  :
 *
 *****************************************************************)

 Function LoadTrueTypeChar( idx : integer ) : boolean;
 var
   x, y   : longint;
   j      : word;
   CR, SR : Real;

   numPoints, numContours : integer;

 begin
   LoadtrueTypeChar := FALSE;

   if not TT_Load_Glyph( instance, idx ) then exit;

   numPoints   := instance^.pts.n;
   numContours := instance^.numContours;

   if (numPoints <= 0) or (numContours <= 0) then exit;

   (* Now compute the scaled glyph point coordinates *)

   (* no rotation nor strecth there *)

   for j := 0 to numPoints-1 do with instance^, pts do
   begin
     x := MulDiv_Round( org_x^[j], Scale1, Scale2 );
     y := MulDiv_Round( org_y^[j], Scale1, Scale2 );

     org_x^[j] := x;
     org_y^[j] := y;
     cur_x^[j] := x;
     cur_y^[j] := y;
   end;

   LoadTrueTypeChar := TRUE;
 end;



begin

  if ParamCount <> 1 then
  begin
    Writeln('Simple Library Debugger -- part of the FreeType project');
    Writeln('-----------------------------------------------------');
    Writeln;
    Writeln(' Usage :  debugger fontfile[.ttf]');
    Writeln;
    halt(2);
  end;

  filename := ParamStr(1);
  if Pos( '.', filename ) = 0 then filename := filename + '.ttf';

  GetMem       ( Font_Buffer, 64000 );
  Init_FontPool( Font_Buffer^, 64000 );

  if not TT_Open_File( filename, stream ) then
  begin
    Writeln('Could not open file ',filename );
    halt(1);
  end;

  if not TT_Load_Resident_Table( stream, resident ) then
  begin
    Writeln('Could not load resident table' );
    halt(1);
  end;

  Writeln('OK');

  Writeln('Resident Table Size = ', resident^.totalSize );

  if not TT_Load_Instance_Data( resident, instance ) then
  begin
    Writeln('Could not create instance table' );
    halt(1);
  end;

  i := resident^.MaxProfile.maxPoints;

  GetMem( XCoord, SizeOf(Fixed) * i );
  GetMem( YCoord, SizeOf(Fixed) * i );
  GetMem( Flag, i );


  Writeln('Instance Data Size  = ', instance^.totalSize );

  TT_Reset_Instance( instance, 12, 96 );

  if not Goto_CodeRange( instance, TT_CodeRange_Font, 0 ) then
  begin
    Writeln('Could not go to font program' );
    halt(1);
  end;

  if not RunIns( instance ) then
  begin
    Writeln('Error while running font program' );
    halt(1);
  end;

  if not Goto_CodeRange( instance, TT_CodeRange_Cvt, 0 ) then
  begin
    Writeln('Could not go to CVT program' );
    halt(1);
  end;

{$IFNDEF DEBUG_CVT}

  if not RunIns( instance ) then
  begin
    Writeln('Error while running CVT program' );
    Writeln( TT_ErrorStr );
    halt(1);
  end;

  TT_Set_Glyph_Defaults( instance );

  if not LoadTrueTypeChar( 5 )  then
  begin
    Writeln('Error while loading glyph' );
    halt(1);
  end;

  Set_CodeRange( instance, 3, instance^.glyphIns, instance^.glyphSize );

  if not Goto_CodeRange( instance, TT_CodeRange_Glyph, 0 ) then
  begin
    Writeln('Could not go to glyph instructions' );
    halt(1);
  end;

  TT_Glyph_Defaults( instance );

  with instance^, pts do
  begin
    cur_x^[n-2] := ((org_x^[n-2]) and -64);
    cur_x^[n-1] := ((org_x^[n-1]+32) and -64);
(*
    cur_x^[n-2] := org_x^[n-2];
    cur_x^[n-1] := org_x^[n-1];
*)
  end;


{$ENDIF}

  instance^.Instruction_Trap := True;

  Initialize;
  InitRows;

  MyApp.Init;
  MyApp.Run;
  MyApp.Done;

  TT_Close_File( stream );

end.
