(*******************************************************************
 *
 *  TTTables.Pas                                              1.1
 *
 *    TrueType Tables structures and handling (specification)
 *
 *  Copyright 1996 David Turner, Robert Wilhelm and Werner Lemberg
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *
 *  Difference between 1.0 and 1.1 : HUGE !!
 *
 *  - Changed the load model to get in touch with TTFile 1.1
 *  - Now loads one whole resident table in one call
 *  - defined resident and instance records/data
 *
 ******************************************************************)

Unit TTTables;

interface

{ $DEFINE DEBUG}

uses FreeType, TTTypes;

(***************************************************************************)
(*                                                                         *)
(*                      TrueType Table Types                               *)
(*                                                                         *)
(***************************************************************************)

type
  (* Graphics State                            *)
  (*                                           *)
  (* The Graphics State (GS) is managed by the *)
  (* instruction field, but does not come from *)
  (* the font file. Thus, we can use 'int's    *)
  (* where needed.                             *)
  (*                                           *)

  PGraphicsState = ^TGraphicsState;
  TGraphicsState = record
                     auto_flip               : boolean;
                     control_value_cutin     : TT_F26dot6;
                     delta_base              : int;
                     delta_shift             : int;

                     dualVector,
                     projVector,
                     freeVector              : TT_UnitVector;

                     gep0,
                     gep1,
                     gep2                    : int;

                     instruct_control        : byte;
                     loop                    : longint;

                     minimum_distance        : TT_F26dot6;
                     round_state             : int;

                     rp0,
                     rp1,
                     rp2                     : int;

                     scan_control            : Boolean;
                     single_width_cutin      : TT_F26dot6;
                     single_width_value      : TT_F26dot6;

                     (* additional fields *)

                     scan_type               : Int;
                   end;

  (* The default Graphics State as given by the TrueType specs  *)
  (* not that, unlike what is told in chapter 7, lots of fields *)
  (* of the GS that are modified by the CVT program must be     *)
  (* reset to their default values, and not kept for glyph      *)
  (* instructions..                                             *)

  (* More on this at the end of this file in the xx_Default     *)
  (* functions..                                                *)

const
  Default_GraphicsState : TGraphicsState
                        = (
                            auto_flip           : True;
                            control_value_cutin : 4*17;
                            delta_Base          : 9;
                            delta_Shift         : 3;
                            dualVector          : ( x:$4000; y:0 );
                            projVector          : ( x:$4000; y:0 );
                            freeVector          : ( x:$4000; y:0 );
                            gep0                : 1;
                            gep1                : 1;
                            gep2                : 1;
                            instruct_control    : 0;
                            loop                : 1;
                            minimum_distance    : 64;
                            round_state         : 1;
                            rp0                 : 0;
                            rp1                 : 0;
                            rp2                 : 0;
                            scan_control        : True;
                            single_width_cutin  : 0;
                            single_width_value  : 0;

                            scan_type           : 0
                          );

type
  (* TrueType Table Directory type *)

  TTableDir = Record
                version        : TT_Fixed;  (* should be $10000 *)
                numTables      : UShort;    (* Tables number    *)

                searchRange,             (* These parameters are only used  *)
                entrySelector,           (* for a dichotomy search in the   *)
                rangeShift     : UShort; (* directory. We ignore them       *)
               end;

  (* The 'TableDir' is followed by 'numTables' TableDirEntries *)

  TTableDirEntry = Record
                     Tag      : array[0..3] of Char; (*        table type *)
                     CheckSum : Long;                (*    table Checksum *)
                     Offset   : Long;                (* Table file offset *)
                     Length   : Long;                (*      Table length *)
                    end;

  TTableDirEntries = array[0..100] of TTableDirEntry;
  PTableDirEntries = ^TTableDirEntries;

  (* 'cmap' tables *)

  TCMapDir = record
               tableVersionNumber : UShort;
               numCMaps           : UShort;
             end;

  TCMapDirEntry = record
                    platformID         : UShort;
                    platformEncodingID : UShort;
                    offset             : Long;
                  end;

  TCMapDirEntries = array[0..10] of TCMapDirEntry;
  PCMapDirEntries = ^TCMapDirEntries;

  (* NOTE : The following types are not defined by the TrueType *)
  (*        spec. However, they represent the layout of the     *)
  (*        character mapping tables in memory. This could      *)
  (*        easily change in future versions of the library     *)

  (* Apple standard character to glyph index mapping table *)
  (* the glyphIdArray for this format has 256 entries      *)

  TCMap0 = record
           end;

  (* Format 2 is used for mixed 8/16bit encodings (usually CJK fonts) *)
  TCMap2 = record
             subHeaderKeys : array[0..255] of UShort;
             (* high byte mapping table            *)
             (* value = subHeader index * 8        *)
           end;

  (* the format 2 table contains a variable-length array of subHeaders   *)
  (* (at most 256 entries) whose size must be determined algorithmically *)
  TCMap2SubHeader = record
                      firstCode     : UShort; (* first valid low byte       *)
                      entryCount    : UShort; (* number of valid low bytes  *)
                      idDelta       : Short;  (* delta value to glyphIndex  *)
                      idRangeOffset : UShort; (* offset fr. here to 1stCode *)
                    end;

  TCMap2SubHeaders = array[0..100] of TCMap2SubHeader;
  PCMap2SubHeaders = ^TCMap2SubHeaders;

  (* Microsoft standard character to glyph index mapping table *)
  TCMap4 = record
             segCountX2    : UShort;  (* segments number * 2          *)
             searchRange   : UShort;  (* these parameters can be used *)
             entrySelector : UShort;  (* for a binary search          *)
             rangeShift    : UShort;
           end;

  (*The format 4 table contains segCount segments *)
  TCMap4Segment = record
                    endCount      : UShort;
                    startCount    : UShort;
                    idDelta       : UShort;
                    idRangeOffset : UShort;
                  end;
  TCMap4Segments = array[0..100] of TCMap4Segment;
  PCMap4Segments = ^TCMap4Segments;

  (* trimmed table mapping (for representing one subrange) *)
  TCMap6 = record
             firstCode  : UShort;  (* first character code of subrange      *)
             entryCount : UShort;  (* number of character codes in subrange *)
           end;

  TCMap0Table = record
                  Header       : TCMap0;
                  glyphIdArray : PUShort;
                end;
  PCMap0Table = ^TCMap0Table;

  TCMap2Table = record
                  Header       : TCMap2;
                  subHeaders   : PCMap2SubHeaders;
                  glyphIdArray : PUShort;
                end;
  PCMap2Table = ^TCMap2Table;

  TCMap4Table = record
                  Header        : TCMap4;
                  Segments      : PCMap4Segments;
                  glyphIdArray  : PUShort;
                end;
  PCMap4Table = ^TCMap4Table;

  TCMap6Table = record
                  Header        : TCMap6;
                  glyphIdArray  : PUShort;
                end;
  PCMap6Table = ^TCMap6Table;

  TCMapTable = record
                 Format  : word;
                 Length  : word;
                 Version : word;

                 case Byte of
                   0 : ( cmap0 : PCMap0Table );
                   2 : ( cmap2 : PCMap2Table );
                   4 : ( cmap4 : PCMap4Table );
                   6 : ( cmap6 : PCMap6Table );
               end;

  TCMapTables = array[0..9] of TCMapTable;
  PCMapTables = ^TCMapTables;

  (* table "maxp" of Maximum Profiles' *)

  TMaxProfile = Record
                  Version                 : TT_Fixed;
                  numGlyphs,
                  maxPoints,
                  maxContours,
                  maxCompositePoints,
                  maxCompositeContours,
                  maxZones,
                  maxTwilightPoints,
                  maxStorage,
                  maxFunctionDefs,
                  maxInstructionDefs,
                  maxStackElements,

                  maxSizeOfInstructions,
                  maxComponentElements,
                  maxComponentDepth       : UShort;
                end;

  (* table de type "head" *)

  TLongDateTime = record
                    L1,
                    L2 : long;
                  end;

  THeader = Record
             Table_Version : TT_Fixed;
             Font_Revision : TT_Fixed;

             CheckSum_Adjust : Long;
             Magic_Number    : Long;

             Flags        : UShort;
             Units_Per_EM : UShort;

             Created  : TLongDateTime;
             Modified : TLongDateTime;

             xMin : Short;
             yMin : Short;
             xMax : Short;
             yMax : Short;

             Mac_Style       : UShort;
             Lowest_Rec_PPEM : UShort;

             Font_Direction       : Short;
             Index_To_Loc_Format  : Short;
             Glyph_Data_Format    : Short;
            end;

  (* table "hhea" *)

  THorizontalHeader = record
                        Version             : Fixed;
                        Ascender            : Short;
                        Descender           : Short;
                        Line_Gap            : Short;

                        advance_Width_Max   : UShort;

                        min_Left_Side_Bearing  : Short;
                        min_Right_Side_Bearing : Short;
                        xMax_Extent            : Short;
                        caret_Slope_Rise       : Short;
                        caret_Slope_Run        : Short;

                        Reserved   : array[0..4] of Short;

                        metric_Data_Format    : Short;
                        number_Of_HMetrics    : UShort;
                      end;

  (* table "HMTX" *)

  TLongHorMetric = record
                     advance_Width : UShort;
                     lsb           : Short;
                   end;

  TTableHorMetrics = array[0..255] of TLongHorMetric;
  PTableHorMetrics = ^TTableHorMetrics;

  (* "loca" tableau des indices dans Glyph *)
  TLoca = Record
            Size  : word;
            Table : PStorage;
          end;

  PGlyph = ^TGlyph;
  TGlyph = Record
            numberOfContours,
            xMin,
            yMin,
            xMax,
            yMax,
            numberOfPoints : Short;

            endContours : PUShort;
            X           : PCoordinates;
            Y           : PCoordinates;
            Flag        : PByte;
           end;

  TGlyphs = array[0..100] of TGlyph;
  PGlyphs = ^TGlyphs;

(***************************************************************************)
(*                                                                         *)
(*                  FreeType Resident table Def                            *)
(*                                                                         *)
(***************************************************************************)

type
  TResidentLeading = record
                       totalSize  : LongInt;

                       stream     : TT_Stream;

                       numTables  : Int;
                       dirTables  : PTableDirEntries;

                       numCMaps   : Int;
                       cMaps      : PCMapTables;

                       maxProfile : TMaxProfile;
                       fontHeader : THeader;

                       numGlyphs  : Int;

                       numLocations   : Int;
                       glyphLocations : PStorage;

                       horizontalHeader : THorizontalHeader;
                       LongHMetrics     : PTableHorMetrics;
                       ShortMetrics     : PShort;

                       fontPgmSize : Int;
                       fontProgram : PByte;

                       cvtPgmSize  : Int;
                       cvtProgram  : PByte;

                       cvtSize : Int;
                       cvt     : PShort;

                     end;
  PResidentLeading = ^TResidentLeading;


(***************************************************************************)
(*                                                                         *)
(*                  FreeType Instance Types                                *)
(*                                                                         *)
(***************************************************************************)

const
  MaxCodeRanges = 3;
  (* There can only be 3 active code ranges at once :  *)
  (*   - the Font Program                              *)
  (*   - the CVT  Program                              *)
  (*   - a glyph's instructions set                    *)

type
  TCodeRange = record
                 Base : PByte;
                 Size : Int;
               end;
  PCodeRange = ^TCodeRange;

  (* defines a code range                                            *)
  (*                                                                 *)
  (* code ranges can be resident to a glyph ( i.e. the Font Program) *)
  (* while some others are volatile ( Glyph instructions )           *)
  (* tracking the state and presence of code ranges allows function  *)
  (* and instruction definitions within a code range to be forgotten *)
  (* when the range is discarded                                     *)

  TCodeRangeTable = array[1..MaxCodeRanges] of TCodeRange;

  (* defines a function/instruction definition record *)
  TDefRecord = record
                 Range  : Int;     (* in which code range is it located ? *)
                 Start  : Int;     (* where does it start ?               *)
                 Opc    : Byte;    (* function #, or instruction code     *)
                 Active : boolean; (* is the entry active ?               *)
               end;

  PDefArray = ^TDefArray;
  TDefArray = array[0..99] of TDefRecord;

  (* defines a call record, used to manage function calls *)
  TCallRecord = record
                  Caller_Range : Int;
                  Caller_IP    : Int;
                  Cur_Count    : Int;
                  Cur_Restart  : Int;
                end;

  (* defines a simple call stack *)
  TCallStack = array[0..99] of TCallRecord;
  PCallStack = ^TCallStack;

  (* subglyph transformation record *)
  PGlyphTransform = ^TGlyphTransform;
  TGlyphTransform = record
                      xx, xy : TT_Fixed;    (* trasnformation *)
                      yx, yy : TT_Fixed;    (*     matrix     *)
                      ox, oy : TT_F26dot6;  (* offsets        *)
                    end;

  PInstanceRecord = ^TInstanceRecord;


  TFunction_Round = function( aInstance    : PInstanceRecord;
                              distance     : TT_F26dot6;
                              compensation : TT_F26dot6 ) : TT_F26dot6;

  TFunction_Project = function( aInstance : PInstanceRecord;
                                Vx, Vy    :TT_F26dot6 ) : TT_F26dot6;

  TFunction_Move    = procedure( aInstance : PInstanceRecord;
                                 var Vx    : TT_F26dot6;
                                 var Vy    : TT_F26dot6;
                                 var touch : Byte;
                                 distance  : TT_F26dot6 );

  TInstanceRecord = record

                      totalSize : LongInt;  (* total size of instance data *)

                      fontRes  : PResidentLeading;

                      glyphIns  : PByte; (* glyph instructions *)
                      glyphSize : Int;   (* glyph ins. size    *)

                      curRange  : Int;   (* current code range number   *)
                      code      : PByte; (* current code range          *)
                      IP        : Int;   (* current instruction pointer *)
                      codeSize  : Int;   (* size of current range       *)

                      numFDefs : Int;        (* number of function defs *)
                      FDefs    : PDefArray;  (* table of FDefs entries  *)

                      numIDefs : Int;        (* number of instruction defs *)
                      IDefs    : PDefArray;  (* table of IDefs entries     *)

                      xMin     : Int;  (* current glyph's x minimum *)
                      yMin     : Int;  (* current glyph's y minimum *)
                      xMax     : Int;  (* current glyph's x maximum *)
                      yMax     : Int;  (* current glyph's y maximum *)

                      leftSideBearing : Int;  (* current glyph's lsb *)
                      advanceWidth    : Int;  (* current glyph's aw  *)

                      (* These two values should be cached with *)
                      (* the bitmap, they are here in FUnits    *)

                      callTop   : Int;
                      callSize  : Int;
                      callStack : PCallStack;

                   (* codeRanges     : Int;   obsolete *)

                      codeRangeTable : TCodeRangeTable;

                      storeSize : Int;      (* size of current storage area *)
                      storage   : PStorage; (* storage area                 *)

                      stackSize   : Int;      (* size of instance stack *)
                      top         : Int;      (* top of instance stack  *)
                      stack       : PStorage; (* current instance stack *)

                      period,                 (* values used for the *)
                      phase,                  (* 'SuperRounding'     *)
                      threshold   : TT_F26dot6;

                      scale1,          (* values used to scale between *)
                      scale2  : Long;  (* FUnits and Pixel coordinates *)

                      pointSize : TT_F26Dot6; (* current pointSize *)
                      ppem      : Int;        (* current PPEM      *)

                      zp0,
                      zp1,
                      zp2,
                      twilight,
                      pts         : TT_VecRecord;

                      numContours : Int;
                      endContours : PUShort;

                      Instruction_Trap : Boolean;

                      GS         : TGraphicsState;
                      default_GS : TGraphicsState;

                      cvtSize   : Int;
                      cvt       : PShort;

                      (* latest interpreter additions *)

                      F_dot_P   : Long;

                      compensations : array[0..3] of TT_F26dot6;

                      func_round    : TFunction_Round;
                      func_project  : TFunction_Project;
                      func_dualproj : TFunction_Project;
                      func_move     : TFunction_Move;

                    end;

 function TT_Load_Resident_Table( astream       : Int;
                                  var aResident : PResidentLeading
                                ) : Boolean;


 function TT_Load_Instance_Data( aResident      : PResidentLeading;
                                 var aInstance  : PInstanceRecord
                               ) : Boolean;

 function TT_Reset_Instance( aInstance  : PInstanceRecord;
                             ptSize     : Int;
                             resolution : Int ) : Boolean;

 procedure TT_Set_Glyph_Defaults( aInstance : PInstanceRecord );

 function TT_Glyph_Defaults( aInstance : PInstanceRecord ) : Boolean;

 function TT_Load_Glyph( aInstance : PInstanceRecord;
                         aIndex    : Int ) : Boolean;

 (* The following functions are used by the interpreter *)

const
  TT_CodeRange_Font   = 1;
  TT_CodeRange_Cvt    = 2;
  TT_CodeRange_Glyph  = 3;

 function Goto_CodeRange( aInstance : PInstanceRecord;
                          aRange,
                          aIP       : Int ): boolean;
 (* Jump to a specified range, at address AIP *)

 function Get_CodeRange( aInstance : PInstanceRecord;
                         aRange    : Int             ) : PCodeRange;
 (* returns a pointer to coderange number 'ARange' *)
 (* used only by the debugger                      *)

 function Set_CodeRange( aInstance : PInstanceRecord;
                         aRange    : Int;
                         aBase     : Pointer;
                         aLength   : Int       ) : boolean;
 (* Set a code range *)

 function Clear_CodeRange( aInstance : PInstanceRecord;
                           aRange    : Int              ) : boolean;
 (* Clear a code range *)


{$IFDEF DEBUG}
  Max_Ins_Glyph : integer;
  (* Maximum number of instructions of simple glyph *)

  Max_Ins_Composite : integer;
  (* Maximum number of instructions of composite glyph *)

  Num_Ins_Glyph      : integer;
  Num_Ins_Composite  : integer;

  Sum_Ins_Glyph     : integer;
  Sum_Ins_Composite : integer;
{$ENDIF}

implementation

uses TTError, TTMemory, TTFile, TTCalc;

  (* Composite glyph decoding flags *)

const
  ARGS_ARE_WORDS      = $01;
  ARGS_ARE_XY_VALUES  = $02;
  ROUND_XY_TO_GRID    = $04;
  WE_HAVE_A_SCALE     = $08;
  (* reserved           $10 *)
  MORE_COMPONENTS     = $20;
  WE_HAVE_AN_XY_SCALE = $40;
  WE_HAVE_A_2X2       = $80;
  WE_HAVE_INSTR       = $100;
  USE_MY_METRICS      = $200;


(*******************************************************************
 *
 *  Function    :  LookUp_TrueType_Table
 *
 *  Description :  Looks for a TrueType table by name
 *
 *  Input  :  aResident   resident table to look for
 *            aTag        searched tag
 *
 *  Output :  index of table if found, -1 otherwise.
 *
 ******************************************************************)

 function LookUp_TrueType_Table( aResident : PResidentLeading;
                                 aTag      : string
                               ) : int;
 var
   TAG : String[4];
   i   : int;
 begin
   TAG[0] := #4;
   LookUp_TrueType_Table := -1;

   for i := 0 to aResident^.numTables-1 do
     begin

       move( aResident^.dirTables^[i].Tag, Tag[1], 4 );

       if Tag = ATag then
         begin
           LookUp_TrueType_Table := i;
           exit;
         end

     end
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Directory
 *
 *  Description :
 *
 *  Input  :  aResident
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_Directory( aResident : PResidentLeading ) : Boolean;
 var
   t, n : Int;

   tableDir : TTableDir;
 begin
    Load_TrueType_Directory := False;

    {$IFDEF DEBUG}
    Write('Directory ');
    {$ENDIF}

    if not TT_Seek_File( 0 ) then exit;

    if not TT_Access_Frame( sizeof(TableDir) ) then exit;

    tableDir.version   := GET_Long;
    tableDir.numTables := GET_UShort;

    tableDir.searchRange   := GET_UShort;
    tableDir.entrySelector := GET_UShort;
    tableDir.rangeShift    := GET_UShort;

{$IFDEF DEBUG}
    Writeln('Tables number : ', tableDir.numTables );
{$ENDIF}

    TT_Forget_Frame;

    with aResident^ do
    begin

      numTables := tableDir.numTables;

      if not Alloc( numTables * sizeof( TTableDirEntry ), Pointer(dirTables) )
        then exit;

      if not TT_Access_Frame( 16 * numTables ) then exit;

      for n := 0 to numTables-1 do with dirTables^[n] do
      begin
        ULong(Tag) := GET_Tag4;
        Checksum   := GET_ULong;
        Offset     := GET_Long;
        Length     := Get_Long;
      end;

      TT_Forget_Frame;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_Directory := True;
   exit;

 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_MaxProfile
 *
 *  Description :
 *
 *  Input  :  aResident
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_MaxProfile( aResident : PResidentLeading ) : boolean;
 var
   t : int;
 begin

   Load_TrueType_MaxProfile := False;

    {$IFDEF DEBUG}
    Write('MaxProfile ');
    {$ENDIF}

   t := LookUp_TrueType_Table( aResident, 'maxp');
   if t < 0 then exit;

   with aResident^ do
   begin

     if not TT_Seek_File( dirTables^[t].Offset ) or
        not TT_Access_Frame( sizeof(MaxProfile) ) then exit;

     with MaxProfile do
      begin

        ULong(Version) := GET_ULong;

        numGlyphs   := GET_UShort;
        maxPoints   := GET_UShort;
        maxContours := GET_UShort;

        maxCompositePoints   := GET_UShort;
        maxCompositeContours := GET_UShort;
        maxZones             := GET_UShort;
        maxTwilightPoints    := GET_UShort;
        maxStorage           := GET_UShort;
        maxFunctionDefs      := GET_UShort;
        maxINstructionDefs   := GET_UShort;
        maxStackElements     := GET_UShort;

        maxSizeOfInstructions := GET_UShort;
        maxComponentElements  := GET_UShort;
        maxComponentDepth     := GET_UShort;
      end;

     TT_Forget_Frame;

     numGlyphs := MaxProfile.numGlyphs;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_MaxProfile := True;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Header
 *
 *  Description :  Load the TrueType header table in the resident
 *                 table
 *
 *  Input  :  aResident   current leading segment.
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function  Load_TrueType_Header( aResident : PResidentLeading ) : Boolean;
 var
   i : int;
 begin
   Load_TrueType_Header := False;

    {$IFDEF DEBUG}
    Write('Header ');
    {$ENDIF}

   i := LookUp_TrueType_Table(aResident, 'head');
   if i <= 0 then exit;

   with aResident^ do
   begin

     if not TT_Seek_File( dirTables^[i].offset ) or
        not TT_Access_Frame( sizeof(THeader) ) then exit;

     with FontHeader do
     begin

       ULong(Table_Version) := GET_ULong;
       ULong(Font_Revision) := GET_ULong;

       Checksum_Adjust := GET_Long;
       Magic_Number    := GET_Long;

       Flags        := GET_UShort;
       Units_Per_EM := GET_UShort;

       Created .L1 := GET_Long; Created .L2 := GET_Long;
       Modified.L1 := GET_Long; Modified.L2 := GET_Long;

       xMin := GET_Short;
       yMin := GET_SHort;
       xMax := GET_SHort;
       yMax := GET_Short;

       Mac_Style       := GET_UShort;
       Lowest_Rec_PPEM := GET_UShort;

       Font_Direction      := GET_Short;
       Index_To_Loc_Format := GET_Short;
       Glyph_Data_Format   := GET_Short;

   {$IFDEF DEBUG}
       Writeln('Units per EM       : ',Units_Per_EM );
       Writeln('IndexToLocFormat   : ',Index_To_Loc_Format );
   {$ENDIF}

     end;

     TT_Forget_Frame;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_Header := True;
 end;

(*******************************************************************
 *
 *  Function    : Load_TrueType_Horizontal_Header
 *
 *  Description :
 *
 *  Input  :  aResident   current resident leading segment
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_Horizontal_Header( aResident : PResidentLeading )
                                         : Boolean;
 var
   t : int;
 begin
   Load_TrueType_Horizontal_Header := False;

    {$IFDEF DEBUG}
    Write('Horizontal Header ');
    {$ENDIF}

   t := LookUp_TrueType_Table( aResident, 'hhea');
   if ( t < 0 ) then exit;

   with aResident^ do
   begin

     if not TT_Seek_File( dirTables^[t].Offset ) or
        not TT_Access_Frame( sizeof( THorizontalHeader ) ) then exit;

     with HorizontalHeader do
     begin

       Long(Version) := GET_ULong;
       Ascender      := GET_Short;
       Descender     := GET_Short;
       Line_Gap      := GET_Short;

       advance_Width_Max := GET_UShort;

       min_Left_Side_Bearing  := GET_Short;
       min_RIght_Side_Bearing := GET_Short;
       xMax_Extent            := GET_Short;
       caret_Slope_Rise       := GET_Short;
       caret_Slope_Run        := GET_Short;

       Reserved[0] := GET_Short;
       Reserved[1] := GET_Short;
       Reserved[2] := GET_Short;
       Reserved[3] := GET_Short;
       Reserved[4] := GET_Short;

       metric_Data_Format := GET_Short;
       number_Of_HMetrics := GET_UShort;

     end;

     TT_Forget_Frame;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_Horizontal_Header := True;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Locations
 *
 *  Description :  Loads the location table in resident table
 *
 *  Input  :  aResident     Current Resident Leading Segment
 *
 *  Output :  True on success. False on failure
 *
 *  NOTES :
 *
 *    The Font Header *must* be loaded in the leading segment
 *    calling this function
 *
 ******************************************************************)

 function Load_TrueType_Locations( aResident : PResidentLeading ): Boolean;
 var
   t, n        : int;
   LongOffsets : int;

 begin

   Load_TrueType_Locations := False;

    {$IFDEF DEBUG}
    Write('Locations ');
    {$ENDIF}

   with aResident^ do
   begin

     LongOffsets :=  fontHeader.Index_To_Loc_Format;

     T := LookUp_TrueType_Table( aResident, 'loca' );
     if t < 0 then exit;

     if not TT_Seek_File( dirTables^[T].Offset ) then exit;

     if LongOffsets <> 0 then
       begin

         numLocations := dirTables^[T].Length shr 2;

         {$IFDEF DEBUG}
         Writeln('Glyph locations # ( 32 bits offsets ) : ', numLocations );
         {$ENDIF}

         if not Alloc( sizeof(Long)*numLocations,
                       Pointer( glyphLocations ) ) or

            not TT_Access_Frame( numLocations*4 ) then exit;

         for n := 0 to numLocations-1 do
           glyphLocations^[n] := GET_Long;

         TT_Forget_Frame;

       end
     else
       begin

         numLocations := dirTables^[T].Length shr 1;

         {$IFDEF DEBUG}
         Writeln('Glyph locations # ( 16 bits offsets ) : ', numLocations );
         {$ENDIF}

         if not Alloc( sizeof(Long)*numLocations,
                       Pointer( glyphLocations ) ) or

            not TT_Access_Frame( numLocations*2 ) then exit;

         for n := 0 to numLocations-1 do
           glyphLocations^[n] := GET_UShort * 2;

         TT_Forget_Frame;

       end;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_Locations := True;
 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_CVT
 *
 *  Description :
 *
 *  Input  :  aResident
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_CVT( aResident : PResidentLeading ): boolean;
 var
   t, n : int;
 begin
   Load_TrueType_CVT := True;

   {$IFDEF DEBUG}
   Write('CVT ');
   {$ENDIF}

   t := LookUp_TrueType_Table( aResident, 'cvt ');
   if t<0 then
   begin
     aResident^.cvt     := nil;
     aResident^.cvtSize := 0;
     exit;
   end;

   with aResident^ do
   begin

     cvtSize := dirTables^[t].Length div 2;

     if not Alloc( sizeof(Short)*cvtSize, Pointer(cvt) ) or

        not TT_Seek_File( dirTables^[t].Offset ) then exit;

     if not TT_Access_Frame( 2*cvtSize ) then exit;

     for n := 0 to cvtSize-1 do
       cvt^[n] := GET_Short;

     TT_Forget_Frame;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_CVT := True;
 end;


(*******************************************************************
 *
 *  Function    :  TT_Load_Glyph
 *
 *  Description :  Load a given glyph in an instance record
 *
 *  Input  :  aInstance   target instance table
 *            aIndex      glyph index
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function TT_Load_Glyph( aInstance : PInstanceRecord;
                         aIndex    : Int ) : Boolean;
 var
   fontRes : PResidentLeading;

   sz, szc, szp : int;
   i, j, k, cnt : int;
   b,
   c      : byte;

   offset : longint;
   locs   : PStorage;

   pts : ^TT_VecRecord;

   GL  : TGlyph;

 label
   Fin;

 begin
   TT_Load_Glyph := False;
(*
   if aInstance = nil then
     begin
       Error := TT_ErrMsg_Invalid_Font_Handle;
       exit;
     end;
*)
   fontRes := aInstance^.fontRes;

   if (aIndex < 0) or (aIndex > fontRes^.numGlyphs) then
     begin
       Error := TT_ErrMsg_Invalid_Glyph_Index;
       exit;
     end;

   i := LookUp_TrueType_Table( fontRes, 'glyf');
   if i < 0 then exit;

   (* clear current execution context *)

   with aInstance^ do
   begin
     codeSize := 0;
     IP       := 0;
     curRange := -1;
     callTop  := 0;
     top      := 0;

     Clear_CodeRange( aInstance, TT_CodeRange_Glyph );
   end;

   (* now try to load everything *)

   if not TT_Lock_Access( aInstance^.fontRes^.stream ) then exit;

   offset := fontRes^.dirTables^[i].Offset +
             fontRes^.glyphLocations^[aIndex];

   if not TT_Seek_File( offset ) or
      not TT_Access_Frame( 5*sizeof(UShort) ) then goto Fin;

   Gl.numberOfContours := GET_UShort;

   Gl.xMin := GET_UShort;
   Gl.yMin := GET_UShort;
   Gl.xMax := GET_UShort;
   Gl.yMax := GET_UShort;

   TT_Forget_Frame;

   aInstance^.xMin := Gl.xMin;
   aInstance^.yMin := Gl.yMin;
   aInstance^.xMax := Gl.xMax;
   aInstance^.yMax := Gl.yMax;

   {$IFDEF DEBUG}
   Writeln('Glyph ', i );

   Writeln(' # of Contours : ',Gl.numberOfContours );
   Writeln(' xMin : ',Gl.xMin:4,'  xMax : ',Gl.xMax);
   Writeln(' yMin : ',Gl.yMin:4,'  yMax : ',Gl.yMax);
   Writeln('-');
   {$ENDIF}

   szc := Gl.numberOfContours;

   if (szc < 0) then goto Fin;

   { Composite Glyphs unsupported yet }

   if (szc > fontRes^.MaxProfile.maxContours) then
     begin
       {$IFDEF DEBUG}
       Writeln( 'ERROR: Glyph index ',i,' has ',Gl.numberOfContours,
                ' contours > ', fontRes^.MaxProfile.maxContours );
       {$ENDIF}
       Error := TT_ErrMsg_Invalid_Contours;
       goto Fin;
     end;

   { Reading the contours endpoints }

   Szp := 0;

   if not TT_Access_Frame( (szc + 1)*sizeof(Short) ) then
     goto Fin;

   For k := 0 to szc-1 do
    begin

     {$IFDEF DEBUG} Write( szp,' '); {$ENDIF}

     szp                        := GET_Short;
     aInstance^.endContours^[k] := szp;
     inc( szp );
    end;

   { Loading instructions }

   k := GET_Short;

   TT_Forget_Frame;

   {$IFDEF DEBUG}
   Writeln('Instructions size : ',k);
   {$ENDIF}

   if k > fontRes^.MaxProfile.maxSizeOfInstructions then
   begin
     {$IFDEF DEBUG}
     Writeln('Too many instructions');
     {$ENDIF}
     Error := TT_ErrMsg_Too_Many_Ins;
     goto Fin;
   end;

   with aInstance^ do
   begin

     if not TT_Read_File( glyphIns^, k ) then
       goto Fin;

     glyphSize := k;

     if not Set_CodeRange( aInstance,
                           TT_CodeRange_Glyph,
                           glyphIns,
                           glyphSize ) then goto Fin;
   end;

   if not TT_Check_And_Access_Frame( szp*( sizeof(Byte) + 2*sizeof(Short) ) )
     then goto Fin;

   pts := @aInstance^.pts;

   { Reading Flags }

     k := 0;

     while ( k < szp ) do
      begin

       c              := GET_Byte;
       Pts^.touch^[k] := c;
       inc(k);

       if c and 8 <> 0 then
        begin
         cnt := GET_Byte;

         while (cnt > 0) do
          begin
           Pts^.touch^[k] := c;
           inc( k );
           dec( cnt );
          end;
        end;
      end;

   { Reading the Xs }

     for k := 0 to szp-1 do with pts^ do

       if touch^[k] and 2 <> 0 then

          if touch^[k] and 16 <> 0 then org_x^[k] :=  GET_Byte
                                   else org_x^[k] := -GET_Byte
       else
          if touch^[k] and 16 <> 0 then org_x^[k] := 0
                                   else org_x^[k] := GET_Short;

   { Reading the Ys }

     for k := 0 to szp-1 do with pts^ do

       if touch^[k] and 4 <> 0 then

         if touch^[k] and 32 <> 0 then org_y^[k] :=  GET_Byte
                                  else org_y^[k] := -GET_Byte
       else
         if touch^[k] and 32 <> 0 then org_y^[k] := 0
                                  else org_y^[k] := GET_Short;

   TT_Forget_Frame;

   { Relative to Absolute conversion }

     for k := 1 to szp-1 do with Pts^ do
      begin
       inc( org_x^[k], org_x^[k-1] );
       inc( org_y^[k], org_y^[k-1] );
      end;

   { Now adds the two shadow points at n and n+1 }

   { We need the left side bearing and advance width }

   with aInstance^, fontres^ do
   begin

     k := fontres^.horizontalHeader.number_Of_HMetrics;

     if aIndex < k then
       begin
         leftSideBearing := LongHMetrics^[aIndex].lsb;
         advanceWidth    := LongHMetrics^[aIndex].advance_Width;
       end
     else
       begin
         leftSideBearing := ShortMetrics^[aIndex-k];
         advanceWidth    := LongHMetrics^[k-1].advance_Width;
       end;

     pts.org_x^[szp] := Gl.xMin - leftSideBearing;
     pts.org_y^[szp] := 0;      (* pp1 = xMin - lsb *)

     pts.org_x^[szp+1] := Gl.xMin - leftSideBearing + advanceWidth;
     pts.org_y^[szp+1] := 0;    (* pp2 = pp1 + aw   *)

   end;

   for k := 0 to szp+1 do with pts^ do cur_x^[k] := org_x^[k];
   for k := 0 to szp+1 do with pts^ do cur_y^[k] := org_y^[k];
   for k := 0 to szp+1 do with pts^ do touch^[k] := touch^[k] and 1;

   inc( szp, 2 );

   (* Note that we now return two more points, that are not *)
   (* part of the glyph outline                             *)

   aInstance^.pts.n       := szp;
   aInstance^.numContours := szc;

   TT_Load_Glyph := True;

 Fin:
   TT_Release_Access( aInstance^.fontRes^.stream );

 end;

{
 function Load_Glyph( instance         : PInstanceRecord;
                      glyphIndex       : Int;
                      var cur_contour  : Int;
                      var cur_point    : Int;
                      var cur_subglyph : Int ) : boolean;
 var
   fontres : PResidentLeading;
   glyph   : PGlyph;

   i, old_subglyph : Int;

   szc : Short;

   offset, here : Long;

   subglyphIndex, temp, flags : UShort;

   arg1, arg2 : Short;

   gs      : PGlyphScales;
   curr_gs : PGlyphScale;

   x_scale : TT_Fixed;
   scale01 : TT_Fixed;
   scale10 : TT_Fixed;
   y_scale : TT_Fixed;

   n1, n2, n3, n4 : TT_Fixed;

   x_scale_2d14 : TT_F2Dot14;
   scale01_2d14 : TT_F2Dot14;
   scale10_2d14 : TT_F2Dot14;
   y_scale_2d14 : TT_F2Dot14;

   x_offset : TT_F26dot6;
   y_offset : TT_F26dot6;

 begin

   Load_Glyph := False;

   x_scale := 1 shl 16;
   scale01 := 0;
   scale10 := 0;
   y_scale := 1 shl 16;

   x_offset := 0;
   y_offset := 0;

   fontres := instance^.fontres;

   i := LookUp_TrueType_Table( fontres, 'glyf' );
   if i < 0 then exit;

   offset := fontres^.dirTables^[i].offset;

   (* prep and read in first 5 words of info about glyph *)
   (* some fonts handle the end of the glyph index in a  *)
   (* non-standard way                                   *)

   if fontres^.glyphLocations^[glyphIndex  ] =
      fontres^.glyphLocations^[glyphIndex+1] then
     begin
       if not TT_Seek_File( offset + fontres^.glyphLocations^[0] )
         then exit;
     end
   else
     begin
       if not TT_Seek_File( offset + fontres^.flyphLocations^[glyphIndex] )
         then exit;
     end;

   if not TT_Access_Frame( 5*2 ) then exit;

   (* please note that these axtremeas are not reliable in all cases *)

   with glyph^ do
   begin
     numberOfContours := GET_UShort;

     xMin := GET_UShort;
     yMin := GET_UShort;
     xMax := GET_UShort;
     yMax := GET_UShort;
   end;

   TT_Forget_Frame;
}
(*
{$IFDEF DEBUG}
  with glyph^ do
  begin
     writeln( 'Glyph ', glyphIndex, ' @', offset );
     writeln( ' # contours : ', numberOfContours );
     writeln( ' xMin       : ', xMin );
     writeln( ' yMin       : ', yMin );
     writeln( ' xMax       : ', xMax );
     writeln( ' yMax       : ', yMax );
     writeln( '-' );
   end;
{$ENDIF}
*)
{
   szc := glyph^.numberOfContours;
   gs  := glyph^.scales;

   if ( szc > fontres^.maxProfile.maxContours          ) and
      ( szc > fontres^.maxProfile.maxCompositeContours )
     then exit
   else
   if szc = 0 then
     begin
       Load_Glyph := True;
       exit;
     end
   else
     if szc > 0 then     (* simple glyph *)
       begin
         gs^[cur_subglyph].start := cur_point;

         if not Load_Simple_Glyph( instance,
                                   glyphIndex,
                                   szc,
                                   cur_contour,
                                   cur_point )
           then exit;

         (* set default values *)

         with gs^[cur_subglyph] do
         begin
           flags    := 0;
           finish   := cur_point-1;
           x_offset := 0;
           y_offset := 0;
           x_scale  := 1 shl 16;
           scale01  := 0;
           scale10  := 0;
           y_scale  := 1 shl 16;
         end;

         inc( cur_contour, szc );
         inc( cur_subglyph );

         with glyph^ do
         begin
           numbreOfContours  := cur_contour;
           numberOfPoints    := cur_point;
           numberOfSubglyphs := cur_subglyph;
         end;

       end
     else      (* composite glyph *)
       begin

         repeat

           old_subglyph := cur_subglyph;

           if not TT_Access_Frame(4) then exit;

           flags         := GET_UShort;
           subglyphIndex := GET_UShort;

           TT_Forget_Frame;

           (*
             the only flag which is of relevance is ROUND_XY_TO_GRID. We
             ignore this flag from deeper levels---USE_MY_METRICS is not
             implemented yet. The grid rounding flag is the very reason
             that we can't apply the scaling values immediately but rather
             have to collect them in a separate array.
           *)

           here := TT_File_Pos;  (* keep track of our current position *)

           Load_Glyph( instance,
                       subglyphIndex,
                       cur_contour,
                       cur_point,
                       cur_subglyph );

           TT_Seek_File( here );

           (* access the entire frame for the composite in one block *)

           i = 2;  (* number of bytes needed if no flags are set *)

           if flags and ARGS_ARE_WORDS <> 0 then inc( i, 2 );

           (* ignore ARGS_ARE_XY_VALUES as it doesn't change the load size *)

           if flags and WE_HAVE_A_SCALE <> 0 then inc( i, 2 );

           if flags and WE_HAVE_AN_XY_SCALE <> 0 then inc( i, 4 );

           if flags and WE_HAVE_A_2X2 <> 0 then inc( i, 8 );

           if not TT_Access_Frame( i ) then exit;

           (* now read the args *)

           if flags and ARGS_ARE_WORDS <> 0 then
             begin
               arg1 := GET_Short;
               arg2 := GET_Short;
             end
           else
             begin
               temp := Get_USHORT;

               (* the specs don't say whether this is signed or unsigned. But *)
               (* arial.ttf needs signed...                                   *)

               arg1 := ShortInt( temp shr 8   );
               arg2 := ShortInt( temp and $FF );
             end;

           if flags and ARGS_ARE_XY_VALUES <> 0 then
             begin
               x_offset := arg1;
               y_offset := arg2;
             end
           else
             begin

               if (arg1 < 0) or (arg1 > cur_point) or
                  (arg2 < 0) or (arg2 > cur_point) then
                 begin
                   TT_Forget_Frame;
                   exit;
                 end;

               x_offset := glyph^.points^[arg2].V.x -
                           glyph^.points^[arg1].V.x;

               y_offset := glyph^.points^[arg2].V.y -
                           glyph^.points^[arg1].V.y;
             end;

           if flags and WE_HAVE_A_SCALE <> 0 then
             begin
               READ_F2Dot14( x_scale_2d14 );

               x_scale := x_scale_2d14 shl (16-14);
               scale01 := 0;
               scale10 := 0;
               y_scale := x_scale;
             end
           else
           if flags and WE_HAVE_AN_XY_SCALE <> 0 then
             begin
               READ_F2Dot14( x_scale_2d14 );
               READ_F2dot14( y_scale_2d14 );
               x_scale := x_scale_2d14 shl (16-14);
               scale01 := 0;
               scale10 := 0;
               y_scale := y_scale_2d14 shl (16-14);
             end
           else
           if flags and WE_HAVE_A_2X2 <> 0 then
             begin
               READ_F2dot14( x_scale_2d14 );
               READ_F2dot14( scale01_2d14 );
               READ_F2dot14( scale10_2d14 );
               READ_F2dot14( y_scale_2d14 );

               x_scale := x_scale_2d14 shl (16-14);
               scale01 := scale01_2d14 shl (16-14);
               scale10 := scale10_2d14 shl (16-14);
               y_scale := y_scale_2d14 shl (16-14);
             end;

           TT_Forget_Frame;

          (* now we apply the current scaling values to all the recursive *)
          (* subglyphs called from this depth level.                      *)

           for i := old_subglyph to cur_subglyph-1 do
           begin

             cur_gs := @glyph^.scales^[i];

             with cur_gs^ do
             begin

               flags := flags or WE_HAVE_A_2X2;

               n1 := Load_Glyph.x_offset +
                     MulDiv_Round( x_offset, x_scale, $10000 ) +
                     MulDiv_Round( y_offset, scale01, $10000 );


               n2 := Load_Glyph.y_offset +
                     MulDiv_Round( x_offset, scale10, $10000 ) +
                     MulDiv_Round( y_offset, y_scale, $10000 );

               x_offset := n1;
               y_offset := n2;

             n1 := MulDiv_Round( x_scale, x_scale, $10000 ) +
                   MulDiv_Round( scale01, scale10, $10000 );

             n2 := MulDiv_Round( x_scale, scale01, $10000 ) +
                   MulDiv_Round( scale01, y_scale, $10000 );

             n3 := MulDiv_Round( scale10, x_scale, $10000 ) +
                   MulDiv_Round( y_scale, scale10, $10000 );

             n4 := MulDiv_Round( scale10, scale01, $10000 ) +
                   MulDiv_Round( y_scale, y_scale, $10000 );

             with cur_gs^ do
             begin
               x_scale := n1;
               scale01 := n2;
               scale10 := n3;
               y_scale := n4;
             end;

         until flags and MORE_COMPONENTS = 0;

         end;

       end;

 end;


 function Load_TrueType_Glyph( instance   : PInstanceRecord;
                               glyphIndex : Int ) : Boolean;
 var
   cur_contour  : Int;
   cur_point    : Int;
   cur_subglyph : Int;

   res : boolean;
 begin

   Load_TrueType_Glyph := False;

   (* verify sanity of user call *)
   if instance = nil then
   begin
     Error := TT_ErrMsg_Invalid_Font_Handle;
     exit;
   end;

   if glyphIndex > instance^.fontres^.numGlyphs then
   begin
     Error := TT_ErrMsg_Invalid_Glyph_Index;
     exit;
   end;

   (* we must lock the stream here *)

   if not TT_Lock_Access( instance^.fontres^.stream ) then exit;

   cur_contour  := 0;
   cur_point    := 0;
   cur_subglyph := 0;

   res := Load_Glyph( instance,
                      glyphIndex,
                      cur_contour,
                      cur_point,
                      cur_subglyph );

   TT_Release_Access( instance^.fontres^.stream );

   Load_TrueType_Glyph := res;
 end;
}

(*******************************************************************
 *
 *  Function    :  Load_TrueType_CMap
 *
 *  Description :
 *
 *  Input  :  aResident
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_CMap( aResident : PResidentLeading ) : boolean;
 var
   n, i, t, num_SH : Int;

   u : UShort;

   cmap_dir : TCMapDir;
   entry    : TCMapDirEntry;
   off, l   : Longint;

   num_Seg  : Int;

   segments : PCMap4Segments;

   glArray  : PUShort;

   table_start : Longint;
 begin

   Load_TrueType_CMap := False;

   {$IFDEF DEBUG}
   Write('CMaps ');
   {$ENDIF}

   t := LookUp_trueType_table( aResident,'cmap' );
   if t < 0 then exit;

   with aResident^ do
   begin

     table_start := dirTables^[t].offset;

     if not TT_Seek_File( dirTables^[t].Offset ) or

        not TT_Access_Frame( sizeof(TCMapDir) ) then exit;

     cmap_dir.tableVersionNumber := GET_UShort;
     cmap_dir.numCMaps           := GET_UShort;

     TT_Forget_Frame;

     numCMaps := cmap_dir.numCMaps;

     if not Alloc( numCMaps * sizeof( TCMapTable ),
                   Pointer( cMaps ) ) then exit;

     off := TT_File_Pos;

     for n := 0 to numCMaps-1 do
     begin

       if not TT_Seek_File   ( off ) or
          not TT_Access_Frame( 8 ) then exit;

       entry.platformID         := GET_UShort;
       entry.platformEncodingID := GET_UShort;
       entry.offset             := GET_Long;

       TT_Forget_Frame;

       off := TT_File_Pos;

       if not TT_Seek_File   ( table_start + entry.offset ) or
          not TT_Access_Frame( 3*2 ) then exit;

       with CMaps^[n] do
       begin
         Format  := GET_UShort;
         Length  := GET_UShort;
         Version := GET_UShort;

         TT_Forget_Frame;

         case Format of

           0 : begin
                 if not Alloc( sizeof(TCMap0Table), Pointer( CMap0 ) )
                   then exit;

                 if not Alloc( 256, Pointer(CMap0^.glyphIdArray) ) or
                    not TT_Read_File( CMap0^.glyphIdArray^, 256 )
                   then exit;
               end;

           2 : begin
                 if not Alloc( sizeof(TCMap2Table), Pointer( CMap2 ) )
                   then exit;

                 num_SH := 0; (* in multiples of 8 =sizeof(TCMap2SubHeader) *)

                 if not TT_Access_Frame( 256*2 ) then exit;

                 for i := 0 to 255 do
                 begin
                   u                              := GET_UShort div 8;
                   CMap2^.Header.subHeaderKeys[i] := u;

                   if num_SH < u then num_SH := u;
                 end;

                 TT_Forget_Frame;

                 (* in multiples of 2 = sizeof(UShort) *)

                 l := Length - (256 + 3)*sizeof(UShort) - num_SH*8;

                 if not Alloc( num_SH, Pointer(CMap2^.SubHeaders) ) or
                    not TT_Access_Frame( num_SH*8 ) then exit;

                 for i := 0 to num_SH-1 do
                   with CMap2^.subHeaders^[i] do
                   begin
                     firstCode     := GET_UShort;
                     entryCount    := GET_UShort;
                     idDelta       := GET_Short;
                     idRangeOffset := GET_UShort - ( num_SH - i )*8 + 2;
                   end;

                 TT_Forget_Frame;

                 if not Alloc( l, Pointer(CMap2^.glyphIdArray) ) or
                    not TT_Access_Frame( l ) then exit;

                 glArray := CMap2^.glyphIdArray;

                 for i := 0 to l div 2-1 do
                   glArray^[i] := GET_UShort;

                 TT_Forget_Frame;

               end;

           4 : begin
                 if not Alloc(  sizeof(TCMap4Table), Pointer( CMap4 ) ) or

                    not TT_Access_Frame( 4*sizeof(UShort) )

                   then exit;

                 with CMap4^.Header do
                 begin
                   segCountX2    := GET_UShort;
                   searchRange   := GET_UShort;
                   entrySelector := GET_UShort;
                   rangeShift    := GET_UShort;

                   num_Seg := segCountX2 div 2;
                 end;

                 TT_Forget_Frame;

                 if not Alloc( num_Seg * sizeof( TCMap4Segment ),
                               Pointer( CMap4^.Segments ) ) or

                    not TT_Access_Frame( (num_SEG*4 + 1)*sizeof(UShort) )

                   then exit;

                 segments := CMap4^.Segments;

                 for i := 0 to num_Seg-1 do
                   segments^[i].endCount := GET_UShort;

                 GET_UShort;

                 for i := 0 to num_Seg-1 do
                   segments^[i].startCount := GET_UShort;

                 for i := 0 to num_Seg-1 do
                   segments^[i].idDelta := GET_UShort;

                 for i := 0 to num_Seg-1 do
                   segments^[i].idRangeOffset := GET_UShort;

                 TT_Forget_Frame;

                 l := cMaps^[n].length - ( 16 + 8*num_Seg );

                 if not Alloc( l, Pointer( CMap4^.glyphIdArray ) ) or

                    not TT_Access_Frame( l )

                   then exit;

                 glArray := CMap4^.glyphIdArray;

                 for i := 0 to l div 2-1 do
                   glArray^[i] := GET_UShort;

                 TT_Forget_Frame;

               end;

           6 : begin

                 if not Alloc( sizeof( TCMap6Table ), Pointer(CMap6) ) or

                    not TT_Access_Frame( 2 * sizeof(UShort) )

                 then exit;

                 CMap6^.Header.firstCode  := GET_UShort;
                 CMap6^.Header.entryCount := GET_UShort;

                 TT_Forget_Frame;

                 l := CMap6^.Header.entryCount;

                 if not Alloc( l*sizeof(UShort),
                               Pointer(CMap6^.glyphIdArray) ) or

                    not TT_Access_Frame( l*sizeof(UShort) )

                    then exit;

                 glArray := CMap6^.glyphIdArray;

                 for i := 0 to l-1 do
                   glArray^[i] := GET_UShort;

                 TT_Forget_Frame;

               end;

         else
           exit;  { Corrupt Character Mappings Table }

         end;

       end;

     end;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_CMap := True;

 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_HMTX
 *
 *  Description :
 *
 *  Input  :  aResident
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_HMTX( aResident : PResidentLeading ) : boolean;
 var
   t, n         : int;
   nsmetrics,
   nhmetrics : integer;

 begin
   Load_trueType_HMTX := false;

   {$IFDEF DEBUG}
   Write('HMTX ');
   {$ENDIF}

   t := LookUp_TrueType_Table(aResident,'hmtx');
   if t < 0 then exit;

   with aResident^ do
   begin

     nhmetrics := horizontalHeader.number_Of_HMetrics;
     nsmetrics := MaxProfile.numGlyphs - nhmetrics;

     if not Alloc( sizeof(TLongHorMetric)*nhmetrics,
                   Pointer( LongHMetrics ) ) or

        not Alloc( sizeof(Short)*nsmetrics,
                   Pointer( ShortMetrics ) ) or

        not TT_Seek_File( dirTables^[t].Offset ) or

        not TT_Access_Frame( dirTables^[t].Length ) then exit;

     for n := 0 to nhmetrics-1 do with LongHMetrics^[n] do
     begin
       advance_width := GET_Short;
       lsb           := GET_Short;
     end;

     for n := 0 to nsmetrics-1 do
       ShortMetrics^[n] := GET_Short;

     TT_Forget_Frame;

   end;

   {$IFDEF DEBUG}
   Writeln('loaded');
   {$ENDIF}

   Load_TrueType_HMTX := True;

 end;

(*******************************************************************
 *
 *  Function    :  Load_TrueType_Programs
 *
 *  Description :  Load the Font and CVT programs in the resident
 *                 table
 *
 *  Input  :  aResident
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function Load_TrueType_Programs( aResident : PResidentLeading ) : Boolean;
 var
   t : Int;
 begin

   Load_TrueType_Programs := False;

   {$IFDEF DEBUG}
   Write('Font program ');
   {$ENDIF}

   (* The font program is optional *)

   t := Lookup_TrueType_Table( aResident, 'fpgm' );

   if t < 0 then

     with aResident^ do
     begin
       fontProgram := nil;
       fontPgmSize := 0;

       {$IFDEF DEBUG}
       Writeln('none in file');
       {$ENDIF}
     end

   else

     with aResident^ do
     begin

       fontPgmSize := dirTables^[t].Length;

       if not Alloc( fontPgmSize, Pointer(fontProgram) ) or

          not TT_Read_At_File( dirTables^[t].offset,
                               fontProgram^,
                               fontPgmSize ) then exit;
       {$IFDEF DEBUG}
       Writeln('loaded, ',fontPgmSize,' bytes');
       {$ENDIF}

     end;

   {$IFDEF DEBUG}
   Write('CVT program ');
   {$ENDIF}

   t := LookUp_trueType_Table( aResident, 'prep' );

   (* The CVT table is optional *)

   if t < 0 then

     with aResident^ do
     begin
       cvtProgram := nil;
       cvtPgmSize := 0;

       {$IFDEF DEBUG}
       Writeln('none in file');
       {$ENDIF}
     end

   else

     with aResident^ do
     begin

       cvtPgmSize := dirTables^[t].Length;

       if not Alloc( cvtPgmSize, Pointer(cvtProgram) ) or

          not TT_Read_At_File( dirTables^[t].offset,
                               cvtProgram^,
                               cvtPgmSize ) then exit;
       {$IFDEF DEBUG}
       Writeln('loaded, ',cvtPgmSize,' bytes');
       {$ENDIF}

     end;

   Load_TrueType_Programs := True;
 end;

(*******************************************************************
 *
 *  Function    :  TT_Load_Resident_Table
 *
 *  Description :  Loads one font file and build its resident table
 *
 *  Input  :  stream     file handle of the opened file.
 *
 *  Output :
 *
 *  Notes  :
 *
 *****************************************************************)

 function TT_Load_Resident_Table( astream       : Int;
                                  var aResident : PResidentLeading ) : Boolean;
 var
   Debut    : TMarkRecord;
   stream   : TT_Stream;
   TableDir : TTableDir;

   size : longint;

   i, j, k, l, n : integer;

 label
   Error_1, Error_2;

 begin

   Mark( Debut );

   size := Mem_Free;

   if not TT_Lock_Access( stream ) then
     goto Error_1;

   if not Alloc( sizeof(TResidentLeading), Pointer(aResident) ) then
     goto Error_1;

   if not Load_TrueType_Directory( aResident ) or

      not Load_TrueType_Header( aResident ) or

      not Load_TrueType_MaxProfile( aResident ) or

      not Load_TrueType_Locations( aResident ) or

      not Load_TrueType_CMap( aResident ) or

      not Load_TrueType_CVT( aResident ) or

      not Load_TrueType_Horizontal_Header( aResident ) or

      not Load_TrueType_Programs( aResident ) or

      not Load_TrueType_HMTX( aResident ) then goto Error_2;

   TT_Release_Access( stream );

   aResident^.stream := stream;

   size := size - Mem_Free;

   aResident^.totalSize := size;

   TT_Load_Resident_Table := True;

   exit;

 Error_2:

   TT_Release_Access( stream );

 Error_1:

   Release( Debut );

   aResident := nil;

   TT_Load_Resident_Table := False;

 end;


(*******************************************************************
 *
 *  Function    :  TT_Set_Glyph_Defaults
 *
 *  Description :  The values computed by the CVT program will be
 *                 set as the new glyph defaults.
 *
 *  Input  :  aInstance
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  :  Must be used after executing the CVT program
 *
 ******************************************************************)

 procedure TT_Set_Glyph_Defaults( aInstance : PInstanceRecord );
 var
   i : integer;
 begin

   with aInstance^ do
   begin

     (* BULLSHIT : The specs stipulates that all values that are   *)
     (*            set within the CVT program should be kept as    *)
     (*            new glyph defaults. Unfortunately, all glyph    *)
     (*            streams expect zp0..2 set to zone 1, while the  *)
     (*            CVT program sets them to zone 0 !!              *)
     (*            Moreover, the vectors must be set to the x-axis *)

     GS.gep0 := 1;
     GS.gep1 := 1;
     GS.gep2 := 1;

     GS.projVector.x := $4000;
     GS.projVector.y := $0000;

     GS.dualVector.x := $4000;
     GS.dualVector.y := $0000;

     GS.freeVector.x := $4000;
     GS.freeVector.y := $0000;

     GS.round_state := 1;   (* even rounding !! *)

     zp0 := pts;
     zp1 := pts;
     zp2 := pts;

     Default_GS := GS;  (* Save the graphics state *)

     (* that's all for now *)

  end;

 end;

(*******************************************************************
 *
 *  Function    :  TT_Reset_Glyph_Defaults
 *
 *  Description :  Resets glyph defaults generated by the
 *                 CVT program.
 *
 *  Input  :  aInstance
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  :  Must be used before executing a glyph's instructions
 *
 ******************************************************************)

 function TT_Glyph_Defaults( aInstance : PInstanceRecord ) : Boolean;
 var
   i : integer;
 begin

   with aInstance^ do
   begin

     GS := Default_GS;  (* restore the graphics state *)

     zp0 := pts;
     zp1 := pts;
     zp2 := pts;

   end;

   (* BULLSHIT : Read the note above in TT_Set_Glyph_Defaults  *)

 end;

(*******************************************************************
 *
 *  Function    :  TT_Reset_Instance
 *
 *  Description :  Resets instance data to load defaults
 *
 *  Input  :  aResident
 *            aInstance
 *            PtSize       Point size
 *            Resolution   Resolution in DPI
 *
 *     Pixels = ( FUnits * Scale1 ) / Scale2
 *
 *     Scale1 = PointSize * Resolution
 *     Scale2 = 72 * UnitsPerEM
 *
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  : Must be called before executing the Font Program,
 *           and the CVT program ( just after the scale change )
 *
 ******************************************************************)

 function TT_Reset_Instance( aInstance  : PInstanceRecord;
                             ptSize     : Int;
                             resolution : Int ) : Boolean;
 var
   i : integer;
 begin

  {$IFDEF DEBUG}
  Num_Ins_Glyph     := 0;
  Num_Ins_Composite := 0;

  Sum_Ins_Glyph     := 0;
  Sum_Ins_Composite := 0;

  Max_Ins_Glyph     := 0;
  Max_Ins_Composite := 0;
  {$ENDIF}

   with aInstance^ do
   begin

     codeSize := 0;
     IP       := 0;
     curRange := -1;
     callTop  := 0;
     top      := 0;

     numFDefs := fontRes^.MaxProfile.maxFunctionDefs;
     numIDefs := fontRes^.MaxProfile.maxInstructionDefs;

     Default_GS := Default_GraphicsState;
     GS         := Default_GraphicsState;

     with fontRes^ do
     begin

       Set_CodeRange( aInstance,
                      TT_CodeRange_Font,
                      fontProgram,
                      fontPgmSize );

       Set_CodeRange( aInstance,
                      TT_CodeRange_CVT,
                      cvtProgram,
                      cvtPgmSize );
     end;

     Clear_CodeRange( aInstance, TT_CodeRange_Glyph );

     for i := 0 to storeSize-1 do storage^[i] := 0;

     period    := 0;
     phase     := 0;
     threshold := 0;

     (* Important note :                                     *)
     (*                                                      *)
     (*   Respecting the order of these computations is      *)
     (*   important. The ppem must be computed first, and    *)
     (*   truncated to an integer. It will then be used to   *)
     (*   scale the rest of the glyphs.                      *)
     (*                                                      *)
     (*   I started with                                     *)
     (*                                                      *)
     (*     scale1  := ptSize * 64 * resolution              *)
     (*     scale2  := 72 * fontHeader.Units_Per_EM          *)
     (*                                                      *)
     (*   which used to generate glyphs that were too high   *)
     (*   by one pixel.                                      *)
     (*                                                      *)

     pointSize := ptSize * 64;
     ppem      := ((ptSize * resolution) + 36 ) div 72;
     scale1    := ppem * 64;
     scale2    := Long(fontres^.fontHeader.Units_Per_EM);

     instruction_trap := false;

     for i := 0 to cvtSize-1 do
       cvt^[i] := MulDiv_Round( fontres^.cvt^[i], scale1, scale2 );

     (* corrective : These values *must* be rounded for correct display *)

     (* All twilight points are originally zero *)

     for i := 0 to twilight.n-1 do
     begin
       twilight.org_x^[i] := 0;
       twilight.org_y^[i] := 0;
       twilight.cur_x^[i] := 0;
       twilight.cur_y^[i] := 0;
     end;

     TT_reset_Instance := True;

   end
 end;

(*******************************************************************
 *
 *  Function    :  TT_Load_Instance_Data
 *
 *  Description :
 *
 *  Input  :  aResident
 *            aInstance
 *
 *  Output :  True on success. False on failure
 *
 ******************************************************************)

 function TT_Load_Instance_Data( aResident      : PResidentLeading;
                                 var aInstance  : PInstanceRecord
                               ) : Boolean;
 var
   Size, i, l : Longint;

 begin
   TT_Load_Instance_Data := False;

   Size := Mem_Free;

   if not Alloc( sizeof( TInstanceRecord ), Pointer(aInstance) )
     then exit;

   with aResident^, aInstance^ do
   begin

     fontRes := aResident;

     { Reserve glyph code range }

     if not Alloc( MaxProfile.maxSizeOfInstructions, Pointer(glyphIns) )
       then exit;

     { Reserve function and instruction definitions }

     if not Alloc( MaxProfile.maxFunctionDefs * sizeof(TDefRecord),
                   Pointer( FDefs ) ) or
        not ALloc( MaxProfile.maxInstructionDefs * sizeof(TDefRecord),
                   Pointer( IDefs ) )
        then exit;

     numFDefs := fontRes^.MaxProfile.maxFunctionDefs;
     numIDefs := fontRes^.MaxProfile.maxInstructionDefs;

     { Reserve call stack }

     if not Alloc( 32 * sizeof(TCallRecord), Pointer( callStack ) )
       then exit;

     callSize := 32;

     { Reserve storage area }

     if not Alloc( MaxProfile.maxStorage * sizeof(Long),
                   Pointer(storage) )
       then exit;

     storeSize := MaxProfile.maxStorage;
     stackSize := MaxProfile.maxStackElements;

     l := stackSize * sizeof( Long );

     { Reserve stack }

     if not Alloc( l, Pointer( stack ) )
       then exit;

     { Reserve point arrays }

     l := MaxProfile.maxCompositePoints;

     if (l < MaxProfile.maxPoints) then
       l := MaxProfile.maxPoints;

     pts.n := l;

     { We should not forget the phantom points }

     l     := (l+2) * sizeof( TT_F26dot6 );

     if not Alloc( l, Pointer( pts.org_x ) ) or
        not Alloc( l, Pointer( pts.org_y ) ) or
        not Alloc( l, Pointer( pts.cur_x ) ) or
        not Alloc( l, Pointer( pts.cur_y ) ) or

        not Alloc( pts.n+2 , Pointer( pts.touch ) )

       then exit;

     twilight.n := MaxProfile.maxTwilightPoints;
     l          := twilight.n * sizeof( TT_F26dot6 );

     if not Alloc( l, Pointer( twilight.org_x ) ) or
        not Alloc( l, Pointer( twilight.org_y ) ) or
        not Alloc( l, Pointer( twilight.cur_x ) ) or
        not Alloc( l, Pointer( twilight.cur_y ) ) or

        not Alloc( twilight.n, Pointer( twilight.touch ) )

       then exit;

     { Reserve contour arrays }

     l := MaxProfile.maxContours;
     if l < MaxProfile.maxCompositeContours then
       l := MaxProfile.maxCompositeContours;

     l :=  l * sizeof( UShort );

     if not Alloc( l, Pointer( endContours ) )
       then exit;

     { Reserve Control Value Table }

     cvtSize := aResident^.cvtSize;

     if not Alloc( cvtSize * sizeof(Short), Pointer(cvt) )
       then exit;

   end;

   Size := Size - Mem_Free;

   aInstance^.totalSize := Size;

   TT_Load_Instance_Data := True;
 end;

(*******************************************************************
 *
 *  Function    :  Get_CodeRange
 *
 *  Description :  Return a pointer to a given coderange. Should
 *                 be used only by the debugger. Returns NIL if
 *                 'ARange' is out of current bounds.
 *
 *  Input  :  aINstance    target font instance
 *            aRange       code range index
 *
 *  Output :  Pointer to the code range record
 *
 *****************************************************************)

 function Get_CodeRange( aInstance : PInstanceRecord;
                         aRange    : Int             ) : PCodeRange;
 begin
   with aInstance^ do
     if (ARange < 1) or (ARange > 3) then
       Get_CodeRange := nil
     else
       Get_CodeRange := @CodeRangeTable[ARange];
 end;

(*******************************************************************
 *
 *  Function    :  Set_CodeRange
 *
 *  Description :  Sets a code range.
 *
 *  Input  :  aInstance       target font instance
 *            aRange          code range index
 *            aBase           new code base
 *            aLength         range size in bytes
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  : Does not set the Error variable
 *
 *****************************************************************)

 function Set_CodeRange( aInstance : PInstanceRecord;
                         aRange    : Int;
                         aBase     : Pointer;
                         aLength   : Int       ) : boolean;
 begin
   Set_CodeRange := false;

   with aInstance^ do
   begin

     if (ARange < 1) or (ARange > 3) then exit;

     with CodeRangeTable[aRange] do
     begin
       base := aBase;
       size := aLength;
     end;

   end;

   Set_CodeRange := true;
 end;

(*******************************************************************
 *
 *  Function    :  Clear_CodeRange
 *
 *  Description :  clears a code range.
 *
 *  Input  :  aInstance       target font instance
 *            aRange          code range index
 *
 *  Output :  True on success. False on failure
 *
 *  Notes  : Does not set the Error variable
 *
 *****************************************************************)

 function Clear_CodeRange( aInstance : PInstanceRecord;
                           aRange    : Int              ) : boolean;
 begin
   Clear_CodeRange := false;

   with aInstance^ do
   begin

     if (ARange < 1) or (ARange > 3) then exit;

     with CodeRangeTable[aRange] do
     begin
       base := nil;
       size := 0;
     end;

   end;

   Clear_CodeRange := true;
 end;

(*******************************************************************
 *
 *  Function    :  Goto_CodeRange
 *
 *  Description :  Switch to a new code range ( updates Code and IP )
 *
 *  Input  :  ARange : Int  New execution code range
 *            AIP    : Int  New IP in new code range.
 *
 *  Output :  True on success. False on failure ( no coderange )
 *
 *****************************************************************)

 function Goto_CodeRange( aInstance : PInstanceRecord;
                          aRange,
                          aIP       : Int ): boolean;
 begin

   Goto_CodeRange := False;

   with aInstance^ do
   begin
     if (aRange<1) or (aRange>3) then
       begin
         Error := TT_ErrMsg_Bad_Argument;
         exit;
       end;

     with CodeRangeTable[ARange] do
       begin

         if Base = nil then  (* invalid coderange *)
         begin
           Error := TT_ErrMsg_Invalid_Coderange;
           exit;
         end;

         (* NOTE : Because the last instruction of a program may be a CALL *)
         (*        which will return to the first byte *after* the code    *)
         (*        range, we test for AIP <= Size, instead of AIP < Size   *)

         if AIP > Size then
           begin
             Error          := TT_ErrMsg_Code_Overflow;
             Goto_CodeRange := False;
             exit;
           end;

         Code     := PByte(Base);
         CodeSize := Size;
         IP       := AIP;
       end;

     curRange := ARange;
   end;

   Goto_CodeRange := True;
 end;



end.

