{****************************************************************************}
{*                                                                          *}
{*  ZoneTV.PAS                                                              *}
{*                                                                          *}
{*  This unit implements a simple TrueType zone points viewer for the       *}
{*  FREETYPE project debugger.                                              *}
{*                                                                          *}
{****************************************************************************}

Unit ZoneTV;

interface

uses Objects, Views, Drivers, FreeType, TTTypes, TTTables, TTDebug;

{$I DEBUGGER.INC}

type

  { TZoneViewer }

  { This TView is a simple point array viewer }

  PZoneViewer = ^TZoneViewer;
  TZoneViewer = object( TListViewer )

                  constructor Init( var Bounds : TRect;
                                    AZone      : TT_PVecRecord );

                  procedure Draw; virtual;
                  procedure HandleEvent( var Event : TEvent  ); virtual;

                  destructor Done; virtual;

                private
                  Zone : TT_PVecRecord; { Pointer to the zone being displayed }
                  Save : TT_VecRecord;  { A copy of the zone to highlight     }
                                        { changes                             }
                  procedure Copy_Zone;

                end;

  { TCodeWindow }

  PZoneWindow = ^TZoneWindow;
  TZoneWindow = object( TWindow )
                  ZoneView : PZoneViewer;
                  constructor Init( var Bounds : TRect;
                                    AZone      : TT_PVecRecord );
                end;

implementation

{ TZoneViewer }

constructor TZoneViewer.Init;
var
  n : Int;
begin
  inherited Init( Bounds, 1, nil, nil );

  GrowMode  := gfGrowHiX or gfGrowHiY;
  DragMode  := dmDragGrow or dmLimitLoX or dmLimitLoY;
  Options   := Options or ofSelectable;
  EventMask := EventMask or evWave;

  Zone := AZone;

  Save.n := Zone^.n;
  n := Save.n * sizeof(Long);
  GetMem( Save.org_x, n );
  GetMem( Save.org_y, n );
  GetMem( Save.cur_x, n );
  GetMem( Save.cur_y, n );
  GetMem( Save.touch, Save.n );

  Copy_Zone;

  SetRange( Save.n );
end;


destructor TZoneViewer.Done;
var
  n : Int;
begin
  n := Save.n * Sizeof(Long);

  freemem( Save.touch, Save.n );
  freemem( Save.cur_y, n );
  freemem( Save.cur_x, n );
  freemem( Save.org_y, n );
  freemem( Save.org_x, n );

  inherited Done;

end;


procedure TZoneViewer.Copy_Zone;
var
  n : Int;
begin
  n := Save.n * sizeof(Long);

  (* Note that we save also the original coordinates, as we're not sure *)
  (* that the debugger is debugged !                                    *)

  move( Zone^.org_x^, Save.org_x^, n );
  move( Zone^.org_y^, Save.org_y^, n );
  move( Zone^.cur_x^, Save.cur_x^, n );
  move( Zone^.cur_y^, Save.cur_y^, n );
  move( Zone^.touch^, Save.touch^, Save.n );
end;


procedure TZoneViewer.HandleEvent( var Event : TEvent );
var
  Limits     : TRect;
  Mini, Maxi : Objects.TPoint;
begin

  inherited HandleEvent(Event);

  Case Event.What of

    evWave : case Event.Command of

               cmNewExecution : Copy_Zone;

               cmRefocus : DrawView;

              end;

    evCommand : case Event.Command of

                 cmResize: begin
                             Owner^.GetExtent(Limits);
                             SizeLimits( Mini, Maxi );
                             DragView(Event, DragMode, Limits, Mini, Maxi );
                             ClearEvent(Event);
                           end;
                end;
  end;
end;


procedure TZoneViewer.Draw;
const
  Colors : array[0..3] of byte
         = ($30,$3F,$0B,$0E);
  Touchs : array[0..3] of Char
         = (' ','x','y','b');
  OnOff  : array[0..1] of Char
         = (' ',':');
var
  I, J, Item : Int;
  B          : TDrawBuffer;
  S          : String;
  Indent     : Int;
  Ligne      : Int;

  Changed : Boolean;

  Back_Color,
  Color       : word;

  On_BP : boolean;
  BP    : PBreakPoint;

begin

  if HScrollBar <> nil then Indent := HScrollBar^.Value
                       else Indent := 0;

  with Save do
  begin

    for I := 0 to Self.Size.Y-1 do
    begin

      MoveChar( B, ' ', Colors[0], Self.Size.X );

      Item := TopItem + I;

      if (Range > 0) and
       ( Focused = Item ) then Back_Color := 2
                          else Back_Color := 0;

      if Item < n then
        begin

          Color := Back_Color;
          if ( touch^[item] <> Zone^.touch^[item] ) then inc( Color );

          S    := Hex16( Item ) + ':  ';
          S[1] := OnOff[Zone^.touch^[item] and 1];
          S[7] := Touchs[(Zone^.touch^[item] and TT_Flag_Touched_Both) shr 1];

          MoveStr( B, S, Colors[Color] );

          Color := Back_Color;
          if ( org_x^[item] <> Zone^.org_x^[item] ) then inc( Color );

          MoveStr ( B[8], Hex32( Zone^.org_x^[item] ), Colors[Color] );
          MoveChar( B[16], ',', Colors[0], 1 );

          Color := Back_Color;
          if ( org_y^[item] <> Zone^.org_y^[item] ) then inc( Color );

          MoveStr( B[17], Hex32( Zone^.org_y^[item] ), Colors[Color] );
          MoveStr( B[25], ' : ', Colors[0] );

          Color := Back_Color;
          if ( cur_x^[item] <> Zone^.cur_x^[item] ) then inc( Color );

          MoveStr ( B[28], Hex32( Zone^.cur_x^[item] ), Colors[Color] );
          MoveChar( B[36], ',', Colors[0], 1 );

          Color := Back_Color;
          if ( cur_y^[item] <> Zone^.cur_y^[item] ) then inc( Color );

          MoveStr( B[37], Hex32( Zone^.cur_y^[item] ), Colors[Color] );

        end;

      WriteLine( 0, I, Self.Size.X, 1, B );
    end;
  end;
end;

{ TZoneWindow }

constructor TZoneWindow.Init;
begin
  inherited Init( Bounds,'Zone',wnNoNumber );
  GetExtent( Bounds );
  Bounds.Grow(-1,-1);
  New( ZoneView, Init( Bounds, AZone ) );
  Insert( ZoneView );
end;

end.

