/***********************************************************************
 Copyright (C) 2002, Vadim Yegorov
 All rights reserved

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

  o Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

  o Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.

  o Neither name of the Vadim Yegorov nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 -----------------------------------------------------------------------
 Module: VYX' Simple SPL Kernel
 Sorry for big pfx, but big asm (aka C language) havn't namespaces :(
 -----------------------------------------------------------------------

 -----------------------------------------------------------------------
 $Id: vyxssplk.h,v 1.10 2002/05/07 18:38:38 vad Exp $
 ***********************************************************************/

#if !defined(__VYX_SSPL_K_H__)
    #define __VYX_SSPL_K_H__

    #if !defined(__IBMCPP__) && !defined(__IBMC__)
        #error This version was optimized for IBM C/C++ compiler (VAC++/xlc) only!
    #endif

    #ifdef __cplusplus
        extern "C" {
        #undef  VAC_EXP
        #define VAC_EXP _Export
    #else
        #undef  VAC_EXP
        #define VAC_EXP
    #endif

    #define API_ENTRY   VAC_EXP


    /** PI */
    #ifndef M_PI
        #define M_PI        3.14159265358979323846
    #endif

    #ifndef M_PI_2
        #define M_PI_2      1.57079632679489661923
    #endif

    #ifndef M_2PI
        #define M_2PI       6.28318530717958
    #endif


    /** FIR types */
    enum FIR_TYPES {
        FIRT_NOT_INITED = -1,
        FIRT_LOW_PASS = 1, FIRT_HIGH_PASS, FIRT_BAND_PASS, FIRT_BAND_STOP,
        FIRT_MIN_AMP, FIRT_MED_AMP, FIRT_MAX_AMP, /** mediana filters */
        FIRT_REJECTOR = FIRT_BAND_STOP,
        FIRT_DIFFER, FIRT_HILBERT_TRANS
    };

    enum FIR_WINDOW_TYPE {
        FIRWT_LANCOZ, FIRWT_WEBER
    };

    /** Errors   */
    enum FIR_ERRORS {
        VYXSSPL_SUCCESS = 0,
        VYXSSPL_NULL_DATA,
        VYXSSPL_WRONG_TYPE,
        VYXSSPL_WRONG_FS,
        VYXSSPL_WRONG_FREQS,
        VYXSSPL_WRONG_WIN_TYPE,
        VYXSSPL_WRONG_WIN_PARAM,
        VYXSSPL_WRONG_POWER,
        VYXSSPL_WRONG_WIN_SIZE,
        VYXSSPL_CANT_CREATE_FIR,
        VYXSSPL_WRONG_PARAM
    };

    /** FIR data structure
     *  If FIR is inited then type != 0
     */
    typedef struct _FIR_DATA {
        int     type;   /** FIR type     */
        int     power;  /** FIR power    */
        int     pos;    /** int position */
        double* H;      /** FIR coefs    */
        double* w;      /** FIR window   */
        double  f;      /** fsample, Hz  */
        double  f1, f2; /** 1st & 2nd freqs */
        /*--- internal data, Don't touch this! (C) */
        unsigned long  internal;
        size_t   ipos,opos;
    } FIR_Data;

    /** Simple EQU cell */
    typedef struct _SEQU_DATA {
        double fs;    /** sample rate, Hz */
        double maxK;  /** max K (in bamd), dB         */
        double KdB;   /** cur K (dB): [-maxK...+maxK] */
        /*--- internal data, Don't touch this! (C)    */
        FIR_Data up;
        FIR_Data dn;
        double k;
    } EQU_Data;

    /** OSC data structure */
    typedef struct _OSC_DATA {
        double  phase;  /** start phase            */
        double  k;      /** amp k                  */
        double  f;      /** freq, Hz               */
        double  fs;     /** sample rate, Hz        */
        double  zero;   /** zero level [-1..1]     */


        double  W;
        double  startt;
        double  dt;
        double  ph;
        double  b[7];
    } OSC_Data;


    /** FHT data structure
     */
    typedef struct _FHT_DATA {
        double* spec;       /* results [0..winsz/2] */
        double* accspec;    /* accumulated spec*/
        int     specsz;     /* winsz / 2 */
        int     winsz;      /* window size, *MUST BE* power of 2! */
        double  fs;         /* sample rate, Hz      */
        double  fbin;       /* bin freq., Hz        */
        int     isdBOutput; /* dB or `pure' results :) */
        double  v0;         /* after calibration contains value for 0dB */
        double  winT;       /* win time, sec */
        /*--- internal data, Don't touch this! (C) */
        double* kw;
        double  delta;
        int     NS;
        unsigned long p;
    } FHT_Data;

    /** Fade fata structure
     */
    typedef struct _FADE_DATA {
        double   kmax;  /* max ampl.                     */
        double   kmin;  /* min ampl.                     */
        int     inout;  /* 1 -- fadein, 0 -- fadeout     */
        int    linexp;  /* 1 -- linear, 0 -- exponential */
        double      T;  /* effect time, sec              */
        double     fs;  /* sample freq., Hz              */
        double   curt;  /* currnet time, sec.            */
                        /* clear for restart effect      */
        /*--- internal data, Don't touch this! (C) */
        double     dt;
        double    a,b,k;
    } FADE_Data;

    /* for optimized routines */
    typedef int (FIR_Routine)(double* H, double* w, double* out, double* in, int pwr, int N, int pos);
    typedef FIR_Routine* pFIR_Routine;

    /* -------------------------------------------------------------- */
    /** Setup internal FIR routine (i.e. optimized for the CPU).
      * For low/high/bands filter ONLY!
      * @param pFIR_Routine pointer on routine.
      * If equal NULL then internal routine was used.
      */
    void API_ENTRY VYXSSPL_setFIRRoutine(pFIR_Routine);

    /** FIR initialize routine (aka constructor P).
     *  @param pfir    pointer on data struct
     *  @param fs      sample rate, Hz (>= 1Hz);
     *                 for mediana filters param is ignored
     *  @param power   filter power, (>= 2, odd rec.)
     *                 for mediana filters param is ignored
     *  @param type    filter type, see FIR_TYPES enum
     *  @param f1      1st freq, Hz;
     *                 for mediana filters param is ignored
     *  @param f2      2nd freq, Hz. Must be greater then 1st freq.
     *                 for mediana  and low/high pass filters
     *                 param is ignored.
     *  @param wint    FIR window type, see FIR_WINDOWS (Weber req.)
     *  @param winp    FIR window coef. for Weber window, use 1.0
     *  @return        error code (may be P)
     */
    int API_ENTRY   VYXSSPL_initFIR(
        FIR_Data* pfir,
        double      fs,
        int      power,
        int       type,
        double      f1,
        double      f2,
        int       wint,
        double    winp
    );

    /** <b>Very simple</b> A.I. for low pass filter.
      * I think it work :)
      * @param pfir pointer on data struct
      * @param fs   sample rate, Hz
      * @param k    coef., dB/oct
      * @param f    freq, Hz
      */
    int API_ENTRY VYXSSPL_initLPFAuto(
        FIR_Data* pfir,
        double      fs,
        double       k,
        double       f
    );

    /** <b>Very simple</b> A.I. for high pass filter.
      * I think it work :)
      * @param pfir pointer on data struct
      * @param fs   sample rate, Hz
      * @param k    coef., dB/oct
      * @param f    freq, Hz
      */
    int API_ENTRY VYXSSPL_initHPFAuto(
        FIR_Data* pfir,
        double      fs,
        double       k,
        double       f
    );

    /** FIR deinitializer aka destructor ;)
      * @param pfir pointer on data struct
      */
    void API_ENTRY VYXSSPL_doneFIR(
        FIR_Data* pfir
    );

    /** FIR process routine. inwave -> outwave
     *  NOTE 1: No one paran checked up !
     *  @param pfir     pointer on data struct
     *  @param outwave  output vector
     *  @param inwave   input  vector
     *  @param N        vector size
     */
    void API_ENTRY VYXSSPL_doFIR(
        FIR_Data* pfir,
        double* outwave,
        double* inwave,
        int     N
    );


    /** For AFC calculate.
     *  @param  pfir  pointer on data struct
     *  @param  f     freq, Hz
     *  @return amplitude, foor mediana filters always 1.0
     */
    double API_ENTRY  VYXSSPL_calcAFC(FIR_Data* pfir, double f);

    /** For AFC calculate (dB output).
     *  @param  pfir  pointer on data struct
     *  @param  f     freq, Hz
     *  @return amplitude (dB), foor mediana filters always 1.0
     */
    double API_ENTRY  VYXSSPL_calcAFC_dB(FIR_Data* pfir, double f);

    /** Simple equalizer channel initialize routine
     *  @param pequ    pointer on data struct
     *  @param fs      sample rate, Hz (>= 1Hz);
     *  @param power   filter power, (>= 2, odd rec.)
     *  @param f1      1st freq, Hz;
     *  @param f2      2nd freq, Hz. Must be greater then 1st freq.
     *  @param maxK    max K in band, dB
     *  @return        error code
     */
    int API_ENTRY   VYXSSPL_initEQUCh(
        EQU_Data* pequ,
        double fs,
        double power,
        double f1,
        double f2,
        double maxK
    );

    /** Simple EQU deinitializer aka destructor
      * @param pequ pointer on data struct
      */
    void API_ENTRY VYXSSPL_doneEQUCh(
        EQU_Data* pequ
    );


    /** Setup K for EQU channel.
     *  @param pequ     pointer on data struct
     *  @param KdB      K, dB. Range: [-maxK..+maxK]
     */
    void API_ENTRY VYXSSPL_setEQUChK(
        EQU_Data* pequ,
        double    KdB
    );

    /** Simple EQU channel process routine. inwave -> outwave
     *  NOTE 1: No one paran checked up !
     *  @param pequ     pointer on data struct
     *  @param outwave  output vector
     *  @param inwave   input  vector
     *  @param N        vector size
     */
    void API_ENTRY VYXSSPL_doEQUCh(
        EQU_Data* pequ,
        double* outwave,
        double* inwave,
        int     N
    );
    /* ------------- Misc funcs ----------------- */
    /** Return version string (ASCIIZ) for kernel library.
      * @return version string
      */
    const char * API_ENTRY VYXSSPL_VersionStrK(void);

    /** Check if x is power 2
      * @param  x positive number
      * @return true if x is power2, false otherwise
      */
    int API_ENTRY VYXSSPL_isPwr2(unsigned long x);

    /** Calc max ampl. according to bits per sample.
      * @param bits    bits per sample
      * @param isSign  1 mean signum sample
      * @return max amplitude
      */
    double API_ENTRY VYXSSPL_bits2MaxA(size_t bits, int isSign);
    /* ------------- oscilators ----------------- */

    /** Init Osc (aka contructor)
     * @param posc  pointer on data struct
     * @param phase start phase, grad
     * @param k     amp k
     * @param f     osc freq, Hz
     * @param fs    sample rate, Hz
     * @param zero  zero level [-1..1]
     * @return error codes
     */
    int API_ENTRY VYXSSPL_initOsc(
        OSC_Data* posc,
        double  phase,
        double  k,
        double  f,
        double  fs,
        double  zero
    );

    /** Sine oscilator
      * @param posc  pointer on data struct
      * @param a     pointer on output wave
      * @param N     samples count
      */
    void API_ENTRY VYXSSPL_SineOsc(OSC_Data* posc, double* a, int N);

    /** Cosine oscilator
      * @param posc  pointer on initialized data struct
      * @param a     pointer on output wave
      * @param N     samples count
      */
    void API_ENTRY VYXSSPL_CosineOsc(OSC_Data* posc, double* a, int N);


    /** White noice oscilator
      * @param posc  pointer on initialized data struct
      * @param a     pointer on output wave
      * @param N     samples count
      */
    void API_ENTRY VYXSSPL_WhiteNoiseOsc(OSC_Data* posc, double* a, int N);

    /** Pink noice oscilator
      * @param posc  pointer on initialized data struct
      * @param a     pointer on output wave
      * @param N     samples count
      */
    void API_ENTRY VYXSSPL_PinkNoiseOsc(OSC_Data* posc, double* a, int N);


    /** Init OnOffPulser (aka contructor)
     * @param posc  pointer on data struct
     * @param phase start phase, grad [0..360]
     * @param v1    'one'  value
     * @param v0    'zero' value
     * @param f     osc freq, Hz
     * @param fs    sample rate, Hz
     * @param rate  onoff rate (T/pulse time) ]1.0 ... [
     * @return error codes
     */
    int API_ENTRY VYXSSPL_initOnOffPulser(
        OSC_Data* posc,
        double  phase,
        double  v1,
        double  v0,
        double  f,
        double  fs,
        double  rate
    );

    /** OnOff pulser
     * @param posc pointer on initialized data struct
     * @param pointer on output wave
     * @param N     samples count
     */
    void API_ENTRY VYXSSPL_OnOffPulser(OSC_Data* posc, double* a, int N);

    /* ------------- FHT ------------------------- */

    /** FHT data Initializer aka constructor.
      * @param  pfht  pointer on data struct
      * @param  winsz window size. MUST BE power of 2!
      * @param  fs    sample rate, Hz
      * @param  maxA  0. mean don't use dB output,
      *               otherwise max wave amplitude
      * @param  kwdelta internal param (use 8.0)
      * @return error codes
      */
    int API_ENTRY VYXSSPL_initFHT(
        FHT_Data*  pfht,
        int       winsz,
        double       fs,
        double     maxA,
        double  kwdelta
    );

    /** Calc FHT
      * @param pfht pointer on initialized data structure
      * @param inwave pointer on input wave
      * @param N wave size.
      *          If N < pfht -> winsz then tail fill zero vaules.
      *          If N > pfht -> winsz then only winsz samples was used
      */
    void API_ENTRY VYXSSPL_doFHT(
        FHT_Data* pfht,
        double* inwave,
        int N
    );

    /** FHT deinitializer aka destructor ;)
     * @param pfht pointer on initialized data structure
     */
    void API_ENTRY VYXSSPL_doneFHT(FHT_Data* pfht);

    /** Clear accum. spec
     * @param pfht pointer on initialized data structure
     */
    void API_ENTRY VYXSSPL_clearAccFHT(FHT_Data* pfht);

    /* ------------- Detectors ------------------- */

    /** 1 half period detector
     * @param inwave  pointer on input wave
     * @param outwave pointer on output wave
     * @param N       wave size.
     */
    void API_ENTRY VYXSSPL_Detector1HP(
        double* outwave, double* inwave, int N
    );

    /** 2 halfs period detector
     * @param inwave  pointer on input wave
     * @param outwave pointer on output wave
     * @param N       wave size.
     */
    void API_ENTRY VYXSSPL_Detector2HP(
        double* outwave, double* inwave, int N
    );

    /* ------------- AddOns ---------------------- */

    /** Amplifier :)
     * @param k       amp. coef
     * @param inwave  pointer on input wave
     * @param outwave pointer on output wave
     * @param N       wave size.
     */
    void API_ENTRY VYXSSPL_Amp(
        double k,
        double* outwave, double* inwave, int N
    );

    /** Reverse wave
     * @param inwave  pointer on input wave
     * @param outwave pointer on output wave
     * @param N       wave size
     */
    void API_ENTRY VYXSSPL_Reverse(
        double* outwave, double* inwave, int N
    );


    /** Fade data initializer aka constructor.
     * @param kmax    max ampl.
     * @param kmin    min ampl.
     * @param inout   1 -- fadein, 0 -- fadeout
     * @param linexp  1 -- linear, 0 -- exponential
     * @param T       effect time, sec
     * @param fs      sample freq., Hz
     * @return error codes
     */
    int API_ENTRY VYXSSPL_initFade(
        FADE_Data* pfade,
        double kmax,
        double kmin,
        int    inout,
        int    linexp,
        double     T,
        double     fs
    );

    /** Do fade
     * @param pfade    pointer on initialized data structure
     * @param outwave  pointer on output wave
     * @param inwave   pointer on input wave
     * @Param N        wave size
     */
    void API_ENTRY VYXSSPL_doFade(
        FADE_Data* pfade,
        double* outwave,
        double* inwave,
        int     N
    );
    /* ------------- Usefull macros -------------- */
    #define VYXSSPL_INIT_LOWPASS(d,fs,pwr,f) \
        VYXSSPL_initFIR(d,fs,pwr,FIRT_LOW_PASS,f,0.,FIRWT_WEBER, 1.)

    #define VYXSSPL_INIT_HIGHPASS(d,fs,pwr,f) \
        VYXSSPL_initFIR(d,fs,pwr,FIRT_HIGH_PASS,f,0.,FIRWT_WEBER, 1.)

    #define VYXSSPL_INIT_BANDPASS(d,fs,pwr,f1,f2) \
        VYXSSPL_initFIR(d,fs,pwr,FIRT_BAND_PASS,f1,f2,FIRWT_WEBER, 1.)

    #define VYXSSPL_INIT_REJECTOR(d,fs,pwr,f1,f2) \
        VYXSSPL_initFIR(d,fs,pwr,FIRT_REJECTOR,f1,f2,FIRWT_WEBER, 1.)


    #define VYXSSPL_GRAD2RAD(x) (x * M_PI / 180.)
    #define VYXSSPL_RAD2GRAD(x) (x * 180. / M_PI)
    #define VYXSSPL_GENW(f)     (f * M_2PI)

    #if defined(__cplusplus)
        }
    #endif

#endif
/***********************************************************************
 $Log: vyxssplk.h,v $
 Revision 1.10  2002/05/07 18:38:38  vad
  Version 1.0.0
  Simple equalizer

 Revision 1.9  2002/04/21 21:26:53  vad
  Version 0.9.10
  Add-ons: amplifier, reverse, fade (in/out, lin/exp)

 Revision 1.8  2002/04/21 19:07:13  vad
  Version 0.9.9
  ⥪ ():    㯥ਮ

 Revision 1.7  2002/04/21 18:47:41  vad
  Version 0.9.8
  OnOffPulser

 Revision 1.6  2002/04/18 21:03:52  vad
  Version 0.9.7
    䨫஢

 Revision 1.5  2002/04/17 21:05:27  vad
  Version 0.9.6
   ⥭쪨 㭪  lpf/hpf  祭  A.I

 Revision 1.4  2002/04/16 21:25:15  vad
    ८ࠧ ⫨ (FHT):
  int  API_ENTRY VYXSSPL_initFHT(FHT_Data*, int, double, double, double)
  void API_ENTRY VYXSSPL_doFHT(FHT_Data*, double*, int)
  void API_ENTRY VYXSSPL_doneFHT(FHT_Data*)
  void API_ENTRY VYXSSPL_clearAccFHT(FHT_Data*)

 Revision 1.3  2002/04/15 20:42:18  vad
  5 osc. funcs
  int  VYXSSPL_initOsc(OSC_Data* posc, double  phase, double  k, double  f, double  fs, double  zero)
  void VYXSSPL_SineOsc(OSC_Data* posc, double* a, int N)
  void VYXSSPL_CosineOsc(OSC_Data* posc, double* a, int N)
  void VYXSSPL_WhiteNoiceOsc(OSC_Data* posc, double* a, int N)
  void VYXSSPL_PinkNoiceOsc(OSC_Data* posc, double* a, int N)

 Revision 1.2  2002/04/14 21:03:57  vad
 3   2 㭪樨:
     const char * API_ENTRY VYXSSPL_VersionStrK(void)
     int API_ENTRY VYXSSPL_isPwr2(unsigned long x)

 Revision 1.1  2002/04/14 20:37:15  vad
 砫쭠 .
 ᭮ 㭪樨 ࠡ  䨫ࠬ.  䨫  ॠ.

 ***********************************************************************/

