/*******************************************************************************
* FILE NAME: igrafctx.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in igrafctx.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*******************************************************************************/

extern "C"    // Declare a set of C constructs.
{
  #define INCL_GPI
  #define INCL_WINWINDOWMGR
  #define INCL_WINSYS
  #include <iwindefs.h>
}

#include <igrafctx.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <igline.hpp>
#include <igpyline.hpp>
#include <igrect.hpp>
#include <igelipse.hpp>
#include <igarc.hpp>
#include <igpie.hpp>
#include <igregion.hpp>
#include <igstring.hpp>
#include <iglist.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <ireslib.hpp>
#include <iwindow.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IGRAFCTX_CPP_
  #include <ipagetun.h>
#endif

// Conversion macro bwtween double and fix.
#define      DBL2FIXED(x)     ((FIXED)  ((x)*65536) )
#define      LONG2DBL(x)      ((double) (((double)x)/(double)65536))

#pragma data_seg(ICLNonConst)   // Class member variable initializations.
IColor                            IGraphicContext::fgCDPenColor(IColor::black);
IColor                            IGraphicContext::fgCDFillColor(IColor::black);
IColor                            IGraphicContext::fgCDBackgroundColor(IColor::white);
IGraphicBundle::DrawOperation     IGraphicContext::fgCDDrawOperation(IGraphicBundle::fillAndFrame);
IGraphicBundle::MixMode           IGraphicContext::fgCDMixMode(IGraphicBundle::overPaint);
IGraphicBundle::BackgroundMixMode IGraphicContext::fgCDBackgroundMixMode(IGraphicBundle::backLeaveAlone);
unsigned long                     IGraphicContext::fgCDPenPattern(IGraphicBundle::filled);
unsigned long                     IGraphicContext::fgCDFillPattern(IGraphicBundle::filled);
IGraphicBundle::PenType           IGraphicContext::fgCDPenType(IGraphicBundle::solid);
IGraphicBundle::PenEndingStyle    IGraphicContext::fgCDPenEndingStyle(IGraphicBundle::flat);
IGraphicBundle::PenJoiningStyle   IGraphicContext::fgCDPenJoiningStyle(IGraphicBundle::bevel);
unsigned long                     IGraphicContext::fgCDPenWidth(1);
IPoint                            IGraphicContext::fgCDPatternOrigin(IPoint(0,0));
#pragma data_seg()


//   A class derived from IBase to store data related to graphic context.
//   This class is a private class defined in the .cpp file to avoid
//   recompilation under different implementations.

class IGCData : public IBase {
// The benefit of applying typedef to the base class is that you only have to
// change one place when you change the base class to a different one.
typedef IBase
  Inherited;
public:
  IGCData  ( );
  ~IGCData ( );

// Wrapper for GpiSetPS
static void
  setPS ( IGraphicContext& gc, SIZEL sizlClient, unsigned long options );

IBitmapHandle
  fBitmapHandle;
unsigned long
  fBitmapId;
unsigned long
  recoordHeight;
Boolean
  fRecoordinate;
};


/*------------------------------------------------------------------------------
| IGCData::IGCData                                                             |
| Default constructor.                                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IGCData::IGCData( )
  : fBitmapHandle(0), fBitmapId(0), recoordHeight(0), fRecoordinate(false)
{}

/*------------------------------------------------------------------------------
| IGCData::~IGCData                                                            |
| Virtual destructor.                                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IGCData::~IGCData( )
{}

/*------------------------------------------------------------------------------
| IGCData::setPS                                                               |
|                                                                              |
------------------------------------------------------------------------------*/
void IGCData::setPS ( IGraphicContext& gc, SIZEL sizl, unsigned long options )
{
  // Get the presentation space
  IPresSpaceHandle fPs = gc.handle();

  // Save current graphic context   settings
  IColor                            currentPenColor    = gc.penColor();
  IColor                            currentFillColor   = gc.fillColor();
  IColor                            currentBackColor   = gc.backgroundColor();
  IGraphicBundle::DrawOperation     currentOperation   = gc.drawOperation();
  IGraphicBundle::MixMode           currentMixMode     = gc.mixMode();
  IGraphicBundle::BackgroundMixMode currentBackMix     = gc.backgroundMixMode();
  unsigned long                     currentPenPattern  = gc.penPattern();
  unsigned long                     currentFillPattern = gc.penPattern();
  IGraphicBundle::PenType           currentPenType     = gc.penType();
  IGraphicBundle::PenEndingStyle    currentEndStyle    = gc.penEndingStyle();
  IGraphicBundle::PenJoiningStyle   currentJoinStyle   = gc.penJoiningStyle();
  unsigned long                     currentPenWidth    = gc.penWidth();
  IPoint                            currentOrigin      = gc.patternOrigin();

  // Call the operating system
  if (!GpiSetPS( fPs, &sizl, options ))
  {
    ITHROWGUIERROR2("GpiSetPS",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // Restore the graphic context to RGB mode
  GpiCreateLogColorTable(fPs,0L,LCOLF_RGB,0L,0L,NULLHANDLE);

  // Restore graphic context settings
  gc.setPenColor(currentPenColor);
  gc.setFillColor(currentFillColor);
  gc.setBackgroundColor(currentBackColor);
  gc.setDrawOperation(currentOperation);
  gc.setMixMode(currentMixMode);
  gc.setBackgroundMixMode(currentBackMix);
  gc.setPenPattern(currentPenPattern);
  gc.setFillPattern(currentFillPattern);
  gc.setPenType(currentPenType);
  gc.setPenEndingStyle(currentEndStyle);
  gc.setPenJoiningStyle(currentJoinStyle);
  gc.setPenWidth(currentPenWidth);
  gc.setPatternOrigin(currentOrigin);
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPenColor                                             |
| Return default pen color.                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IColor IGraphicContext::defaultPenColor()
{
  return fgCDPenColor;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultFillColor                                            |
| Return default color for filling.                                                      |
|                                                                              |
------------------------------------------------------------------------------*/
IColor IGraphicContext::defaultFillColor()
{
  return fgCDFillColor;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultBackgroundColor                                      |
| Return the default background color.                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IColor IGraphicContext::defaultBackgroundColor()
{
  return fgCDBackgroundColor;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultDrawOperation                                        |
| Return the default draw operation. The DrawOperation is a data member of     |
| IGraphicBundle which contains a set of graphic attributes used by the graphic|
| context. DrawOperation basically determines the method used to draw a closed |
| graphic object.                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::DrawOperation IGraphicContext::defaultDrawOperation()
{
  return fgCDDrawOperation;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultMixMode                                              |
| Return the default mix mode. The MixMode is a data member of the class       |
| IGraphicBundle which contains a set of graphic attributes used by the graphic|
| context. MixMode basically determines how the pen and the color fill are     |
| combined with any existing drawings.                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::MixMode IGraphicContext::defaultMixMode()
{
  return fgCDMixMode;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultBackgroundMixMode                                    |
| Return the default background mix mode. The BackgroundMixMode is a data      |
| member of the class IGraphicBundle which contains a set of graphic attributes|
| used by the graphic context. BackgroundMixMode basically determines how the  |
| background color is combined with any existing drawings.                     |
------------------------------------------------------------------------------*/
IGraphicBundle::BackgroundMixMode IGraphicContext::defaultBackgroundMixMode()
{
  return fgCDBackgroundMixMode;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPenPattern                                           |
| Return default pen pattern.                                                  |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::defaultPenPattern()
{
  return fgCDPenPattern;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultFillPattern                                          |
| Return default filling pattern.                                              |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::defaultFillPattern()
{
  return fgCDFillPattern;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPenType                                              |
| Return default pen type;                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::PenType IGraphicContext::defaultPenType()
{
  return fgCDPenType;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPenEndingStyle                                       |
| Return default pen ending style.                                             |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::PenEndingStyle IGraphicContext::defaultPenEndingStyle()
{
  return fgCDPenEndingStyle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPenJoin                                              |
| Return default pen joining style.                                            |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::PenJoiningStyle IGraphicContext::defaultPenJoiningStyle()
{
  return fgCDPenJoiningStyle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPenWidth                                             |
| Return default pen width.                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::defaultPenWidth()
{
  return fgCDPenWidth;
}

/*------------------------------------------------------------------------------
| IGraphicContext::defaultPatternOrigin                                        |
| Return default pattern origin where the pattern spread vertically or         |
| horizontally.                                                                |
------------------------------------------------------------------------------*/
IPoint IGraphicContext::defaultPatternOrigin()
{
  return fgCDPatternOrigin;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPenColor                                          |
| Set color as default pen color.                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPenColor( const IColor& color )
{
  fgCDPenColor = color;
}
/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultFillColor                                         |
| Set color as default filling color.                                          |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultFillColor( const IColor& color )
{
  fgCDFillColor = color;
}
/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultBackgroundColor                                   |
| Set color as default background color.                                       |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultBackgroundColor( const IColor& color )
{
  fgCDBackgroundColor = color;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultDrawOperation                                     |
| Set the default draw operation. The DrawOperation is a data member of the    |
| IGraphicBundle which contains a set of graphic attributes used by the graphic|
| context. DrawOperation basically determines the method used to draw a closed |
| graphic object.                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultDrawOperation( IGraphicBundle::DrawOperation drawOperation )
{
  fgCDDrawOperation = drawOperation;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultMixMode                                           |
| Set the default mix mode. The MixMode is a data member of the class          |
| IGraphicBundle which contains a set of graphic attributes used by the graphic|
| context. MixMode basically determines how the pen and the color fill are     |
| combined with any existing drawings.                                         |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultMixMode( IGraphicBundle::MixMode mixMode )
{
  fgCDMixMode = mixMode;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultBackgroundMixMode                                 |
| Set the default background mix mode. The BackgroundMixMode is a data         |
| member of the class IGraphicBundle which contains a set of graphic attributes|
| used by the graphic context. BackgroundMixMode basically determines how the  |
| background color is combined with any existing drawings.                     |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultBackgroundMixMode(
                           IGraphicBundle::BackgroundMixMode backgroundMixMode )
{
  fgCDBackgroundMixMode = backgroundMixMode;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPenPattern                                        |
| Set default pen pattern.                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPenPattern( unsigned long penPattern )
{
  fgCDPenPattern = penPattern;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultFillPattern                                       |
| Set default filling pattern.                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultFillPattern( unsigned long fillPattern )
{
  fgCDFillPattern = fillPattern;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPenType                                           |
| Set default pen style.                                                       |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPenType( IGraphicBundle::PenType penType )
{
  fgCDPenType = penType;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPenEndingStyle                                    |
| Set default pen ending style.                                                |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPenEndingStyle(
                                 IGraphicBundle::PenEndingStyle penEndingStyle )
{
  fgCDPenEndingStyle = penEndingStyle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPenJoin                                           |
| Set default pen joining style.                                               |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPenJoiningStyle(
                               IGraphicBundle::PenJoiningStyle penJoiningStyle )
{
  fgCDPenJoiningStyle = penJoiningStyle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPenWidth                                          |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPenWidth( unsigned long penWidth )
{
  fgCDPenWidth = penWidth;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDefaultPatternOrigin                                     |
| Set default pattern origin where the pattern spreads out either vertically or|
| horizontally.                                                                |
------------------------------------------------------------------------------*/
void IGraphicContext::setDefaultPatternOrigin( const IPoint& point )
{
  fgCDPatternOrigin = point;
}

/*------------------------------------------------------------------------------
| IGraphicContext::initialize                                                  |
| Initialization of all graphical attributes.                                  |
| All defaults are static data assigned in the begining of this .cpp file.     |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::initialize()
{
  fGCData = new IGCData();
  // Create and set the logical color table. A GPI call.
  GpiCreateLogColorTable(fPs,0L,LCOLF_RGB,0L,0L,NULLHANDLE);

  // Set all graphic attributes to their default values.
  setPenColor(defaultPenColor());
  setFillColor(defaultFillColor());
  setBackgroundColor(defaultBackgroundColor());
  setDrawOperation(defaultDrawOperation());
  setMixMode(defaultMixMode());
  setBackgroundMixMode(defaultBackgroundMixMode());
  setPenPattern(defaultPenPattern());
  setFillPattern(defaultFillPattern());
  setPenType(defaultPenType());
  setPenEndingStyle(defaultPenEndingStyle());
  setPenJoiningStyle(defaultPenJoiningStyle());
  setPenWidth(defaultPenWidth());
  setPatternOrigin(defaultPatternOrigin());

  // Return a reference of this object.
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::IGraphicContext                                             |
|                                                                              |
| Create a memory PS. fCm is the creation mode.                                |
------------------------------------------------------------------------------*/
IGraphicContext::IGraphicContext( )
  : fGCData(0),fFillColor(defaultFillColor()),
    fFillPattern(defaultFillPattern()), fDrawOp(defaultDrawOperation()),
    fFont(0), fCm( Graphics ), fDc(0), fBitmap(0)

{
  // Context data structure. Device open structure.
  DEVOPENSTRUC dop = {0, (PSZ)"DISPLAY", 0, 0, 0, 0, 0, 0, 0};
  // Use the same page size as the device context.
  SIZEL        sizl={0, 0};
  // Get the handle of the anchor block of the current thread.
  HAB          habTemp = IThread::current().anchorBlock();
  char*        pszError = 0;

  // Create memory device context
  fDc = (unsigned long)DevOpenDC( habTemp,           //anchor block handle
                                 OD_MEMORY,          //type of device context: here memory!
                                 (PSZ)"*",           //device information token.
                                 5L,                 //number of item.
                                 (PDEVOPENDATA)&dop, //data area.
                                 NULLHANDLE);        //device context handle.

  if(!fDc)
  {
     pszError = "DevOpenDC";
  }
  else
  {
    // Create a presentation space associated with the context
    fPs = GpiCreatePS(habTemp, (HDC)fDc, &sizl,
                     GPIA_ASSOC | PU_PELS | GPIT_MICRO);

    // Load the Bitmap or Pointer
    if (!fPs)
      pszError = "GpiCreatePS";
    else
      initialize(); //initialize all graphical attributes.
  }
  if(pszError)
    ITHROWGUIERROR(IString(pszError));  //throw exception.
}

/*------------------------------------------------------------------------------
| IGraphicContext::IGraphicContext                                             |
|                                                                              |
| Wrapper an existing PS. Construct an IGraphicContext from an existing        |
| presentation space handle. fCm is the creation mode.                         |
------------------------------------------------------------------------------*/
IGraphicContext::IGraphicContext( const IPresSpaceHandle& presSpaceHandle )
  : fGCData(0),fPs( presSpaceHandle ), fFillColor(defaultFillColor()),
    fFillPattern(defaultFillPattern()), fDrawOp(defaultDrawOperation()),
    fFont(0), fCm( None ), fDc(0), fBitmap(0)
{
  // Initialize all graphical attributes.
  initialize();
}

/*------------------------------------------------------------------------------
| IGraphicContext::IGraphicContext                                             |
|                                                                              |
| Get a screen PS associated with the given window handle.                     |
| Construct a IGraphicContext from an IWindowHandle.                           |
|-----------------------------------------------------------------------------*/
IGraphicContext::IGraphicContext( const IWindowHandle& windowHandle )
  : fGCData(0),fFillColor(defaultFillColor()),
    fFillPattern(defaultFillPattern()), fDrawOp(defaultDrawOperation()),
    fFont(0), fCm( WindowHandle ), fDc(0), fBitmap(0)
{
  // If desktop handle passed in, need special case call
  if (windowHandle == IWindow::desktopWindow()->handle())
    fPs = WinGetScreenPS( HWND_DESKTOP );
  else
    fPs = WinGetPS( windowHandle );

  initialize();

#ifdef IC_PM
  // Check for BIDI (bidirectional character set) support
  if ( IBidiSettings::isBidiSupported() )
  {
    // Copy the BIDI attributes from the window to the graphic context
    IBidiSettings bidi(windowHandle);
    bidi.apply(*this);
  }
#endif

}

/*------------------------------------------------------------------------------
| IGraphicContext::IGraphicContext                                             |
|                                                                              |
| Create a memory PS with the specified size for temporary drawing. fCm is the |
| creation mode.                                                               |
------------------------------------------------------------------------------*/
IGraphicContext::IGraphicContext(const ISize& contextSize)
  : fGCData(0),fFillColor(defaultFillColor()),
    fFillPattern(defaultFillPattern()), fDrawOp(defaultDrawOperation()),
    fFont(0), fCm( Graphics ), fDc(0), fBitmap(0)

{
  // Context data structure, the device open data structure
  DEVOPENSTRUC dop = {0, (PSZ)"DISPLAY", 0, 0, 0, 0, 0, 0, 0};
  // Convert the current device context size.
  SIZEL        sizl= contextSize.asSIZEL();
  // Get the current residing thread anchor block handler.
  HAB          habTemp = IThread::current().anchorBlock();
  char*        pszError = 0;

  // Create memory device context.
  fDc = (unsigned long)DevOpenDC( habTemp,
                                 OD_MEMORY,     //type: memory device context.
                                 (PSZ)"*",      //device information token.
                                 5L,            //number of items
                                 (PDEVOPENDATA)&dop,   //data area.
                                 NULLHANDLE);          //device context handle.

  // Create a presentation space associated with the context.
  if(!fDc)
  {
     pszError = "DevOpenDC";
  }
  else
  {
    // Create a presentation space, a micro space.
    fPs = GpiCreatePS(habTemp, (HDC)fDc, &sizl,
                     GPIA_ASSOC | PU_PELS | GPIT_MICRO);

    // Load the Bitmap or Pointer.
    if (!fPs)
      pszError = "GpiCreatePS";
    else
      initialize();
  }

  // Throw exception if failure.
  if(pszError)
    ITHROWGUIERROR(IString(pszError));

  // The user specified a size, so make a bitmap with the specified
  // size and set it into the presentation space

  BITMAPINFOHEADER2  bmih2;
  memset ( &bmih2, 0, sizeof(BITMAPINFOHEADER2) );  // RESET BMIH2 TO ALL 0.

  // Set bitmapinforheader bmih2.
  bmih2.cbFix = sizeof(BITMAPINFOHEADER2);
  // Copy over the size of the device context.
  bmih2.cx = contextSize.width();
  bmih2.cy = contextSize.height();
  bmih2.cPlanes = 1;
  bmih2.cBitCount = 24;
  // Create a bitmap in the device context for drawing.
  fBitmap = GpiCreateBitmap ( fPs, &bmih2, 0L, NULL, NULL );
  if (!fBitmap)
    ITHROWGUIERROR("GpiCreateBitmap");

  // Set the bitmap as a current selected bitmap
  // to the memory device context.
  GpiSetBitmap(fPs, fBitmap);
}

/*------------------------------------------------------------------------------
| IGraphicContext::setMappingMode                                              |
| Define the units of measure in the mapping between page and device space.    |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setMappingMode( MappingMode mappingMode )
{
  unsigned long options = GPIF_LONG;   // GPIF_LONG is a 4-byte int.
  SIZEL sizlClient = {0,0};

  switch (mappingMode)
  {
    // Logical units in the x and y direction are equal in the mapping.
    case isotropic:
      options |= PU_ARBITRARY;
      // Query for client size.
      sizlClient.cx = WinQuerySysValue(HWND_DESKTOP, SV_CXFULLSCREEN);
      sizlClient.cy = WinQuerySysValue(HWND_DESKTOP, SV_CYFULLSCREEN);
    break;
    case pels:        //pel coordinates.
      options |= PU_PELS;
    break;
    case lowMetric:   //units of 0.1 millimeter.
      options |= PU_LOMETRIC;
    break;
    case highMetric:  //units of 0.01 millimeter.
      options |= PU_HIMETRIC;
    break;
    case lowEnglish:  //units of 0.01 inch.
      options |= PU_LOENGLISH;
    break;
    case highEnglish: //units of 0.001 inch.
      options |= PU_HIENGLISH;
    break;
    case twips:       //units of 1/1440 inch.
      options |= PU_TWIPS;
    break;
  } /* endswitch */
  IGCData::setPS( *this, sizlClient, options );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::mappingMode                                                 |
| Return the mapping mode currently in use by the device context.              |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicContext::MappingMode IGraphicContext::mappingMode( ) const
{
  SIZEL temp;
  long map = GpiQueryPS( fPs, &temp );

  switch (map & PS_UNITS)
  {
    // Logical units in the x and y direction are equal in the mapping.
    case PU_ARBITRARY:
      return isotropic;
    case PU_PELS:          //pel coordinates.
      return pels;
    case PU_LOMETRIC:      //units of 0.1 millimeter.
      return lowMetric;
    case PU_HIMETRIC:      //units of 0.01 millimeter.
      return highMetric;
    case PU_LOENGLISH:     //units of 0.01 inch.
      return lowEnglish;
    case PU_HIENGLISH:     //units of 0.001 inch.
      return highEnglish;
    case PU_TWIPS:         //units of 1/1440 inch.
      return twips;
    default:
      return pels;
  } /* endswitch */
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPageSize                                                 |
| Set the presentation page size.                                              |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPageSize( const ISize& pageSize )
{
  SIZEL size;
  // Query the map used in the folloing GpiSetPS.
  long map(GpiQueryPS( fPs, &size ));
  size = pageSize.asSIZEL();   // Convert the page size.
  IGCData::setPS( *this, size, (map & PS_UNITS) | GPIF_LONG | PS_NORESET );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::pageSize                                                    |
| Return the current presentation page size.                                   |
|                                                                              |
------------------------------------------------------------------------------*/
ISize IGraphicContext::pageSize( ) const
{
  SIZEL pageSize;
  // Query the page size.
  long map(GpiQueryPS( fPs, &pageSize ));
  return ISize( pageSize );
}

/*------------------------------------------------------------------------------
| IGraphicContext::setViewPortRect                                             |
| Set a viewport on the device context.                                        |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setViewPortRect( const IRectangle& viewportRect )
{
  // Convert object to system dependent struct.
  RECTL rect(viewportRect.asRECTL());
  // Set a viewport rect on the device context.
  GpiSetPageViewport( fPs, &rect );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::viewPortRect                                                |
| Return viewport rectangle.                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IRectangle IGraphicContext::viewPortRect( ) const
{
  RECTL rect;
  // Query the viewport rectangle.
  GpiQueryPageViewport( fPs, &rect );
  return IRectangle( rect );   // convert a RECTL to an IRectangle.
}

/*------------------------------------------------------------------------------
| IGraphicContext::~IGraphicContext                                            |
| Virtual destructor which closes the device context opened in the constructor.|
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicContext::~IGraphicContext( )
{
  // Delete the IFont set for the gc.
  if (fFont)
  {
    fFont->endUsingFont( fPs );
    delete fFont;
  }

  // Delete temporary drawing bitmap (if needed)
  if (fBitmap)
  {
     GpiSetBitmap(fPs,0);
     GpiDeleteBitmap(fBitmap);
  }

  // If this is a memory device context. Here fCm is the creation mode.
  if ( fCm == Graphics )
  {
    // Disassociate a presnetation space from its device context.
    GpiAssociate( fPs, 0 );
    // Destory the PS.
    GpiDestroyPS( fPs );
    // Close the DC.
    DevCloseDC((HDC)fDc);
  }
  else if ( fCm == WindowHandle )  // created using a window handle.
  {
    // Only release the PS.
    WinReleasePS( fPs );
  }
  if (fGCData)
  {
    delete fGCData;
  }
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw  is a set of overloaded methods on all IG* to be       |
| drawn on the device context.                                                 |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGLine& line )
{
  // Get the individual points of the line.
  IPoint    start1(line.startingPoint());
  IPoint    end1(line.endingPoint());
  IGraphicBundle*   currentBundle;
  ITransformMatrix* currentTransform;
  long      rc;

  if ( fGCData->fRecoordinate )
  {
    start1.setY( fGCData->recoordHeight - start1.y() );
    end1.setY( fGCData->recoordHeight - end1.y() );
  }

  // Convert ICLUI object to PM struct.
  POINTL    start( start1.asPOINTL() );
  POINTL    end( end1.asPOINTL() );

  // Apply line's transformation matrix and IGraphicBundle to this
  // graphic context.
  apply( line, &currentTransform, &currentBundle );

  // Specify the start of a path if pen width is greater than 1.
  if ( penWidth() > 1 )
    // The drawing has to be put in a path bracket if the pen
    // width is greater than 1.
    if (!GpiBeginPath(fPs, 1L ))
      ITHROWGUIERROR2("GpiBeginPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
  // Move to the start point of the line.
  GpiMove( fPs, &start );
  // Draw the line from start to end point.
  rc = GpiLine( fPs, &end );
  if (rc == GPI_ERROR)
  {
    ITHROWGUIERROR2("GpiLine",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  else if (rc == GPI_HITS && line.isHitSelectable()) //the draw is in hit detect mode
  {
    ((IGLine&)line).setHitSelected(true);
  }
  if ( penWidth() > 1 )
  {
    // The drawing has to be put in a path bracket if the pen
    // width is greater than 1. Here the path is ended.
    if (!GpiEndPath(fPs))    //end "path"
      ITHROWGUIERROR2("GpiEndPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    // You have to use GpiStrokePath to draw if pen width is greater than 1.
    rc = GpiStrokePath(fPs, 1L, 0);
    if (rc == GPI_HITS && line.isHitSelectable())
      ((IGLine&)line).setHitSelected(true);
  }
  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( line, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw polyline.                                                               |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGPolyline& polyline )
{
  if (polyline.numberOfPoints() > 1)
  {
    ITransformMatrix* currentTransform;
    IGraphicBundle*   currentBundle;
    long              rc;
    // Get the number of points in the polyline.
    unsigned long     arraySize(polyline.pointArray().size());
    // Create a POINTL array to store the points.
    PPOINTL           pPtl(new POINTL[arraySize-1]);
    IPoint            startPt( polyline.point(0) );

    // Apply polyline's transformation matrix and IGraphicBundle to
    // this graphic context.
    apply( polyline, &currentTransform, &currentBundle );

    // Convert all IPoints to POINTLs.
    for (int i=1; i < arraySize; i++)
    {
      pPtl[i-1] = polyline.point(i).asPOINTL();
      if ( fGCData->fRecoordinate )
        pPtl[i-1].y = fGCData->recoordHeight - pPtl[i-1].y;
    } /* endfor */

    // Move the drawing position to first point.
    if ( fGCData->fRecoordinate )
      startPt.setY( fGCData->recoordHeight - startPt.y() );
    setCurrentDrawingPosition( startPt );

    // The drawing has to be put in a path bracket if the pen
    // width is greater than 1.
    if ( penWidth() > 1 )
      if (!GpiBeginPath(fPs, 1L ))      //if pen width >1 wrap the draw within a path.
        ITHROWGUIERROR2("GpiBeginPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
    // Draw the rest of the polyline.
    rc = GpiPolyLine( fPs, arraySize-1, pPtl );
    if ( rc == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiPolyLine",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    else if ( rc == GPI_HITS && polyline.isHitSelectable()) //hit detecting, not drawing
    {
      ((IGPolyline&)polyline).setHitSelected(true);  //I am hit!
    }

    // The drawing has to be put in a path bracket if the pen
    // width is greater than 1. Here the path is ended.
    if ( penWidth() > 1 )
    {
      if (!GpiEndPath(fPs))   // End path if pen width > 1.
        ITHROWGUIERROR2("GpiEndPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      rc = GpiStrokePath(fPs, 1L, 0);
      if (rc == GPI_HITS && polyline.isHitSelectable())  // This is a hit detection.
        ((IGLine&)polyline).setHitSelected(true);
    }

    delete [] pPtl;  // Delete the temp POINTL array.
    //Restore this graphic context's own transformation matrix and graphic bundle.
    remove( polyline, &currentTransform, &currentBundle );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw polygon.                                                                |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGPolygon& polygon )
{
  if (polygon.numberOfPoints() > 1)
  {
    ITransformMatrix* currentTransform;
    IGraphicBundle*   currentBundle;
    // Get the number of points in the polygon.
    unsigned long     arraySize(polygon.pointArray().size());
    // Create a temp POINTL array to store points for drawing.
    PPOINTL           pPtl(new POINTL[arraySize-1]);
    // Convert the first point in the polygon.
    IPoint            startPt( polygon.point(0) );
    // Get the polygon filling mode.
    long              fillMode(polygon.fillMode() == IGPolygon::alternate
                                                   ? FPATH_ALTERNATE
                                                   : FPATH_WINDING);
    long    rc;

    // Apply polygon's transformation matrix and IGraphicBundle to
    // this graphic context.
    apply( polygon, &currentTransform, &currentBundle );

    // Recoordinate the starting point if needed.
    if ( fGCData->fRecoordinate )
      startPt.setY( fGCData->recoordHeight - startPt.y() );

    // Convert all IPoints to POINTLs.
    for (int i=1; i < arraySize; i++)
    {
      pPtl[i-1] = polygon.point(i).asPOINTL();
      if ( fGCData->fRecoordinate )
        pPtl[i-1].y = fGCData->recoordHeight - pPtl[i-1].y;
    } /* endfor */

    // Move the current drawing position to the first point
    // of the polygon.
    setCurrentDrawingPosition( startPt );

    // A closed graphic is usually drawn twice; first the interior
    // and then the border.
    if ( drawOperation() == IGraphicBundle::fill ||  //drawing in filled pattern.
         drawOperation() == IGraphicBundle::fillAndFrame )
    {
      // Save current IGraphicContext's graphic attributes.
      unsigned long currentPenPattern(penPattern());
      IColor currentPenColor(penColor());
      setPenColor( fillColor() );
      setPenPattern( fillPattern() );
        // Create a path because there is a filling pattern.
        if (!GpiBeginPath(fPs, 1L ))
          ITHROWGUIERROR2("GpiBeginPath",
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
      // Draw the rest of the polygon.
      if (GpiPolyLine( fPs, arraySize-1, pPtl ) == GPI_ERROR )
      {
        ITHROWGUIERROR2("GpiPolyLine",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      }
      if (!GpiEndPath(fPs))
        ITHROWGUIERROR2("GpiEndPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      // Hit detection mode. GpiFillPath(...) outputs the interior of the polygon.
      if ( GpiFillPath( fPs, 1L, fillMode ) == GPI_HITS && polygon.isHitSelectable())
      {
        ((IGPolygon&)polygon).setHitSelected(true);
      }
      // Restore current IGraphicContext's graphic attributes.
      setPenColor( currentPenColor );
      setPenPattern( currentPenPattern );
      setCurrentDrawingPosition( startPt );
    }

    // If in frame mode, draw frame too.
    if ( drawOperation() == IGraphicBundle::frame ||
         drawOperation() == IGraphicBundle::fillAndFrame )
    {
      if ( penWidth() > 1)   // Create a path if pen width > 1
        if (!GpiBeginPath(fPs, 1L ))
          ITHROWGUIERROR2("GpiBeginPath",
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
      // Draw the border of this polygon.
      rc = GpiPolyLine( fPs, arraySize-1, pPtl );
      if (rc == GPI_ERROR )
      {
        ITHROWGUIERROR2("GpiPolyLine",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      }
      // If this is a hit detection rather than drawing.
      else if ( rc == GPI_HITS && polygon.isHitSelectable())
      {
        ((IGPolygon&)polygon).setHitSelected(true);
      }
      // Close the border.
      POINTL tempPt( startPt.asPOINTL() );
      rc = GpiLine( fPs, &tempPt );
      if (rc == GPI_ERROR )
      {
        ITHROWGUIERROR2("GpiLine",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      }
      else if (rc == GPI_HITS && polygon.isHitSelectable()) // Hit detection.
      {
        ((IGPolygon&)polygon).setHitSelected(true); // Polygon is hit.
      }
      if ( penWidth() > 1)
      {
        if (!GpiEndPath(fPs))
          ITHROWGUIERROR2("GpiEndPath",
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
        rc = GpiStrokePath(fPs, 1L, 0);  // draw the entire path for border.
        if (rc == GPI_HITS && polygon.isHitSelectable())
          ((IGLine&)polygon).setHitSelected(true);
      }
    } /* endif */

    // Use [] to delete an array of points.
    delete [] pPtl;
    // Restore this graphic context's own transformation matrix and graphic bundle.
    remove( polygon, &currentTransform, &currentBundle );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw ellipse.                                                                |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGEllipse& ellipse )
{
  IRectangle bRect(ellipse.enclosingRect());  // Convert the enclosing rect.
  IGraphicBundle*   currentBundle;
  ITransformMatrix *currentTransform;
  long       rc;

  // Apply ellipse's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( ellipse, &currentTransform, &currentBundle );

  if ( fGCData->fRecoordinate )
  {
    IPoint minXminY(bRect.minXMinY());
    IPoint maxXmaxY(bRect.maxXMaxY());
    minXminY.setY( fGCData->recoordHeight - minXminY.y() );
    maxXmaxY.setY( fGCData->recoordHeight - maxXmaxY.y() );
    bRect = IRectangle( minXminY, maxXmaxY );
  }

  // Convert the center point.
  POINTL centerPt( bRect.center().asPOINTL() );

  // Move drawing position to center.
  GpiMove(fPs, &centerPt);

  // Construct a ARCPARAMS for the ellipse which will be used by PM GPI call.
  ARCPARAMS arcp = { 0, 0, bRect.rightCenter().x() - bRect.center().x(),
                           bRect.topCenter().y() - bRect.center().y() };

  if (!GpiSetArcParams( fPs, &arcp ))  // Set the arc parameter.
  {
    ITHROWGUIERROR2("GpiSetArcParams",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // If there is a filling pattern for the interior.
  if ( drawOperation() == IGraphicBundle::fill ||
       drawOperation() == IGraphicBundle::fillAndFrame )
  {
    // Save current IGraphicContext's graphic attributes.
    unsigned long currentPenPattern(penPattern());
    IColor currentPenColor(penColor());
    setPenPattern( fillPattern() );
    setPenColor( fillColor() );
    // Draw the interior of the full arc with the filling pattern.
    rc = GpiFullArc( fPs, DRO_FILL, MAKEFIXED(1, 0));
    if (rc == GPI_ERROR)
    {
      ITHROWGUIERROR2("GpiFullArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If this drawing is just for hit detection.
    else if ( rc == GPI_HITS && ellipse.isHitSelectable())
    {
      ((IGEllipse&)ellipse).setHitSelected(true); // ellipse set to be hit.
    }
    // Restore current IGraphicContext's graphic attributes.
    setPenPattern( currentPenPattern );
    setPenColor( currentPenColor );
  }

  // Draw the border of the ellipse.
  if ( drawOperation() == IGraphicBundle::frame ||
       drawOperation() == IGraphicBundle::fillAndFrame )
  {
    if ( penWidth() > 1 )     // Create a path if pen width is > 1
        if (!GpiBeginPath(fPs, 1L ))
          ITHROWGUIERROR2("GpiBeginPath",
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
    // Draw the border of the ellipse.
    rc = GpiFullArc( fPs, DRO_OUTLINE, MAKEFIXED(1, 0));
    if (rc == GPI_ERROR)
    {
      ITHROWGUIERROR2("GpiFullArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If the drawing is just for hit detection.
    else if (rc == GPI_HITS && ellipse.isHitSelectable())
    {
      ((IGEllipse&)ellipse).setHitSelected(true);  // Ellipse set to be hit.
    }
    if ( penWidth() > 1)
    {
      if (!GpiEndPath(fPs))
        ITHROWGUIERROR2("GpiEndPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      rc = GpiStrokePath(fPs, 1L, 0); // Draw the eitire path.
      if (rc == GPI_HITS && ellipse.isHitSelectable())   // Hit detection, not drawing.
        ((IGLine&)ellipse).setHitSelected(true);
    }
  } /* endif */
  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( ellipse, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw rectangle.                                                              |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGRectangle& graphicRectangle )
{
  IRectangle        rect(graphicRectangle.enclosingRect());
  IGraphicBundle*   currentBundle;
  ITransformMatrix* currentTransform;
  long              rc;

  // Apply graphicRectangle's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( graphicRectangle, &currentTransform, &currentBundle );

  if ( fGCData->fRecoordinate )
  {
    IPoint minXminY(rect.minXMinY());
    IPoint maxXmaxY(rect.maxXMaxY());
    minXminY.setY( fGCData->recoordHeight - minXminY.y() );
    maxXmaxY.setY( fGCData->recoordHeight - maxXmaxY.y() );
    rect = IRectangle( minXminY, maxXmaxY );
  }

  // Convert the top right point to POINTL.
  POINTL ptl( rect.topRight().asPOINTL());

  // Set the current drawing position to bottom left corner of the rect.
  setCurrentDrawingPosition( rect.bottomLeft() );

  // If there is a filling pattern for the interior.
  if ( drawOperation() == IGraphicBundle::fill ||
       drawOperation() == IGraphicBundle::fillAndFrame )
  {
    // Save current IGraphicContext's graphic attributes
    // which is not in the graphic bundle.
    unsigned long currentPenPattern(penPattern());
    IColor currentPenColor(penColor());
    setPenColor( fillColor() );
    setPenPattern( fillPattern() );

    // Draw the rect. You do not need a path here! Check GPI manual.
    rc = GpiBox( fPs, DRO_FILL, &ptl,
                 graphicRectangle.rounding().coord1(),
                 graphicRectangle.rounding().coord2() );
    if (rc == GPI_ERROR)
    {
      ITHROWGUIERROR2("GpiBox",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If the drawing is for hit detection.
    else if (rc == GPI_HITS && graphicRectangle.isHitSelectable())
    {
      ((IGRectangle&)graphicRectangle).setHitSelected(true); // set to be hit.
    }
    // Restore current IGraphicContext's graphic attributes.
    setPenColor( currentPenColor );
    setPenPattern( currentPenPattern );
  }

  // Draw the border if border is needed.
  if ( drawOperation() == IGraphicBundle::frame ||
       drawOperation() == IGraphicBundle::fillAndFrame )
  {
    if ( penWidth() > 1 )   // Create a path if pen width > 1.
      if (!GpiBeginPath(fPs, 1L ))
        ITHROWGUIERROR2("GpiBeginPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
    // Draw box's border.
    rc = GpiBox( fPs, DRO_OUTLINE, &ptl,
                 graphicRectangle.rounding().coord1(),
                 graphicRectangle.rounding().coord2() );
    if (rc == GPI_ERROR)
    {
      ITHROWGUIERROR2("GpiBox",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If the drawing is for hit detection.
    else if (rc == GPI_HITS && graphicRectangle.isHitSelectable())
    {
      ((IGRectangle&)graphicRectangle).setHitSelected(true);  // set to be hit.
    }
    if ( penWidth() > 1 )
    {
      if (!GpiEndPath(fPs))
        ITHROWGUIERROR2("GpiEndPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      rc = GpiStrokePath(fPs, 1L, 0); // draw the entire path.
      // If the drawing is for hit detection. Here, the pen width > 1.
      if (rc == GPI_HITS && graphicRectangle.isHitSelectable())
        ((IGLine&)graphicRectangle).setHitSelected(true);
    }
  } /* endif */

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( graphicRectangle, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw arc.                                                                    |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGArc& arc )
{
  // Here arc is represented in its enclosing rec.
  IRectangle        rect(arc.enclosingRect());
  IGraphicBundle*   currentBundle;
  ITransformMatrix* currentTransform;
  long              rc;

  if ( fGCData->fRecoordinate )
  {
    IPoint minXminY(rect.minXMinY());
    IPoint maxXmaxY(rect.maxXMaxY());
    minXminY.setY( fGCData->recoordHeight - minXminY.y() );
    maxXmaxY.setY( fGCData->recoordHeight - maxXmaxY.y() );
    rect = IRectangle( minXminY, maxXmaxY );
  }
  POINTL centerPt( rect.center().asPOINTL() );

  // Apply arc's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( arc, &currentTransform, &currentBundle );

  // Convert an IGArc to a PM struct.
  ARCPARAMS arcp = { rect.rightCenter().x() - rect.center().x(),
                     rect.topCenter().y()   - rect.center().y(), 0,0 };
  // Set the direction of arc to arcp.
  if (arc.direction() == IGArc::clockwise)
  {
    arcp.lQ = -arcp.lQ;
  }

  // Set GPI arc parameter.
  if (!GpiSetArcParams( fPs, &arcp ))
  {
    ITHROWGUIERROR2("GpiSetArcParams",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // Save current IGraphicContext's graphic attributes which
  // are not in the graphic bundle.
  IGraphicBundle::PenType currentPenType(penType());

  // Set the pen to invisible to eliminate the partial arc drawing
  // from the center the the edge of the arc. Move the current
  // drawing position to the edge of the arc without drawing the line
  // from the center to the edge of the arc.
  setPenType( IGraphicBundle::invisible );
  if ( GpiPartialArc( fPs, &centerPt, MAKEFIXED(1, 0),
                      DBL2FIXED(arc.startAngle()),
                      (FIXED)0 ) == GPI_ERROR )  //no sweeping!!!
  {
    ITHROWGUIERROR2("GpiPartialArc",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // Restore current IGraphicContext's graphic attributes which
  // are not in the graphic bundle.
  setPenType( currentPenType );

  if ( penWidth() > 1 )
    if (!GpiBeginPath(fPs, 1L ))
      ITHROWGUIERROR2("GpiBeginPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);

  // Draw only the arc.
  rc = GpiPartialArc( fPs, &centerPt, MAKEFIXED(1, 0),
                      DBL2FIXED(arc.startAngle()),
                      DBL2FIXED(arc.sweepAngle()) );
  if (rc == GPI_ERROR)
  {
    ITHROWGUIERROR2("GpiPartialArc",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // If the drawing is to hit detection.
  else if (rc == GPI_HITS && arc.isHitSelectable())
  {
    ((IGArc&)arc).setHitSelected(true); //arc set to be hit.
  }
  if ( penWidth() > 1 )
  {
    if (!GpiEndPath(fPs))
      ITHROWGUIERROR2("GpiEndPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    rc = GpiStrokePath(fPs, 1L, 0);
    // If the drawing is to hit detection. Here, the pen width > 1.
    if (rc == GPI_HITS && arc.isHitSelectable())
      ((IGLine&)arc).setHitSelected(true); // Arc set to be hit.
  }

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( arc, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw 3 point arc.                                                            |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IG3PointArc& arc )
{
  // Convert IG3PointArc to PM struct.
  POINTL    aptl[2] = {arc.intermediatePoint().x(),arc.intermediatePoint().y(),
                       arc.endingPoint().x(),arc.endingPoint().y()};
  ARCPARAMS arcp = { 65536, 65536, 0, 0 }; // Initialize the arc parameter.
  IGraphicBundle*  currentBundle;
  ITransformMatrix *currentTransform;
  IPoint    start( arc.startingPoint() );
  long      rc;

  if ( fGCData->fRecoordinate )
  {
    start.setY( fGCData->recoordHeight - start.y() );
    aptl[0].y = fGCData->recoordHeight - aptl[0].y;
    aptl[1].y = fGCData->recoordHeight - aptl[1].y;
  }

  // Apply arc's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( arc, &currentTransform, &currentBundle );

  if (!GpiSetArcParams( fPs, &arcp ))
  {
    ITHROWGUIERROR2("GpiSetArcParams",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // Move the drawing position to the start of the arc.
  setCurrentDrawingPosition( start );
  if ( penWidth() > 1 )   // begin a path if pen width > 1.
    if (!GpiBeginPath(fPs, 1L ))
      ITHROWGUIERROR2("GpiBeginPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
  // Draw the arc.
  rc = GpiPointArc( fPs, aptl );
  if (rc == GPI_ERROR )
  {
    ITHROWGUIERROR2("GpiPointArc",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // If the drawing is for hit detection.
  else if ( rc == GPI_HITS && arc.isHitSelectable())
  {
    ((IG3PointArc&)arc).setHitSelected(true);  // set arc to be hit.
  }
  if ( penWidth() > 1 )
  {
    if (!GpiEndPath(fPs))
      ITHROWGUIERROR2("GpiEndPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    rc = GpiStrokePath(fPs, 1L, 0);
    // If the drawing is for hit detection. Here pen width > 1.
    if (rc == GPI_HITS && arc.isHitSelectable())
      ((IGLine&)arc).setHitSelected(true); // set arc to be hit.
  }

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( arc, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw pie: an arc plus two lines from the center to two edges of the arc.     |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGPie& pie )
{
  IRectangle rect(pie.enclosingRect()); //the pie is represented by its enclosing box.
  IGraphicBundle*   currentBundle;
  ITransformMatrix* currentTransform;
  long      rc;

  // Convert data structure to PM.
  ARCPARAMS arcp = { rect.rightCenter().x() - rect.center().x(),
                     rect.topCenter().y()   - rect.center().y(), 0,0 };
  // Convert the center point.
  POINTL    ptl( rect.center().asPOINTL() );

  // Apply pie's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( pie, &currentTransform, &currentBundle );

  // Set the GPI arc parameter.
  if (!GpiSetArcParams( fPs, &arcp ))
  {
    ITHROWGUIERROR2("GpiSetArcParams",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // Set the current drawing position.
  setCurrentDrawingPosition( rect.center() );

  // If there is a fillinf pattern for the interior.
  if (drawOperation() == IGraphicBundle::fill ||
      drawOperation() == IGraphicBundle::fillAndFrame )
  {
    // Save current IGraphicContext's graphic attributes
    // which are not in the graphic bundle.
    unsigned long currentPenPattern(penPattern());
    IColor currentPenColor(penColor());
    setPenColor( fillColor() );
    setPenPattern( fillPattern() );
    if (!GpiBeginPath(fPs, 1L ))
      ITHROWGUIERROR2("GpiBeginPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    rc = GpiPartialArc( fPs, &ptl, MAKEFIXED(1, 0),
                        DBL2FIXED(pie.startAngle()),
                        DBL2FIXED(pie.sweepAngle()) );
    if (rc == GPI_ERROR)
    {
      ITHROWGUIERROR2("GpiPartialArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    else if (rc == GPI_HITS && pie.isHitSelectable())  // If hit detection is on...
    {
      ((IGPie&)pie).setHitSelected(true);   // Set pie to be hit.
    }

    // Finish the line from end of the arc to the center.
    if ( GpiLine( fPs, &ptl ) == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiLine",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    if (!GpiEndPath(fPs))
      ITHROWGUIERROR2("GpiEndPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);

    // Fill the interior with pattern.
    if (GpiFillPath( fPs, 1L, FPATH_ALTERNATE ) == GPI_HITS && pie.isHitSelectable())
    {
      ((IGPie&)pie).setHitSelected(true);
    }
    // Restore current IGraphicContext's graphic attributes
    // which are not in the graphic bundle.
    setPenColor( currentPenColor );
    setPenPattern( currentPenPattern );
    setCurrentDrawingPosition( rect.center() );
  }

  // If a border is needed.
  if (drawOperation() == IGraphicBundle::frame ||
      drawOperation() == IGraphicBundle::fillAndFrame )
  {
    if ( penWidth() > 1 )    //create a path when pen width > 1.
      if (!GpiBeginPath(fPs, 1L ))
        ITHROWGUIERROR2("GpiBeginPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
    // Draw pie as a partial arc.
    rc = GpiPartialArc( fPs, &ptl, MAKEFIXED(1, 0),
                        DBL2FIXED(pie.startAngle()),
                        DBL2FIXED(pie.sweepAngle()) );
    if ( rc == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiPartialArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If the drawing is for hit detection.
    else if (rc == GPI_HITS && pie.isHitSelectable())
      ((IGPie&)pie).setHitSelected(true); // Set pie to be selected.

    rc = GpiLine( fPs, &ptl ); // Close the pie.
    if ( rc == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiLine",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If the drawing is for hit detection.
    else if (rc == GPI_HITS && pie.isHitSelectable())
      ((IGPie&)pie).setHitSelected(true); // Set pie to be selected.

    if ( penWidth() > 1 )
    {
      if (!GpiEndPath(fPs))
        ITHROWGUIERROR2("GpiEndPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      // Draw the entire path.
      rc = GpiStrokePath(fPs, 1L, 0);
      // If the drawing is for hit detection and pen width is > 1.
      if (rc == GPI_HITS && pie.isHitSelectable())
        ((IGPie&)pie).setHitSelected(true); // Set pie to be selected.
    }
  } /* endif */

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( pie, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw chord.                                                                  |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGChord& chord )
{
  // Chord is represented by its enclosing box.
  IRectangle rect(chord.enclosingRect());
  IGraphicBundle*   currentBundle;
  ITransformMatrix* currentTransform;
  long      rc;

  if (fGCData->fRecoordinate)
  {
    IPoint minXminY(rect.minXMinY());
    IPoint maxXmaxY(rect.maxXMaxY());
    minXminY.setY( fGCData->recoordHeight - minXminY.y() );
    maxXmaxY.setY( fGCData->recoordHeight - maxXmaxY.y() );
    rect = IRectangle( minXminY, maxXmaxY );
  }

  //Set the arc parameter for GPI.
  ARCPARAMS arcp = { rect.rightCenter().x() - rect.center().x(),
                     rect.topCenter().y()   - rect.center().y(), 0,0 };

  // Convert IPoint to POINTL.
  POINTL    ptl( rect.center().asPOINTL() );

  // Apply chord's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( chord, &currentTransform, &currentBundle );

  // Set the GPI arc parameter.
  if (!GpiSetArcParams( fPs, &arcp ))
  {
    ITHROWGUIERROR2("GpiSetArcParams",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // Set the current drawing position.
  setCurrentDrawingPosition( rect.center() );

  // If the interior needs to be filled with a pattern.
  if (drawOperation() == IGraphicBundle::fill ||
      drawOperation() == IGraphicBundle::fillAndFrame )
  {
    // Save current IGraphicContext's graphic attributes
    //which is not in the graphic bundle.
    IColor currentPenColor(penColor());
    unsigned long currentPenPattern(penPattern());
    IGraphicBundle::PenType currentPenType(penType());
    // Move pen from center to the ending edge of the chord
    // without drawing the line.
    setPenType( IGraphicBundle::invisible );
    if ( GpiPartialArc( fPs, &ptl, MAKEFIXED(1, 0),
                        DBL2FIXED(chord.startAngle() + chord.sweepAngle()),
                        (FIXED)0 ) == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiPartialArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // Restore current IGraphicContext's graphic attributes which is not in the graphic bundle.
    setPenType( currentPenType );
    setPenColor( fillColor() );
    setPenPattern( fillPattern() );
    if (!GpiBeginPath(fPs, 1L )) // need to create a path when there is a pattern.
      ITHROWGUIERROR2("GpiBeginPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    // Draw a closed arc based on the previous pen position.
    if ( GpiPartialArc( fPs, &ptl, MAKEFIXED(1, 0),
                        DBL2FIXED(chord.startAngle()),
                        DBL2FIXED(chord.sweepAngle()) ) == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiPartialArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    if (!GpiEndPath(fPs))
      ITHROWGUIERROR2("GpiEndPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    // Fill the chord interior with a pattern.
    rc = GpiFillPath( fPs, 1L, FPATH_ALTERNATE );
    if ( rc == GPI_ERROR )
    {
      ITHROWGUIERROR2("GpiFillPath",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    else if (rc == GPI_HITS && chord.isHitSelectable())
    {
      ((IGChord&)chord).setHitSelected(true);
    }
    // Restore current IGraphicContext's graphic attributes which
    // is not in the graphic bundle.
    setPenColor( currentPenColor );
    setPenPattern( currentPenPattern );
    setCurrentDrawingPosition( rect.center() );
  }
  // Move the pen to certain position without drawing.
  IGraphicBundle::PenType currentPenType(penType());  // Save the current pen type.
  setPenType( IGraphicBundle::invisible );
  if ( GpiPartialArc( fPs, &ptl, MAKEFIXED(1, 0),
                      DBL2FIXED(chord.startAngle() + chord.sweepAngle()),
                      (FIXED)0 ) == GPI_ERROR )
  {
    ITHROWGUIERROR2("GpiPartialArc",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // Restore the pen type.
  setPenType( currentPenType );

  // If the border is needed.
  if (drawOperation() == IGraphicBundle::frame ||
      drawOperation() == IGraphicBundle::fillAndFrame )
  {
    if ( penWidth() > 1 )  // Create a path when pen width > 1.
      if (!GpiBeginPath(fPs, 1L ))
        ITHROWGUIERROR2("GpiBeginPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
    // Draw the chord starting from the previous pen position.
    rc = GpiPartialArc( fPs, &ptl, MAKEFIXED(1, 0),
                        DBL2FIXED(chord.startAngle()),
                        DBL2FIXED(chord.sweepAngle()) );
    if (rc == GPI_ERROR)
    {
      ITHROWGUIERROR2("GpiPartialArc",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
    // If the drawing is for hit detection.
    else if (rc == GPI_HITS && chord.isHitSelectable())
    {
      ((IGChord&)chord).setHitSelected(true);  // Set chord to be hit.
    }
    if ( penWidth() > 1 )
    {
      if (!GpiEndPath(fPs))
        ITHROWGUIERROR2("GpiEndPath",
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
      rc = GpiStrokePath(fPs, 1L, 0); // Draw the entire path.
      // If the drawing is for hit detection.
      if (rc == GPI_HITS && chord.isHitSelectable())
        ((IGLine&)chord).setHitSelected(true); // Set chord to be hit.
    }
  } /* endif */

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( chord, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw text string as graphics on IGraphicContext.                             |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw ( const IGString& graphicString )
{
  CHARBUNDLE        cBundle;
  // Is there a default font set in the IGraphicContext?
  Boolean           defaultFont(!hasFont());
  ITransformMatrix* currentTransform;
  IGraphicBundle*   currentBundle;
  unsigned long     options(0);
  IRectangle        rect( graphicString.clippingRect() );
  RECTL             clippingRect;

  if (fGCData->fRecoordinate)
  {
    IPoint minXminY(rect.minXMinY());
    IPoint maxXmaxY(rect.maxXMaxY());
    minXminY.setY( fGCData->recoordHeight - minXminY.y() );
    maxXmaxY.setY( fGCData->recoordHeight - maxXmaxY.y() );
    rect = IRectangle( minXminY, maxXmaxY );
  }

  // Convert the clipping rectangle.
  clippingRect = rect.asRECTL();

  // Apply graphicString's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( graphicString, &currentTransform, &currentBundle );

  // If a font is set for the graphicString.
  if (graphicString.hasFont())
  {
    // Save the character attributes currently set for this PS
    // This is done because IFont is not doing this (as it should)
    GpiQueryAttrs( fPs, PRIM_CHAR, CBB_COLOR | CBB_BACK_COLOR | CBB_MIX_MODE |
                   CBB_BACK_MIX_MODE | CBB_SET | CBB_MODE | CBB_BOX |
                   CBB_ANGLE | CBB_SHEAR | CBB_DIRECTION | CBB_TEXT_ALIGN |
                   CBB_EXTRA | CBB_BREAK_EXTRA, &cBundle );

    // Set string's font to presentation space: fPs.
    graphicString.font().beginUsingFont(fPs);
  }

  if (graphicString.isClippingRectSet())
  {
    options |= CHS_CLIP;  // Clipping required if there is a clipping rect.
  }

  // Convert position to PM's POINTL.
  POINTL ptl = graphicString.position().asPOINTL();
  if ( fGCData->fRecoordinate )
    ptl.y = fGCData->recoordHeight - ptl.y - graphicString.font().maxCharHeight();

  // Output the string at position ptl.
  long rc(GpiCharStringPosAt( fPs,
                              &ptl,
                              &clippingRect,
                              options,
                              graphicString.text().length(),
                              (PCH)graphicString.text(),
                              0));

  if (rc == GPI_ALTERROR)
  {
    ITHROWGUIERROR2("GpiCharStringAt",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // Is this call used for hit detection or for drawing?
  else if (rc == GPI_HITS && graphicString.isHitSelectable())
  {
    ((IGString&)graphicString).setHitSelected(true);  //hit occurs, not drawing.
  } /* endif */

  // If the graphicString has its own font, stop using the font
  // of the graphic context.
  if (graphicString.hasFont())
  {
    // Disassociate the string's font with fPs.
    graphicString.font().endUsingFont(fPs);

    // Restore the character attributes previously set for this PS
    // This is done because IFont is not doing this (as it should)
    GpiSetAttrs( fPs, PRIM_CHAR, CBB_COLOR | CBB_BACK_COLOR | CBB_MIX_MODE |
                 CBB_BACK_MIX_MODE | CBB_SET | CBB_MODE | CBB_BOX |
                 CBB_ANGLE | CBB_SHEAR | CBB_DIRECTION | CBB_TEXT_ALIGN |
                 CBB_EXTRA | CBB_BREAK_EXTRA, 0, &cBundle );
  }

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( graphicString, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw a list of IGraphics.                                                    |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw( const IGList& list )
{
  IGraphicBundle*  currentBundle;
  ITransformMatrix *currentTransform;

  // Apply list's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( list, &currentTransform, &currentBundle );

  // Create a cursor to iterate through the list.
  IGList::Cursor cursor( list );

  for( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
    IGraphic& graphic(list.graphicAt(cursor));
    graphic.setHitSelected(false); // Reset graphic hit selection.
    // List is hit if any of its components is hit.
    graphic.drawOn(*this);
    // If the drawing is for hit detection.
    if (graphic.isHitSelected() && list.isHitSelectable())
      ((IGList&)list).setHitSelected(true); // Set list to be hit.
  }

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( list, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::draw                                                        |
| Draw region.                                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::draw( const IGRegion& region )
{
  IGraphicBundle*  currentBundle;
  ITransformMatrix *currentTransform;
  long rc;

  // Apply region's transformation matrix and IGraphicBundle to
  // this graphic context.
  apply( region, &currentTransform, &currentBundle );

  // DEFECT 22970: Determine if we're currently accumulating 
  // the bounding rectangle, instead of actually drawing.
  // (In which case, DON'T need to recoordinate.  Will be 
  //  the same size region, no matter what the offset is.)
  Boolean bAccumulatingBoundary(false);
  if ( GpiQueryDrawControl( fPs, DCTL_BOUNDARY ) == DCTL_ON )
      bAccumulatingBoundary = true;

  // If recoordinating, move the region.
  unsigned long regionHeight(0);
  POINTL        ptl = {0,0};
  if ( !bAccumulatingBoundary && fGCData->fRecoordinate )
  {
    IGRegion      tempRegion( region );
    RECTL         regionRect( tempRegion.boundingRect(*this).asRECTL() );

    regionHeight = regionRect.yBottom - regionRect.yTop;
    ptl.y = fGCData->recoordHeight - regionHeight;
    GpiOffsetRegion( this->handle(), (IRegionHandle)region, &ptl );
  }

  // Draw the region.
  rc = GpiPaintRegion( fPs, (IRegionHandle)region );
  if (rc == GPI_ERROR )
  {
    ITHROWGUIERROR2("GpiPaintRegion",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // If the drawing is for hit detection.
  else if (rc == GPI_HITS && region.isHitSelectable())
  {
    ((IGRegion&)region).setHitSelected(true);  // Set region to be hit.
  }

  // Now move the region back to its original location
  if ( !bAccumulatingBoundary && fGCData->fRecoordinate )
  {
    ptl.y = regionHeight - fGCData->recoordHeight;
    GpiOffsetRegion( this->handle(), (IRegionHandle)region, &ptl );
  }

  // Restore this graphic context's own transformation matrix and graphic bundle.
  remove( region, &currentTransform, &currentBundle );

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setRecoordinationHeight                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setRecoordinationHeight( unsigned long height )
{
  fGCData->recoordHeight = height;
  if (!ICoordinateSystem::isConversionNeeded())
  {
    fGCData->fRecoordinate = false;
  }
  else
  {
    fGCData->fRecoordinate = true;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::recoordinationHeight                                        |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::recoordinationHeight( ) const
{
  return fGCData->recoordHeight;
}

/*------------------------------------------------------------------------------
| IGraphicContext::handle                                                      |
| return presentation space handle.                                            |
|                                                                              |
------------------------------------------------------------------------------*/
IPresSpaceHandle IGraphicContext::handle( ) const
{
  return fPs;
}


/*------------------------------------------------------------------------------
| IGraphicContext::currentAttributes                                           |
| return graphic bundle.                                                       |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle IGraphicContext::graphicBundle( ) const
{
  IGraphicBundle bundle(*this);
  return bundle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setGraphicBundle                                            |
| Set the graphic bundle of this device context.                               |
| Set only those attributes set in the IGraphicBundle: bundle                  |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setGraphicBundle(const IGraphicBundle& bundle)
{
  if (bundle.hasPenColor())
    setPenColor(bundle.penColor());
  if (bundle.hasFillColor())
    setFillColor(bundle.fillColor());
  if (bundle.hasBackgroundColor())
    setBackgroundColor(bundle.backgroundColor());
  if (bundle.hasDrawOperation())
    setDrawOperation(bundle.drawOperation());
  if (bundle.hasMixMode())
    setMixMode(bundle.mixMode());
  if (bundle.hasBackgroundMixMode())
    setBackgroundMixMode(bundle.backgroundMixMode());
  if (bundle.hasPenType())
    setPenType(bundle.penType());
  if (bundle.hasPenEndingStyle())
    setPenEndingStyle(penEndingStyle());
  if (bundle.hasPenJoiningStyle())
    setPenJoiningStyle(bundle.penJoiningStyle());
  if (bundle.hasPenWidth())
    setPenWidth(bundle.penWidth());
  if (bundle.hasPenPattern())
    setPenPattern(bundle.penPattern());
  if (bundle.hasFillPattern())
    setFillPattern(bundle.fillPattern());
  if (bundle.hasPatternOrigin())
    setPatternOrigin(bundle.patternOrigin());
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setCurrentDrawingPosition                                   |
| Set the current drawing position of this graphic context.                    |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setCurrentDrawingPosition( const IPoint& point )
{
  // Convert IPoint to POINTL.
  POINTL ptl = point.asPOINTL();
  // Set current drawing position.
  if (!GpiSetCurrentPosition(fPs, &ptl ))
  {
    ITHROWGUIERROR2("GpiSetCurrentPosition",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::currentDrawingPosition                                      |
| Get the current drawing position.                                            |
|                                                                              |
------------------------------------------------------------------------------*/
IPoint IGraphicContext::currentDrawingPosition( ) const
{
  POINTL ptl;
  if (!GpiQueryCurrentPosition(fPs, &ptl ))
  {
    ITHROWGUIERROR2("GpiQueryCurrentPosition",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return IPoint( ptl );
}

/*------------------------------------------------------------------------------
| IGraphicContext::setFont                                                     |
| Set the current font used by the graphic context.                            |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setFont( const IFont& font )
{
  // Need an assignment operator for IFont.
  if ( fFont )
  {
    fFont->endUsingFont( fPs );
    delete fFont;
  }
  fFont = new IFont( font );

  // Set the presentation to begin using font.
  fFont->beginUsingFont( fPs );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::resetFont                                                   |
| Reset the font, i.e. delete the font used currently.                         |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::resetFont( )
{
  if ( fFont )
  {
    // Disassociate a font from the presentation space.
    fFont->endUsingFont( fPs );
    delete fFont;
    fFont = 0;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::currentFont                                                 |
| Get the font currently used.                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IFont IGraphicContext::currentFont( ) const
{
  if ( fFont )
  {
    return IFont( *fFont );
  }
  else
  {
    return IFont( fPs );
  }
}

/*------------------------------------------------------------------------------
| IGraphicContext::hasFont                                                     |
| Test if a font has been set to the graphic context.                          |
| Return false if the default font is used.                                    |
------------------------------------------------------------------------------*/
IBase::Boolean IGraphicContext::hasFont( ) const
{
  // Query the character-set local identifier.
  long lReturn = GpiQueryCharSet( fPs );
  if ( lReturn == LCID_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryCharSet",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  if (lReturn == LCID_DEFAULT) // Default font is used.
    return false;
  else
    return true;
}

/*------------------------------------------------------------------------------
| IGraphicContext::drawOperation                                               |
| Return the draw operation defined in the graphic bundle.                     |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::DrawOperation IGraphicContext::drawOperation() const
{
  return fDrawOp;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setDrawOperation                                            |
| Set the draw operation defined in the graphic bundle.                        |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setDrawOperation(
                                          IGraphicBundle::DrawOperation drawOperation )
{
  fDrawOp = drawOperation;
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::penColor                                                    |
| Get the pen color.                                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IColor IGraphicContext::penColor() const
{
  // Query the current value of the character color attribute.
  long lColor = GpiQueryColor( fPs );
  if ( lColor == CLR_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryColor",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  if ( lColor == CLR_DEFAULT )  // Default pen color.
  {
    return IGraphicContext::defaultPenColor();
  }

  // If not the default color, query the nearest RGB color index
  // in the RGB logical color table on the current associated device.
  long lRGBColor = GpiQueryRGBColor(fPs, 0, lColor);

  if ( lRGBColor == GPI_ALTERROR )
  {
    ITHROWGUIERROR2("GpiQueryRGBColor",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // Convert to a RGB color value structure.
  RGB2* prgb2 = (RGB2*)&lRGBColor;
  // Return the color with the resulting RGB value.
  return IColor(prgb2->bRed, prgb2->bGreen, prgb2->bBlue);
}

/*------------------------------------------------------------------------------
| IGraphicContext::backgroundColor                                             |
| Get the background color.                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IColor IGraphicContext::backgroundColor() const
{
  // Query the background color.
  long lColor = GpiQueryBackColor( fPs );
  if ( lColor == CLR_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryColor",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  if ( lColor == CLR_DEFAULT )  // If default, return the default color.
  {
    return IGraphicContext::defaultBackgroundColor();
  }

  // If non-default color is used, query the nearest RGB color index
  // in the RGB logical color table on the current associated device.
  long lRGBColor = GpiQueryRGBColor(fPs, 0, lColor);
  if ( lRGBColor == GPI_ALTERROR )
  {
    ITHROWGUIERROR2("GpiQueryRGBColor",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

  // Convert the indexed color to RGB color structure.
  RGB2* prgb2 = (RGB2*)&lRGBColor;
  // Return the color with the result RGB value.
  return IColor(prgb2->bRed, prgb2->bGreen, prgb2->bBlue);
}

/*------------------------------------------------------------------------------
| IGraphicContext::fillColor                                                   |
| Return the filling color for closed graphics.                                |
|                                                                              |
------------------------------------------------------------------------------*/
IColor IGraphicContext::fillColor() const
{
  return fFillColor;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setFillColor                                                |
| Set the filling color for closed graphics.                                   |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setFillColor( const IColor& color )
{
  fFillColor = color;
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPenColor                                                 |
| Set the pen color for closed graphics.                                       |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPenColor( const IColor& color )
{
  if (!GpiSetColor( fPs, color.asRGBLong() ))
  {
    ITHROWGUIERROR2("GpiSetColor",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setBackgroundColor                                          |
| Set the background color.                                                    |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setBackgroundColor( const IColor& background )
{
  if (!GpiSetBackColor( fPs, background.asRGBLong() ))
  {
    ITHROWGUIERROR2("GpiSetBackColor",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setMixMode                                                  |
| Set foreground mix mode.                                                     |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setMixMode(IGraphicBundle::MixMode mixMode)
{
  long lMix;
/*------------------------------map the mix mode.-----------------------------------
|
|  or             Resulting color is determined by a bitwise OR of the pen color and the
|                 drawing surface.
|
|  overPaint      Resulting color is determined by pen color.  This is the default for the pen
|                 mix attribute.
|
|  xor            Resulting color is determined by a bitwise XOR of the pen color and the
|                 drawing surface.  This mode enables graphic objects to be drawn so that
|                 they can be removed easily by simply drawing them a second time using the
|                 xor mix mode.  This is useful for graphic animation when an application must
|                 move a graphic object and completely restore the graphic objects that it
|                 originally overlapped.
|
|  leaveAlone     Resulting color is determined by the drawing surface.
|
|  and            Resulting color is determined by a bitwise AND of the pen color and the
|                 drawing surface.
|
|  subtract       Resulting color is determined by inverting the pen color and performing a
|                 bitwise AND with the drawing surface.
|
|  maskSourceNot  Resulting color is determined by inverting the surface color and performing
|                 a bitwise AND with the pen color.
|
|  black          Resulting color is black.
|
|  notMergeSource Resulting color is the inverse of the "or" mix mode.
|
|  notXorSource   Resulting color is the inverse of the "xor" mix mode.
|
|  invert         Resulting color is the inverse of the drawing surface.
|
|  mergeSourceNot Resulting color is determined by a bitwise OR of the pen color and the
|                 inverse color of the drawing surface.
|
|  notCopySource  Resulting color is the inverse of the pen color.
|
|  mergeNotSource Resulting color is determined by the inverse of the pen color and
|                 performing a bitwise AND of the drawing surface.
|
|  notMaskSource  Resulting color is the inverse of the pen mix.
|
|  white          Resulting color is white.
-----------------------------------------------------------------------------------------------*/
  switch (mixMode)
  {
    case IGraphicBundle::or:
      lMix = FM_OR;
    break;
    case IGraphicBundle::overPaint:
      lMix = FM_OVERPAINT;
    break;
    case IGraphicBundle::xor:
      lMix = FM_XOR;
    break;
    case IGraphicBundle::leaveAlone:
      lMix = FM_LEAVEALONE;
    break;
    case IGraphicBundle::and:
      lMix = FM_AND;
    break;
    case IGraphicBundle::subtract:
      lMix = FM_SUBTRACT;
    break;
    case IGraphicBundle::maskSourceNot:
      lMix = FM_MASKSRCNOT;
    break;
    case IGraphicBundle::black:
      lMix = FM_ZERO;
    break;
    case IGraphicBundle::notMergeSource:
      lMix = FM_NOTMERGESRC;
    break;
    case IGraphicBundle::notXorSource:
      lMix = FM_NOTXORSRC;
    break;
    case IGraphicBundle::invert:
      lMix = FM_INVERT;
    break;
    case IGraphicBundle::mergeSourceNot:
      lMix = FM_MERGESRCNOT;
    break;
    case IGraphicBundle::notCopySource:
      lMix = FM_NOTCOPYSRC;
    break;
    case IGraphicBundle::mergeNotSource:
      lMix = FM_MERGENOTSRC;
    break;
    case IGraphicBundle::notMaskSource:
      lMix = FM_NOTMASKSRC;
    break;
    case IGraphicBundle::white:
      lMix = FM_ONE;
    break;
  } /* endswitch */

  // Set the mix mode to the graphic context.
  if (!GpiSetMix( fPs, lMix ))
  {
    ITHROWGUIERROR2("GpiSetMix",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setBackgroundMixMode                                        |
| Set the background mix mode.                                                 |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setBackgroundMixMode(
                            IGraphicBundle::BackgroundMixMode backgroundMixMode)
{
  long lMix;
  switch (backgroundMixMode)
  {
    case IGraphicBundle::backOverPaint:
      lMix = BM_OVERPAINT;
    break;
    case IGraphicBundle::backLeaveAlone:
      lMix = BM_LEAVEALONE;
    break;
    case IGraphicBundle::backOr:
      lMix = BM_OR;
    break;
    case IGraphicBundle::backXor:
      lMix = BM_XOR;
    break;
  } /* endswitch */

  // Set the mix mode to the graphic context.
  if (!GpiSetBackMix( fPs, lMix ))
  {
    ITHROWGUIERROR2("GpiSetBackMix",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::mixMode                                                     |
| Get the foreground mix mode.                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::MixMode IGraphicContext::mixMode( ) const
{
  IGraphicBundle::MixMode mixMode;
  long lMixMode = GpiQueryMix( fPs );
  if ( lMixMode == FM_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryMix",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }

/*------------------------------map the mix mode.-------------------------------------------------
|
|  or             Resulting color is determined by a bitwise OR of the pen color and the
|                 drawing surface.
|
|  overPaint      Resulting color is determined by pen color.  This is the default for the pen
|                 mix attribute.
|
|  xor            Resulting color is determined by a bitwise XOR of the pen color and the
|                 drawing surface.  This mode enables graphic objects to be drawn so that
|                 they can be removed easily by simply drawing them a second time using the
|                 xor mix mode.  This is useful for graphic animation when an application must
|                 move a graphic object and completely restore the graphic objects that it
|                 originally overlapped.
|
|  leaveAlone     Resulting color is determined by the drawing surface.
|
|  and            Resulting color is determined by a bitwise AND of the pen color and the
|                 drawing surface.
|
|  subtract       Resulting color is determined by inverting the pen color and performing a
|                 bitwise AND with the drawing surface.
|
|  maskSourceNot  Resulting color is determined by inverting the surface color and performing
|                 a bitwise AND with the pen color.
|
|  black          Resulting color is black.
|
|  notMergeSource Resulting color is the inverse of the "or" mix mode.
|
|  notXorSource   Resulting color is the inverse of the "xor" mix mode.
|
|  invert         Resulting color is the inverse of the drawing surface.
|
|  mergeSourceNot Resulting color is determined by a bitwise OR of the pen color and the
|                 inverse color of the drawing surface.
|
|  notCopySource  Resulting color is the inverse of the pen color.
|
|  mergeNotSource Resulting color is the determined by the inverse of the pen color and
|                 performing a bitwise AND of the drawing surface.
|
|  notMaskSource  Resulting color is the inverse of the pen mix.
|
|  white          Resulting color is white.
-----------------------------------------------------------------------------------------------*/
  // Map the mix mode.
  switch ( lMixMode )
  {
    case FM_OR:
      mixMode = IGraphicBundle::or;
    break;
    case FM_DEFAULT:
    case FM_OVERPAINT:
      mixMode = IGraphicBundle::overPaint;
    break;
    case FM_XOR:
      mixMode = IGraphicBundle::xor;
    break;
    case FM_LEAVEALONE:
      mixMode = IGraphicBundle::leaveAlone;
    break;
    case FM_AND:
      mixMode = IGraphicBundle::and;
    break;
    case FM_SUBTRACT:
      mixMode = IGraphicBundle::subtract;
    break;
    case FM_MASKSRCNOT:
      mixMode = IGraphicBundle::maskSourceNot;
    break;
    case FM_ZERO:
      mixMode = IGraphicBundle::black;
    break;
    case FM_NOTMERGESRC:
      mixMode = IGraphicBundle::notMergeSource;
    break;
    case FM_NOTXORSRC:
      mixMode = IGraphicBundle::notXorSource;
    break;
    case FM_INVERT:
      mixMode = IGraphicBundle::invert;
    break;
    case FM_MERGESRCNOT:
      mixMode = IGraphicBundle::mergeSourceNot;
    break;
    case FM_NOTCOPYSRC:
      mixMode = IGraphicBundle::notCopySource;
    break;
    case FM_MERGENOTSRC:
      mixMode = IGraphicBundle::mergeNotSource;
    break;
    case FM_NOTMASKSRC:
      mixMode = IGraphicBundle::notMaskSource;
    break;
    case FM_ONE:
      mixMode = IGraphicBundle::white;
    break;
  } /* end switch */
  return mixMode;
}

/*------------------------------------------------------------------------------
| IGraphicContext::backgroundMixMode                                           |
| Get the background mix mode.                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::BackgroundMixMode IGraphicContext::backgroundMixMode( ) const
{
  IGraphicBundle::BackgroundMixMode backgroundMixMode;
  long lMixMode = GpiQueryBackMix( fPs );
  if ( lMixMode == BM_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryMix",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  // Map the mix mode.
  switch ( lMixMode )
  {
    case BM_OVERPAINT:
      backgroundMixMode = IGraphicBundle::backOverPaint;
    break;
    case BM_OR:
      backgroundMixMode = IGraphicBundle::backOr;
    break;
    case BM_XOR:
      backgroundMixMode = IGraphicBundle::backXor;
    break;
    case BM_DEFAULT:
    case BM_LEAVEALONE:
      backgroundMixMode = IGraphicBundle::backLeaveAlone;
    break;
  } /* end switch */
  return backgroundMixMode;
}

/*------------------------------------------------------------------------------
| IGraphicContext::penPattern                                                  |
| Return the current pen pattern.                                              |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::penPattern( ) const
{
  unsigned long pat = GpiQueryPattern( fPs );

  if ((long)pat == PATSYM_ERROR)
  {
    ITHROWGUIERROR2("GpiQueryPattern",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  } /* endif */
  return pat;
}

/*------------------------------------------------------------------------------
| IGraphicContext::fillPattern                                                 |
| Return the current filling pattern.                                          |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::fillPattern( ) const
{
  return fFillPattern;
}

/*------------------------------------------------------------------------------
| IGraphicContext::patternOrigin                                               |
| Return the pattern original where the pattern spreads.                       |
|                                                                              |
------------------------------------------------------------------------------*/
IPoint IGraphicContext::patternOrigin( ) const
{
  POINTL ptl;

  if (!GpiQueryPatternRefPoint( fPs, &ptl ))  // Retrieve the current pattern
  {                                           // Reference point.
    ITHROWGUIERROR2("GpiQueryPatternRefPoint",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return IPoint( ptl );
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPenType                                                  |
| Set pen type.                                                                |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPenType(IGraphicBundle::PenType penType)
{
  long lType;
  switch ( penType )   //the big switch statement is used to map from
  {                    //IGraphicBundle::PenType to PM line type.

/*------------------------------------------------------------------------------
|  solid          An uninterrupted line is drawn.  This is the default pen type.
|
|  dot            A series of dots are drawn.
|
|  shortDash      A series of short dashes are drawn.
|
|  dashDot        A series of dashes followed by a dot is drawn.
|
|  longDash       A series of long dashes are drawn.
|
|  dashDoubleDot  A series of a dash followed by two dots are drawn.
|
|  alternate      Alternate pels are drawn.
|
|  invisible      The pen is not visible.
|------------------------------------------------------------------------------*/

    case IGraphicBundle::solid:
      lType = LINETYPE_SOLID;
      break;
    case IGraphicBundle::dot:
      lType = LINETYPE_DOT;
      break;
    case IGraphicBundle::shortDash:
      lType = LINETYPE_SHORTDASH;
      break;
    case IGraphicBundle::dashDot:
      lType = LINETYPE_DASHDOT;
      break;
    case IGraphicBundle::doubleDot:
      lType = LINETYPE_DOUBLEDOT;
      break;
    case IGraphicBundle::longDash:
      lType = LINETYPE_LONGDASH;
      break;
    case IGraphicBundle::dashDoubleDot:
      lType = LINETYPE_DASHDOUBLEDOT;
      break;
    case IGraphicBundle::alternate:
      lType = LINETYPE_ALTERNATE;
      break;
    case IGraphicBundle::invisible:
      lType = LINETYPE_INVISIBLE;
      break;
  } /* endswitch */

  if (!GpiSetLineType( fPs, lType )) // Set the pen type.
  {
    ITHROWGUIERROR2("GpiSetLineType",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::penType                                                     |
| Return pen type.                                                             |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::PenType IGraphicContext::penType( ) const
{
  IGraphicBundle::PenType pentype;
  long lTemp = GpiQueryLineType( fPs );
  if ( lTemp == LINETYPE_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryLineType",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  switch (lTemp)       //the big switch statement is used to map from
  {                    //PM line type to IGraphicBundle::PenType.
    case LINETYPE_DEFAULT:
    case LINETYPE_SOLID:
      pentype = IGraphicBundle::solid;
      break;
    case LINETYPE_DOT:
      pentype = IGraphicBundle::dot;
      break;
    case LINETYPE_SHORTDASH:
      pentype = IGraphicBundle::shortDash;
      break;
    case LINETYPE_DASHDOT:
      pentype = IGraphicBundle::dashDot;
      break;
    case LINETYPE_DOUBLEDOT:
      pentype = IGraphicBundle::doubleDot;
      break;
    case LINETYPE_LONGDASH:
      pentype = IGraphicBundle::longDash;
      break;
    case LINETYPE_DASHDOUBLEDOT:
      pentype = IGraphicBundle::dashDoubleDot;
      break;
    case LINETYPE_ALTERNATE:
      pentype = IGraphicBundle::alternate;
      break;
    case LINETYPE_INVISIBLE:
      pentype = IGraphicBundle::invisible;
      break;
  } /* endswitch */
  return pentype;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPenWidth                                                 |
| Set pen width which in PM is line width                                      |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPenWidth(unsigned long width )
{
  if (!GpiSetLineWidthGeom( fPs, width ))
  {
    ITHROWGUIERROR2("GpiSetLineWidthGeom",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::penWidth                                                    |
| Return pen width which, in PM, is the line width.                            |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IGraphicContext::penWidth( ) const
{
  unsigned long width = GpiQueryLineWidthGeom( fPs );
  if ((long)width == LINEWIDTHGEOM_ERROR)
  {
    ITHROWGUIERROR2("GpiQueryLineWidthGeom",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return width;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPenEndingStyle                                           |
| Set pen ending style which, in PM, is the line ending style.                 |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPenEndingStyle(
                                 IGraphicBundle::PenEndingStyle penEndingStyle )
{
  long lEnd;
  switch ( penEndingStyle )
  {
    case IGraphicBundle::flat:
      lEnd = LINEEND_FLAT;
      break;
    case IGraphicBundle::square:
      lEnd = LINEEND_SQUARE;
      break;
    case IGraphicBundle::rounded:
      lEnd = LINEEND_ROUND;
      break;
  } /* endswitch */
  if (!GpiSetLineEnd( fPs, lEnd ))
  {
    ITHROWGUIERROR2("GpiSetLineEnd",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::penEndingStyle                                              |
| Get pen ending style which, in PM, is the line ending style.                 |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::PenEndingStyle IGraphicContext::penEndingStyle( ) const
{
  IGraphicBundle::PenEndingStyle penEndingStyle;
  long lEnd = GpiQueryLineEnd( fPs );
  if (lEnd == LINEEND_ERROR)
  {
    ITHROWGUIERROR2("GpiQueryLineEnd",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  switch (lEnd)
  {
    case LINEEND_FLAT:
      penEndingStyle = IGraphicBundle::flat;
      break;
    case LINEEND_SQUARE:
      penEndingStyle = IGraphicBundle::square;
      break;
    case LINEEND_ROUND:
      penEndingStyle = IGraphicBundle::rounded;
      break;
  } /* endswitch */
  return penEndingStyle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPenJoiningStyle                                          |
| Set pen joining style which in PM is line joining style.                     |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPenJoiningStyle(
                                IGraphicBundle::PenJoiningStyle penJoiningStyle)
{
  long lJoin;
  switch ( penJoiningStyle )
  {
    case IGraphicBundle::bevel:
      lJoin = LINEJOIN_BEVEL;
      break;
    case IGraphicBundle::round:
      lJoin = LINEJOIN_ROUND;
      break;
    case IGraphicBundle::miter:
      lJoin = LINEJOIN_MITRE;
      break;
  } /* endswitch */
  if (!GpiSetLineJoin( fPs, lJoin ))
  {
    ITHROWGUIERROR2("GpiSetLineJoin",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::penJoin                                                     |
| Get pen joining style which in PM is line joining style.                     |
|                                                                              |
------------------------------------------------------------------------------*/
IGraphicBundle::PenJoiningStyle IGraphicContext::penJoiningStyle( ) const
{
  IGraphicBundle::PenJoiningStyle penJoiningStyle;
  long lJoin = GpiQueryLineJoin( fPs );
  if (lJoin == LINEJOIN_ERROR)
  {
    ITHROWGUIERROR2("GpiQueryLineJoin",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  switch ( lJoin )
  {
    case LINEJOIN_BEVEL:
      penJoiningStyle = IGraphicBundle::bevel;
      break;
    case LINEJOIN_ROUND:
      penJoiningStyle = IGraphicBundle::round;
      break;
    case LINEJOIN_MITRE:
      penJoiningStyle = IGraphicBundle::miter;
      break;
  } /* endswitch */
  return penJoiningStyle;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPatternOrigin                                            |
| Set pattern origin where pattern spreads.                                    |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPatternOrigin( const IPoint& point )
{
  POINTL ptl = point.asPOINTL();

  if (!GpiSetPatternRefPoint( fPs, &ptl ))  // Set the pattern current reference point.
  {
    ITHROWGUIERROR2("GpiSetPatternRefPoint",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setPenPattern                                               |
| Set pen pattern.                                                             |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setPenPattern( unsigned long penPattern )
{
  // If penPattern is a user-defined pattern
  // like bitmap, character.
  if (penPattern > PATSYM_DIAGHATCH) // PATSYM_DIAGHATCH is dialogal cross hatched,
  {                                  // the next to last pattern in the pattern list.
    if (fGCData->fBitmapId)          // the last one is the blank pattern
    {
      // Delete the old logical font or bitmap tag id.
      GpiDeleteSetId( fPs, fGCData->fBitmapId);
    }
    // Try to get our local Id which is not already in use;
    // Start from 1.
    fGCData->fBitmapId = 1;

    // Have to ensure our local ID is not already in use
    // First, get the total number of local IDs in use.
    long NumLcId = GpiQueryNumberSetIds(fPs);

    // Check if there are any in use.
    if (NumLcId)
    {
      long Types[255]; // 255 is the estimated maximum number of local IDs.
      STR8 Names[255];
      long LcIds[255];

      // Get information about all the local IDs.
      Boolean RC = GpiQuerySetIds(fPs, NumLcId, Types, Names, LcIds);
      if (!RC)
      {
        // Error.
        return *this;
      } /* endif */

      // Check all used IDs to see if ours is already used.
      for (unsigned long LCNum = 0; LCNum < NumLcId ; LCNum++ )
      {
        if (LcIds[LCNum] == fGCData->fBitmapId)
        {
          // Ours is already used, increment to next ID and restart search.
          fGCData->fBitmapId++;
          LCNum = -1;     // Restart the loop. Reset the index here.

          // NewLCId must be in the range 1 to 254.
          if (fGCData->fBitmapId > 254)   // All local IDs are in use.
            {
               ITHROWGUIERROR2("GpiSetPattern local id exhausted",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
            }
        } /* endif */
      } /* endfor */
    } /* endif */

    // Tag a bitmap with a local id so that it can be used as a pattern.
    // Set in GpiSetPatternSet() call.
    GpiSetBitmapId( fPs, (HBITMAP)penPattern, fGCData->fBitmapId);
    // Set the current pattern set to a specific value, here
    // the newly gained local id.
    GpiSetPatternSet( fPs, fGCData->fBitmapId);
    fGCData->fBitmapHandle = (IBitmapHandle)penPattern;
  }
  else   // Use the system-defined pattern set.
  {
    GpiSetPatternSet( fPs, LCID_DEFAULT );
    if (!GpiSetPattern( fPs, penPattern ))
    {
      ITHROWGUIERROR2("GpiSetPattern",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setFillPattern                                              |
| Set filled pattern.                                                          |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setFillPattern( unsigned long fillPattern )
{
  fFillPattern = fillPattern;
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setClipRegion                                               |
| Set clip region using IRegionHandle.                                         |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setClipRegion( const IRegionHandle& region )
{
  HRGN oldClipRgn;

  if ( GpiSetClipRegion( fPs, region, &oldClipRgn ) == RGN_ERROR )
  {
    ITHROWGUIERROR2("GpiSetClipRegion",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setClipRegion                                               |
| Set clip region using IGRegion.                                              |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setClipRegion( const IGRegion& region )
{
  setClipRegion( (IRegionHandle)region );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::clearClipRegion                                             |
| Reset the clip region to the device context, from now on, no clip.           |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::clearClipRegion( )
{
  HRGN oldRgn;

  if ( GpiSetClipRegion( fPs, 0, &oldRgn ) == RGN_ERROR )
  {
    ITHROWGUIERROR2("GpiSetClipRegion",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}


/*------------------------------------------------------------------------------
| IGraphicContext::clipRegion                                                  |
| Return the handle to the current selected clip region.                       |
|                                                                              |
------------------------------------------------------------------------------*/
IRegionHandle IGraphicContext::clipRegion( ) const
{
  IRegionHandle clipRegion;

  clipRegion = GpiQueryClipRegion( fPs );

  if ( clipRegion == HRGN_ERROR )
  {
    ITHROWGUIERROR2("GpiQueryClipRegion",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return IRegionHandle( clipRegion );
}

/*------------------------------------------------------------------------------
| IGraphicContext::setWorldTransformMatrix                                     |
| Set world transformation matrix and its method.                              |
| The world matrix is the matrix currently used by the IGraphicContext.        |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setWorldTransformMatrix(
                                       const ITransformMatrix& transformMatrix,
                                       IGraphic::TransformMethod tm )
{
  MATRIXLF mat(transformMatrix.asMATRIXLF());   // Convert object to PM struct.
  long option;

  if ( tm == IGraphic::replace ) // Copy over the option, i.e. transform method.
    option = TRANSFORM_REPLACE;
  else if ( tm == IGraphic::rightMultiply )
    option = TRANSFORM_ADD;
  else    // Left multiply.
    option = TRANSFORM_PREEMPT;

  if (!GpiSetModelTransformMatrix( fPs, 9L, &mat, option ))  // Set the matrix.
  {
    ITHROWGUIERROR2("GpiSetModelTransformMatrix",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::WorldTransformMatrix                                        |
| Return the world transformation matrix, the transformation matrix used by the|                                      |
| presentation manager.                                                        |
------------------------------------------------------------------------------*/
ITransformMatrix IGraphicContext::worldTransformMatrix( ) const
{
  MATRIXLF mat;
  if (!GpiQueryModelTransformMatrix(fPs, 9L, &mat)) // Retrieve the 3x3(9) element matrix.
  {
    ITHROWGUIERROR2("GpiQueryModelTransformMatrix",
                    IErrorInfo::invalidRequest,
                    IException::recoverable);
  }
  return ITransformMatrix( mat );
}

/*------------------------------------------------------------------------------
| IGraphicContext::startBoundaryAccumulation                                   |
| Accumulate the bounding rectangle of the drawing without actually drawing    |
| the primitives.  Return a reference of this object.                          |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::startBoundaryAccumulation ( )
{
  // Do not display.
  GpiSetDrawControl( fPs, DCTL_DISPLAY, DCTL_OFF );
  // Accumulate boundary.
  GpiSetDrawControl( fPs, DCTL_BOUNDARY, DCTL_ON );
  GpiResetBoundaryData( fPs );     // Reset boundary data to null.
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::stopBoundaryAccumulation                                    |
| Return the accumulated bounding box for the drawing.                         |
|                                                                              |
------------------------------------------------------------------------------*/
IRectangle IGraphicContext::stopBoundaryAccumulation ( )
{
  RECTL rctl;
  GpiQueryBoundaryData( fPs, &rctl );         // Retrieve the bounding rectangle.
  // Restore display.
  GpiSetDrawControl( fPs, DCTL_DISPLAY, DCTL_ON );
  // Stop accumulating boundary.
  GpiSetDrawControl( fPs, DCTL_BOUNDARY, DCTL_OFF );

  if ((rctl.xLeft < rctl.xRight) && (rctl.yBottom < rctl.yTop))
    return IRectangle( rctl );
  else
    return IRectangle();
}

/*------------------------------------------------------------------------------
| IGraphicContext::startHitTesting                                             |
| The following two methods are used as a pair.                                |
| Start hit testing without outputing the drawing.                             |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::startHitTesting( )
{
  // The GpiSetDrawControl sets the options for subsequent drawing operation.
  // DCTL_DISPLAY set to DCTL_OFF means no output to the output medium.
  GpiSetDrawControl( fPs, DCTL_DISPLAY, DCTL_OFF );
  // DCTL_CORRELATE set to DCTL_ON  means a correlation operation will be performed
  // and the return code set if a hit occurs.
  GpiSetDrawControl( fPs, DCTL_CORRELATE, DCTL_ON );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::stopHitTesting                                              |
| Used as a pair with startHitTesting.                                         |
| End hit testing enable drawing. Return a reference of this object.           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::stopHitTesting( )
{
  // Reset output enable and no correlation performed.
  GpiSetDrawControl( fPs, DCTL_DISPLAY, DCTL_ON );
  GpiSetDrawControl( fPs, DCTL_CORRELATE, DCTL_OFF );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setHitPoint                                                 |
| Set hit point for hit detection.                                             |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setHitPoint( const IPoint& point )
{
  // Convert to PM data structure.
  POINTL pickPoint(point.asPOINTL());
  // Convert coordinates from device to page space.
  GpiConvert( fPs, CVTC_DEVICE, CVTC_PAGE, (LONG)1, &pickPoint );
  GpiSetPickAperturePosition( fPs, &pickPoint);
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::setHitApertureSize                                          |
| Set hit aperture size with hitSize.                                          |
| Return a reference of this object.                                           |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::setHitApertureSize( const ISize& hitSize )
{
  // Convert to a PM data structure.
  SIZEL pickSize(hitSize.asSIZEL());
  GpiSetPickApertureSize( fPs, PICKAP_REC, &pickSize );
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::hitPoint                                                    |
| Return the hit point used for hit detection.                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IPoint IGraphicContext::hitPoint( ) const
{
  POINTL pickPoint;
  GpiQueryPickAperturePosition( fPs, &pickPoint );
  return IPoint( pickPoint );
}

/*------------------------------------------------------------------------------
| IGraphicContext::hitApertureSize                                             |
| Return the hit aperture size, an ISize object.                               |
|                                                                              |
------------------------------------------------------------------------------*/
ISize IGraphicContext::hitApertureSize( ) const
{
  SIZEL pickSize;
  GpiQueryPickApertureSize( fPs, &pickSize );
  return ISize( pickSize );
}

/*------------------------------------------------------------------------------
| IGraphicContext::apply                                                       |
| Preserve the current ITransformMatrix and IGraphicBundle of IGraphicContext, |
| apply the ITransformMatrix and IGraphicBundle of graphic to IGraphicContext. |
| If the graphic already has an ITransformMatrix and IGraphicBundle,           |
| make a temporary copy to *transformMatrix and *graphicBundle for drawing.    |
|-----------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::apply( const IGraphic&    graphic,
                                         ITransformMatrix** transformMatrix,
                                         IGraphicBundle**   graphicBundle )
{
  if (graphic.hasTransformMatrix() ||
      graphic.transformMethod() == IGraphic::replace)
  {
    *transformMatrix = new ITransformMatrix(worldTransformMatrix());
    setWorldTransformMatrix(graphic.transformMatrix(),
                            graphic.transformMethod());
  }
  if (graphic.hasGraphicBundle())
  {
    *graphicBundle = new IGraphicBundle(*this);
    setGraphicBundle( graphic.graphicBundle() );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicContext::remove                                                      |
| Restore back the old ITransformMatrix and IGraphicBundleContext to           |
| the IGraphicContext.                                                         |
------------------------------------------------------------------------------*/
IGraphicContext& IGraphicContext::remove( const IGraphic&    graphic,
                                          ITransformMatrix** transformMatrix,
                                          IGraphicBundle**   graphicBundle )
{
  if (graphic.hasGraphicBundle())
  {
    setGraphicBundle(**graphicBundle);
    delete *graphicBundle;
  }
  if (graphic.hasTransformMatrix() ||
      graphic.transformMethod() == IGraphic::replace)
  {
    setWorldTransformMatrix(**transformMatrix, IGraphic::replace );
    delete *transformMatrix;
  }
  return *this;
}

