/*******************************************************************************
* FILE NAME: ilistbas.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ilistbas.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C"
{
  #define INCL_WINLISTBOXES      // LIT_xxx, LM_xxx, LS_xxx
  #define INCL_WINSYS            // SV_xxx, PP_xxx
  #include <iwindefs.h>
}

#include <inotifev.hpp>
#include <ilistbas.hpp>
#include <istring.hpp>
#include <ireslib.hpp>
#include <ihandle.hpp>
#include <iexcept.hpp>
#include <icconst.h>
#include <itrace.hpp>
#include <icolor.hpp>
#include <ifont.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ILISTBAS_CPP_
  #include <ipagetun.h>
#endif

/***************************************************************/
/* Public list box styles.                                     */
/***************************************************************/
#pragma data_seg(ICLStaticConst)
const IBaseListBox::Style
  IBaseListBox::multipleSelect    = LS_MULTIPLESEL,            // 0x00000001
  IBaseListBox::drawItem          = LS_OWNERDRAW,              // 0x00000002
  IBaseListBox::noAdjustPosition  = LS_NOADJUSTPOS,            // 0x00000004
  IBaseListBox::horizontalScroll  = LS_HORZSCROLL,             // 0x00000008
  IBaseListBox::extendedSelect    = LS_EXTENDEDSEL,            // 0x00000010
  IBaseListBox::border3D          ( 0, IWS_BORDER3D ),
  IBaseListBox::classDefaultStyle ( LS_HORZSCROLL  |
                                LS_NOADJUSTPOS |
                                WS_VISIBLE,
                                0 );

const unsigned long
  IBaseListBox::notFound = (unsigned long)ULONG_MAX,
  IBaseListBox::first    = (unsigned long)ULONG_MAX;
#pragma data_seg()

/***************************************************************/
/* Default style for new objects (initial value)               */
/***************************************************************/
#pragma data_seg(ICLNonConst)
  IBaseListBox::Style IBaseListBox::currentDefaultStyle ( LS_HORZSCROLL  |
                                                  LS_NOADJUSTPOS |
                                                  WS_VISIBLE,
                                                  0 );
#pragma data_seg()

class IBaseListBoxData : public IBase {
/*******************************************************************************
* The IBaseListBoxData class encapsulates private data and functions used by the   *
* IBaseListBox class.  An object of this class is created in the IBaseListBox          *
* constructors.                                                                *
*******************************************************************************/
public:
  IBaseListBoxData ( )
    : minimumRows( 4 ),
      minimumCharacters( 25 )
  { }
virtual
 ~IBaseListBoxData ( )
  { }

unsigned long
  minimumRows,
  minimumCharacters;

private:
  IBaseListBoxData ( const IBaseListBoxData& );
IBaseListBoxData
 &operator=    ( const IBaseListBoxData& );
};

/*------------------------------------------------------------------------------
| IBaseListBox::IBaseListBox                                                           |
|                                                                              |
| Construct an IBaseListBox on an IWindow.                                         |
------------------------------------------------------------------------------*/
IBaseListBox::IBaseListBox ( unsigned long     Id,
                     IWindow*          pParent,
                     IWindow*          pOwner,
                     const IRectangle& rectInit,
                     const Style&      style )
  : fBaseListBoxData(new IBaseListBoxData)
{
  // assertions on input parms
  IASSERTPARM(pParent!=0);

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

#ifdef __BORLANDC__
  IWindowHandle owner;
  if (pOwner == 0)
    owner = IWindowHandle(0);
  else
    owner = pOwner->handle();
#else
  IWindowHandle owner = (pOwner == 0) ? IWindowHandle(0) : pOwner->handle();
#endif

  IWindowHandle whListBox =
      this -> create( Id,
                      0,
                      convertToGUIStyle( style ),
                      WC_LISTBOX,
                      pParent->handle(),
                      owner,
                      rectInit,
                      0,
                      0 );

   startHandlingEventsFor(whListBox);
}

/*------------------------------------------------------------------------------
| IBaseListBox::IBaseListBox                                                           |
|                                                                              |
| Construct an IBaseListBox on a dialog.                                           |
------------------------------------------------------------------------------*/
IBaseListBox::IBaseListBox ( unsigned long id,
                     IWindow* parent )
  : fBaseListBoxData (new IBaseListBoxData)
{
  setAutoDestroyWindow(false);
  startHandlingEventsFor(id, parent);
}

/*------------------------------------------------------------------------------
| IBaseListBox::IBaseListBox                                                           |
|                                                                              |
| Wrapper an existing ListBox.                                                 |
------------------------------------------------------------------------------*/
IBaseListBox::IBaseListBox ( const IWindowHandle& handle )
  : fBaseListBoxData (new IBaseListBoxData)
{
   setAutoDestroyWindow(false);
   startHandlingEventsFor(handle);
}

/*------------------------------------------------------------------------------
| IBaseListBox::~IBaseListBox                                                          |
|                                                                              |
| Default destructor here for page tuning.                                     |
------------------------------------------------------------------------------*/
IBaseListBox::~IBaseListBox()
{
  if ( fBaseListBoxData != 0 )
    delete fBaseListBoxData;
}

/*------------------------------------------------------------------------------
| IBaseListBox::isMultipleSelect                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
Boolean IBaseListBox::isMultipleSelect() const
{
  return( (style() & multipleSelect.asUnsignedLong()) ? true : false );
}

/*------------------------------------------------------------------------------
| IBaseListBox::isExtendedSelect                                                   |
------------------------------------------------------------------------------*/
Boolean IBaseListBox::isExtendedSelect()  const
{
  return( (style() & extendedSelect.asUnsignedLong()) ? true : false );
}

/*------------------------------------------------------------------------------
| IBaseListBox::isDrawItem                                                         |
------------------------------------------------------------------------------*/
Boolean IBaseListBox::isDrawItem()  const
{
  return( (style() & drawItem.asUnsignedLong()) ? true : false );
}

/*------------------------------------------------------------------------------
| IBaseListBox::isNoAdjustPosition                                                 |
------------------------------------------------------------------------------*/
Boolean IBaseListBox::isNoAdjustPosition()  const
{
  return( (style() & noAdjustPosition.asUnsignedLong()) ? true : false );
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableMultipleSelect                                               |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableMultipleSelect ( Boolean enable )
{
#ifndef IC_WIN
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);
  Boolean       refreshListBox(false);

  if (enable)
  {
    ulStyle |= multipleSelect.asUnsignedLong();
  }
  else
  {
    // Since extended and multiple select can be set at the
    // same time, do not deselect all items if extended select
    // is still set.
    if (!(ulStyle & extendedSelect.asUnsignedLong()))
    {
      deselectAll();
      refreshListBox = true;
    }
    ulStyle &= ~multipleSelect.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    if (refreshListBox)
      refresh();
  }
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableMultipleSelect                                              |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableMultipleSelect()
{
  enableMultipleSelect(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableExtendedSelect                                               |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableExtendedSelect ( Boolean enable )
{
#ifndef IC_WIN
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);
  Boolean       refreshListBox(false);

  if (enable)
  {
    ulStyle |= extendedSelect.asUnsignedLong();
  }
  else
  {
    // Since extended and multiple select can be set at the
    // same time, do not deselect all items if multiple select
    // is still set.
    if (!(ulStyle & multipleSelect.asUnsignedLong()))
    {
      deselectAll();
      refreshListBox = true;
    }
    ulStyle &= ~extendedSelect.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    if (refreshListBox)
      refresh();
  }
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableExtendedSelect                                              |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableExtendedSelect()
{
  enableExtendedSelect(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableDrawItem                                                     |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableDrawItem ( Boolean enable )
{
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);

  if (enable)
  {
    ulStyle |= drawItem.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~drawItem.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);

    if ( enable )
    {         // Force draw-item handlers to be called.
       IWindow* owner = this->owner();
       if (owner)
       {
          owner->sendEvent( WM_MEASUREITEM,
                            IEventParameter1( this->id() ),
                            0 );
       }
    }

    refresh();
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableDrawItem                                                    |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableDrawItem()
{
  enableDrawItem(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableNoAdjustPosition                                             |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableNoAdjustPosition ( Boolean enable )
{
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);

  if (enable)
  {
    ulStyle |= noAdjustPosition.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~noAdjustPosition.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableNoAdjustPosition                                            |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableNoAdjustPosition()
{
  enableNoAdjustPosition(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::Style IBaseListBox::defaultStyle                                       |
------------------------------------------------------------------------------*/
IBaseListBox::Style IBaseListBox::defaultStyle()
{
   return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setDefaultStyle                                                    |
------------------------------------------------------------------------------*/
void IBaseListBox::setDefaultStyle(const Style& style)
{
  currentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IBaseListBox::convertToGUIStyle                                                  |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IBaseListBox::convertToGUIStyle(const IBitFlag& guiStyle,
                                          Boolean bExtOnly) const
{
  // Obtain the style from the class (IControl) that we inherit from
  unsigned long ulStyle( Inherited::convertToGUIStyle( guiStyle, bExtOnly ));

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= extendedStyle() & IS_EXTMASK;
  }
  else
  {
    // LS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= guiStyle.asUnsignedLong() & ILS_MASK;

#ifdef IC_WIN
    ulStyle |= WS_CHILD | LBS_NOTIFY | WS_BORDER | LBS_HASSTRINGS | WS_VSCROLL;
#endif
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| IBaseListBox::selectAll                                                          |
|                                                                              |
| Sequentially select all items in listbox.                                    |
------------------------------------------------------------------------------*/
IBaseListBox&  IBaseListBox::selectAll()
{
   //----------------------------------------------------------------
   // Defect 28100: Don't thrown exception on empty listbox.
   //----------------------------------------------------------------
   // Verify that the list contains items and if not, fail call
//   if (isEmpty())
//     ITHROWLIBRARYERROR( IC_EMPTY_LIST,
//                         IErrorInfo::invalidRequest,
//                         IException::recoverable );

   // Verify that the list supports selection of multiple items
   // i.e. it is not a single select list box
   if ( isMultipleSelect() || isExtendedSelect() )
   {
     unsigned long
       index,
       lastIndex;
     for (index = 0, lastIndex = count(); index < lastIndex; index++)
     {
       IBaseListBox::select(index);
     }
   }
   else
     ITHROWLIBRARYERROR( IC_SINGLE_SELECT_LIST,
                         IErrorInfo::invalidRequest,
                         IException::recoverable );

   return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setItemHeight                                                      |
|                                                                              |
| Set height of the items in a listbox.                                        |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::setItemHeight(unsigned long lNewHeight)
{
   IEventResult evt = handle().sendEvent(LM_SETITEMHEIGHT,
                              IEventParameter1((unsigned long)lNewHeight),
                              IEventParameter2(0));
   if (!(evt.asUnsignedLong()))
      ITHROWGUIERROR("LM_SETITEMHEIGHT");
   return *this;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IBaseListBox::itemHeight                                                         |
|                                                                              |
| Queries the item height in a listbox.                                        |
------------------------------------------------------------------------------*/
unsigned long IBaseListBox::itemHeight () const
{
  IEventResult evt = handle().sendEvent(LB_GETITEMHEIGHT,
                                        IEventParameter1(0),
                                        IEventParameter2(0));
  if (evt.asLong() == LB_ERR)
    ITHROWSYSTEMERROR( LB_ERR, "LB_GETITEMHEIGHT", IErrorInfo::accessError,
                       IException::recoverable );
  return evt.asUnsignedLong();
}
#endif

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IBaseListBox::backgroundColor                                                    |
|                                                                              |
| Returns the background color of the IBaseListBox.                                |
------------------------------------------------------------------------------*/
IColor IBaseListBox::backgroundColor () const
{
  return (IWindow::color(PP_BACKGROUNDCOLOR,
                         IGUIColor(IGUIColor::listBoxBgnd)));
}
#endif
#ifdef IC_NOTYET
  Will this color model work with Windows???  No!
#endif

/*------------------------------------------------------------------------------
| IBaseListBox::calcMinimumSize                                                    |
|                                                                              |
| Calculate the minimum size needed by the control.                            |
------------------------------------------------------------------------------*/
ISize  IBaseListBox :: calcMinimumSize() const
{
  ISize sizMin;
#ifndef IC_WIN
  IFont fnt(this);

  sizMin.setHeight ( fBaseListBoxData->minimumRows * fnt.maxCharHeight() +
                     2 * IQUERYSYSVALUE(SV_CYBORDER) + 2);

  sizMin.setWidth  ( fBaseListBoxData->minimumCharacters * fnt.avgCharWidth() +
                     2 * IQUERYSYSVALUE(SV_CXBORDER) +
                     IQUERYSYSVALUE(SV_CXVSCROLL));

  if (isHorizontalScroll())
     sizMin.setHeight (sizMin.height() + IQUERYSYSVALUE(SV_CYHSCROLL));
#endif
#ifdef IC_NOTYET
// Need to update for Windows NT
#endif
  return sizMin;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setLayoutDistorted                                                 |
|                                                                              |
| Cause a font change to result in a new minimum size.                         |
------------------------------------------------------------------------------*/
IBaseListBox&
  IBaseListBox::setLayoutDistorted ( unsigned long layoutAttributesOn,
                                     unsigned long layoutAttributesOff )
{
  unsigned long flagsOn = layoutAttributesOn;
  if ( layoutAttributesOn & IWindow::fontChanged )
  {                // New font means a new minimum size.
     flagsOn |= IWindow::minimumSizeChanged;
  }
  Inherited::setLayoutDistorted( flagsOn, layoutAttributesOff );
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setMinimumCharacters                                               |
|                                                                              |
| Sets the number of characters in an item of a minimum size listbox.          |
------------------------------------------------------------------------------*/
IBaseListBox&  IBaseListBox :: setMinimumCharacters (unsigned long minimumCharacters)
{
  fBaseListBoxData->minimumCharacters = minimumCharacters;
  setLayoutDistorted (minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::minimumCharacters                                                  |
|                                                                              |
| Returns the number of characters in an item in a minimum size listbox.       |
------------------------------------------------------------------------------*/
unsigned long  IBaseListBox ::  minimumCharacters() const
{
  return fBaseListBoxData->minimumCharacters;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setMinimumRows                                                     |
|                                                                              |
| Sets the number of rows in a minimum size listbox.                           |
------------------------------------------------------------------------------*/
IBaseListBox&  IBaseListBox :: setMinimumRows (unsigned long minimumRows)
{
  fBaseListBoxData->minimumRows = minimumRows;
  setLayoutDistorted (minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::minimumRows                                                        |
|                                                                              |
| Returns the number of rows in a minimum size listbox.                        |
------------------------------------------------------------------------------*/
unsigned long  IBaseListBox :: minimumRows ( ) const
{
  return fBaseListBoxData->minimumRows;
}


#include <ilistbx3.hpp>
