/*******************************************************************************
* FILE NAME: ithread.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IThread                                                                  *
*     IThread::Cursor                                                          *
*     IThreadFn                                                                *
*     IThreadMemberFn                                                          *
*     ICurrentThread                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C" {
  #include <stdlib.h>
  #define INCL_DOSPROCESS
  #define INCL_DOSSESMGR
  #define INCL_DOSERRORS
  #define INCL_WINMESSAGEMGR
  #define INCL_WINFRAMEMGR
  #define INCL_WINHOOKS
  #include <iwindefs.h>
}

#include <ithread.hpp>

#ifndef _ICCONST_
  #include <icconst.h>
#endif
#ifndef _ICRITSEC_
  #include <icritsec.hpp>
#endif
#ifndef _IEXCEPT_
  #include <iexcept.hpp>
#endif
#ifndef _IRESLOCK_
  #include <ireslock.hpp>
#endif
#ifndef _ISTRING_
  #include <istring.hpp>
#endif
#ifndef _ITRACE_
  #include <itrace.hpp>
#endif
#ifndef _IWINLSTS_
  #include <iwinlsts.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ITHREAD_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_WIN          // Needs checkout - DAB
  #include <malloc.h>  // for stackavail()
#endif


/*------------------------------------------------------------------------------
| Objects of the class IStartThread maintain information about started threads.|
| Instances of IStartedThread are managed by class IThread.                    |
------------------------------------------------------------------------------*/
class IStartedThread : public IRefCounted {
typedef IRefCounted
  Inherited;
public:
  IStartedThread ( const IThreadId &tid = 0 );
 ~IStartedThread ( );

IString
  asDebugInfo ( ) const;

Boolean
  isStarted ( );

void
  end ( );

tib_t
 *tib ( );

IString
  variable ( const IString &key ) const;

IStartedThread
 &setVariable ( const IString &key,
                const IString &value );

static void
  add        ( IStartedThread &aThread ),
  remove     ( IStartedThread &aThread ),
  initThread ( IStartedThread *pThread );

static IStartedThread
 *locate ( const IThreadId &tid );

static Boolean
  isTIDValid ( const IThreadId &tid );

static void
  flushCache ( );

static IStartedThread
 *checkCacheFor ( const IThreadId &tid );

static void
  cache ( const IThreadId &tid,
          IStartedThread  *pStarted );

static void
  watch ( );

static IString
  classDebugInfo ( );

IThreadId
  id;            // TID.

IMessageQueueHandle
  hmq;           // HMQ.

IAnchorBlockHandle
  hab;           // HAB.

Boolean
  autoInitPM,    // Auto-init PM flag.
  started,       // Started flag.
  startedHere,   // Started-by-initThread flag.
  pmStartedHere, // PM-initialized-by-IThread flag.
  inArray;       // In-threads-array flag.

long
  queueSize;     // PM message queue size.

unsigned long
  stkSize,       // Stack size.
  fCount,        // Creation order.
  inMsgLoop;     // In-msg-polling-loop count (increment when polling loop
                 // begins and decrement when polling loop ends to effectively
                 // track threads that are calling processMsgs(), and issue
                 // the quit message to each thread when it terminates.

IApplication::PriorityClass
  priorityClass; // Priority class.

int
  priorityLevel; // Priority level.

IThreadFn
 *fnObj;         // Function object reference.

unsigned
  slot;          // Slot in threads-array.

IWindowList
 *windowList;    // Pointer to info list.

tib_t
 *pTIB;          // Pointer to TIB structure.

enum { varSize = 16 };

unsigned
  numVars;

// Class for storage of data that is associated with the started thread.
// The data is stored as IStrings and can be set and retrieved through the
// functions setVariable and variable of IStartedThread and IThread.
struct Assoc
  {
  Assoc ( );
 ~Assoc ( );
  IString
    key,
    value;
  } vars[ varSize ];

static IStartedThread
 **threads;      // Array of IStartedThread pointers.

static unsigned
  arraySize;     // Size of threads array.

static IStartedThread
 *cachedThread;  // Cached ptr.

static IThreadId
  cachedTID;     // Cached id.

static Boolean
  watching;      // Watching for non-IThread terminations.

static unsigned long
  fStartedThreadCount,
  cacheHits,
  cacheMisses;

static IPrivateResource
 *pResource,
 &lock ( );
}; // Class IStartedThread

/*------------------------------------------------------------------------------
                        DLL termination thread cleanup
------------------------------------------------------------------------------*/
//defect: 9868 start
class IThreadWatcher
{
    public:
    static TID tid;
    static void threadWatcher_Terminate();
    IThreadWatcher(){
        ITRACE_DEVELOP( "Threadwatcher constructor" );
    }
    ~IThreadWatcher(){
        ITRACE_DEVELOP( "Threadwatcher destructor" );
        IThreadWatcher::threadWatcher_Terminate();
     }
};
void IThreadWatcher::threadWatcher_Terminate()
{
    ITRACE_DEVELOP( "Threadwatcher terminating" );
    if( IStartedThread::watching == true )
    {
        ITRACE_DEVELOP( IString("killing thread:") + IString(tid) );
#ifdef IC_PM
        DosKillThread( IThreadWatcher::tid );
#endif
        IStartedThread::watching = false;
    }
}
extern "C" void threadWatcher_Terminate(void)
{
    IThreadWatcher::threadWatcher_Terminate();
}
IThreadWatcher iThreadWatcher;
TID IThreadWatcher::tid = 0;
//defect 9868 stop

/*------------------------------------------------------------------------------
                        S T A T I C    O B J E C T S
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
unsigned
  IStartedThread::arraySize = 0;

unsigned long
  IThread::dfltStackSize = 32768;

Boolean
  IThread::dfltAutoInitGUI = -1;

long
  IThread::dfltQueueSize = 30;

ICurrentThread
 *IThread::pCurrent = 0;

IStartedThread
 **IStartedThread::threads = 0;

IStartedThread
 *IStartedThread::cachedThread = 0;

Boolean
  IStartedThread::watching = false;

IThreadId
  IStartedThread::cachedTID = 0;

unsigned long
  IStartedThread::fStartedThreadCount = 0,
  IStartedThread::cacheHits   = 0,
  IStartedThread::cacheMisses = 0;

IPrivateResource
 *IStartedThread::pResource = 0;

/*------------------------------------------------------------------------------
| IThread::Cursor::timeStamp - Incremented each time a thread is started or    |
|                              stopped.  A cursor is valid if its created data |
|                              member is equal to this value.                  |
------------------------------------------------------------------------------*/
unsigned
  IThread::Cursor::timeStamp = 1;
#pragma data_seg()

/*------------------------------------------------------------------------------
| IStartedThread::IStartedThread                                               |
|                                                                              |
| Initialize all the members.  If a thread ID is given, it is used.            |
------------------------------------------------------------------------------*/
IStartedThread :: IStartedThread ( const IThreadId &tid )
  : id            ( tid ),
    hmq           ( 0 ),
    hab           ( 0 ),
    inArray       ( false ),
    started       ( ( tid.asUnsigned() ) ? true : false ),
    startedHere   ( false ),
    pmStartedHere ( false ),
    slot          ( 0 ),
    pTIB          ( 0 ),
    autoInitPM    ( IThread::defaultAutoInitGUI() ),
    queueSize     ( IThread::defaultQueueSize() ),
    stkSize       ( IThread::defaultStackSize() ),
    fCount        ( ++IStartedThread::fStartedThreadCount ),
    inMsgLoop     ( 0 ),
    priorityClass ( IApplication::noChange ),
    priorityLevel ( -1 ),
    fnObj         ( 0 ),
    windowList    ( 0 ),
    numVars       ( 0 )
  {
  IMODTRACE_DEVELOP( IString( "IStartedThread::IStartedThread(" ) +
                     IString( tid.asUnsigned() ) +
                     IString( ')' ) );
  }

/*------------------------------------------------------------------------------
| IStartedThread::Assoc::Assoc                                                 |
|                                                                              |
| Constructor for Assoc class. This class allows storing data associated with  |
| the started thread.                                                          |
|                                                                              |
| Empty ctor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
IStartedThread::Assoc :: Assoc ( )
  {
  }

/*------------------------------------------------------------------------------
| IStartedThread::Assoc::~Assoc                                                |
|                                                                              |
| Empty dtor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
IStartedThread::Assoc :: ~Assoc ( )
  {
  }

/*------------------------------------------------------------------------------
| IStartedThread::~IStartedThread                                              |
|                                                                              |
| Remove this object from the static array.                                    |
------------------------------------------------------------------------------*/
IStartedThread :: ~IStartedThread ( )
  {
  IMODTRACE_DEVELOP( IString( "IStartedThread::~IStartedThread(" ) +
                     IString( this->id.asUnsigned() ) +
                     IString( ')' ) );
  delete windowList;
  }

/*------------------------------------------------------------------------------
| IStartedThread::isStarted                                                    |
|                                                                              |
| Return whether the thread represented by this object is currently started.   |
|                                                                              |
| If the "started here" flag is set, then the thread was started via           |
| IStartedThread::initThread and we know it hasn't ended.                      |
|                                                                              |
| Otherwise, we check the thread ID via a call to DosWaitThread and if         |
| not OK, return false (and set the "started" flag appropriately).             |
------------------------------------------------------------------------------*/
Boolean IStartedThread :: isStarted ( )
  {
  Boolean result = true;
  if ( !this->startedHere )
    if ( this->started )
      // Verify it, though!
      result = this->started = IStartedThread::isTIDValid( this->id );
    else
      result = false;
  return result;
  }

/*------------------------------------------------------------------------------
| IStartedThread::isTIDValid                                                   |
|                                                                              |
| Return whether the thread represented by this object is currently started.   |
------------------------------------------------------------------------------*/
Boolean IStartedThread :: isTIDValid ( const IThreadId &id )
  {
  IThreadId tid = id;

  Boolean result = ( tid.asUnsigned() == 1
                     ||
                     tid == IThread::currentId()
                     ||
#ifdef IC_PM
                     DosWaitThread( (PTID)&tid,
                                    DCWW_NOWAIT ) ==
                       ERROR_THREAD_NOT_TERMINATED );
#endif
#ifdef IC_WIN
                     WaitForSingleObject((HANDLE)id.asUnsigned(),0)==WAIT_TIMEOUT);
#endif
  return result;
  }

/*------------------------------------------------------------------------------
| IStartedThread::asDebugInfo                                                  |
------------------------------------------------------------------------------*/
IString IStartedThread :: asDebugInfo ( ) const
  {
  IResourceLock
    lock( IStartedThread::lock() );
  IString result( "IStartedThread(" );
  result += this->Inherited::asDebugInfo();
  result += ",useCount=";
  result += IString( this->useCount() );
  result += ",id=";
  result += this->id.asDebugInfo();
  result += ",hmq=";
  result += this->hmq.asDebugInfo();
  result += ",hab=";
  result += this->hab.asDebugInfo();
  result += ",inArray=";
  result += IString( this->inArray );
  result += ",started=";
  result += IString( this->started );
  result += ",startedHere=";
  result += IString( this->startedHere );
  result += ",pmStartedHere=";
  result += IString( this->pmStartedHere );
  result += ",autoInitPM=";
  result += IString( this->autoInitPM );
  result += ",slot=";
  result += IString( this->slot );
  result += ",pTIB=";
  result += IString( (unsigned long) this->pTIB );
  result += ",queueSize=";
  result += IString( this->queueSize );
  result += ",stkSize=";
  result += IString( this->stkSize );
  result += ",fCount=";
  result += IString( this->fCount );
  result += ",priorityClass=";
  result += IString( (int) this->priorityClass );
  result += ",priorityLevel=";
  result += IString( this->priorityLevel );
  result += ",fnObj=";
  result += IString( (unsigned long) this->fnObj );
  result += ",windowList=";
  result += IString( (unsigned long) this->windowList );
  result += ")";
  return result;
  }

/*------------------------------------------------------------------------------
| IStartedThread::classDebugInfo                                               |
------------------------------------------------------------------------------*/
IString IStartedThread :: classDebugInfo ( )
  {
  IResourceLock
    lock( IStartedThread::lock() );
  IString result( "IStartedThread::(cacheHits=" );
  result += IString( IStartedThread::cacheHits );
  result += ",cacheMisses=";
  result += IString( IStartedThread::cacheMisses );
  result += ",arraySize=";
  result += IString( IStartedThread::arraySize );
  result += ",threads=";
  result += IString( (unsigned long) IStartedThread::threads );
  result += "->[";
  IString comma;
  for ( unsigned i = 0;
        i < IStartedThread::arraySize;
        i++ )
    {
    result += comma;
    comma = ",";
    if ( IStartedThread::threads[i] )
      result += IStartedThread::threads[i]->asDebugInfo();
    else
      result += "0";
    }
  result += "])";
  return result;
  }

/*------------------------------------------------------------------------------
| IStartedThread::flushCache                                                   |
|                                                                              |
| Reset cache.                                                                 |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources protected.                                |
------------------------------------------------------------------------------*/
void IStartedThread :: flushCache ( )
  {
  ICritSec
    lock;
  IStartedThread::cachedTID    = 0;
  IStartedThread::cachedThread = 0;
  IThread::Cursor::timeStamp++; // Invalidate any existing cursors.
  }

/*------------------------------------------------------------------------------
| IStartedThread::cache                                                        |
|                                                                              |
| Set cached thread id/ptr.                                                    |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources protected.                                |
------------------------------------------------------------------------------*/
inline void IStartedThread :: cache ( const IThreadId &tid,
                                      IStartedThread  *thread )
  {
  IStartedThread::cachedTID    = tid;
  IStartedThread::cachedThread = thread;
  }

/*------------------------------------------------------------------------------
| IStartedThread::checkCacheFor                                                |
|                                                                              |
| Check cached thread id/ptr.                                                  |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources protected.                                |
------------------------------------------------------------------------------*/
inline IStartedThread *IStartedThread :: checkCacheFor ( const IThreadId &tid )
  {
  IStartedThread *result = 0;
  if ( tid == IStartedThread::cachedTID )
    {
    result = IStartedThread::cachedThread;
    IStartedThread::cacheHits++;
    }
  else
    IStartedThread::cacheMisses++;
  return result;
  }

/*------------------------------------------------------------------------------
| IStartedThread::add                                                          |
|                                                                              |
| Search the static array for a free slot and add the argument IStartedThread  |
| object there.  If there is no room, allocate a larger array.                 |                                             |
|                                                                              |
| Notes:                                                                       |
|   1. Access to the static data is protected via a critical section.  Most    |
|      other IStartedThread methods that utilize this data do likewise.        |
|   2. This function is intended to be called by IThread::start().             |
------------------------------------------------------------------------------*/
void IStartedThread :: add ( IStartedThread &aThread )
  {
  IMODTRACE_DEVELOP( IString( "IStartedThread::add(" ) +
                     IString( aThread.id.asUnsigned() ) +
                     IString( ')' ) );
  // Lock static info.
  IResourceLock
    lock( IStartedThread::lock() );
  // If flag set, simply return.
  if ( aThread.inArray )
    return;
  // Search for free slot.
  for ( unsigned i = 0;
        i < IStartedThread::arraySize
        &&
        IStartedThread::threads[i];
        i++ ) {}
  if ( i == IStartedThread::arraySize )
    {
    // Extend array size.
    unsigned n = 2 * IStartedThread::arraySize + 1;
    // Allocate new array.
    IStartedThread **t = new IStartedThread*[ n ];
    // Copy old portion.
    while ( i-- )
      t[i] = IStartedThread::threads[ i ];
    // Zero remainder.
    i = IStartedThread::arraySize;
    while ( i < n )
      t[i++] = 0;
    // Free old array.
    delete [] IStartedThread::threads;
    // Update static info.
    IStartedThread::threads   = t;
    i                         = IStartedThread::arraySize;
    IStartedThread::arraySize = n;
    }
  // Set i-th slot to point to aThread.
  IStartedThread::threads[i] = &aThread;
  aThread.inArray = true;
  aThread.slot    = i;
  aThread.started = true;
  aThread.addRef();
  }

/*------------------------------------------------------------------------------
| IStartedThread::remove                                                       |
|                                                                              |
| Search the static array for the argument IStartedThread object and reset the |
| slot to zero if found.                                                       |
|                                                                              |
| Notes:                                                                       |
|   1. Access to the static data is protected via a critical section.  Most    |
|      other IStartedThread methods that utilize this data do likewise.        |
------------------------------------------------------------------------------*/
void IStartedThread :: remove ( IStartedThread &aThread )
  {
  IMODTRACE_DEVELOP( IString( "IStartedThread::remove(" ) +
                     IString( aThread.id.asUnsigned() ) +
                     IString( " - " ) +
                     IString::d2x( IString( (int)&aThread )) +
                     IString( ')' ) );
  IResourceLock
    lock( IStartedThread::lock() );
  if ( aThread.inArray )
    {
    IStartedThread::threads[ aThread.slot ] = 0;
    aThread.inArray = false;
    aThread.slot    = 0;
    aThread.removeRef();
    IStartedThread::flushCache();
    }
  }

/*------------------------------------------------------------------------------
| IStartedThread::locate                                                       |
|                                                                              |
| Search through the static array of started threads and return a pointer to   |
| the one with the argument threadID.                                          |
|                                                                              |
| If not found, then determine if the thread ID is valid and if so, allocate a |
| new IStartedThread and insert it into the array.                             |
|                                                                              |
| Notes:                                                                       |
|   1. The use count of the returned IStartedThread will be incremented        |
|      by one (or initialized to one if the object is allocated here).         |
|      The caller must ensure it gets decremented (IThread ensure this         |
|      by calling removeRef() at destruction).                                 |
|   2. Care is taken to ensure the static thread info is stable while          |
|      being examined/updated.                                                 |
|   3. This function throws an exception if the argument threadID is           |
|      not a valid OS/2 thread.                                                |
------------------------------------------------------------------------------*/
IStartedThread *IStartedThread :: locate ( const IThreadId &id )
  {
  IMODTRACE_ALL( IString( "IStartedThread::locate(" ) +
                 IString( id.asUnsigned() ) +
                 IString( ')' ) );
  IThreadId tid = id;
  IStartedThread
   *result = 0;
  {
  // Check cache first:
  ICritSec
    lock;
  result = IStartedThread::checkCacheFor( id );
  if ( result )
    result -> addRef();
  }
  // If not found, search array for this thread...
  if ( !result )
    {
    // Lock static info.
    IResourceLock
      lock( IStartedThread::lock() );
    for ( unsigned i = 0;

          i < IStartedThread::arraySize
          &&
          ( IStartedThread::threads[i] == 0
            ||
            IStartedThread::threads[i]->id != id );

          i++ ) {}

    if ( i == IStartedThread::arraySize )
      // Thread not found, see if it is a real thread.
      if ( IStartedThread::isTIDValid( tid ) )
        // This thread looks OK.
        {
        // Allocate new IStartedThread for this TID.
        result = new IStartedThread( id );
        // Insert into array.
        IStartedThread::add( *result );
        // Cache this thread.
        IStartedThread::cache( id, result );
        // Watch for when it ends...
        if ( tid.asUnsigned() != 1 )
          IStartedThread::watch();
        }
      else
        {
        ITHROWLIBRARYERROR( IC_INVALID_THREAD_ID,
                            IErrorInfo::invalidParameter,
                            IException::recoverable );
        }
    else
      // Found a match.
      {
      result = IStartedThread::threads[i];
      result -> addRef();
      IStartedThread::cache( id, result );
      }
    }
  return result;
  }

/*------------------------------------------------------------------------------
| initThread                                                                   |
------------------------------------------------------------------------------*/
extern "C" void initThread ( void *aThread )
  {
  IStartedThread::initThread( (IStartedThread *)aThread );
  }

/*------------------------------------------------------------------------------
| IStartedThread::initThread                                                   |
|                                                                              |
| Implementation:                                                              |
|   Dispatch the run() virtual function of the IThreadFn object pointed        |
|   to by the argument.                                                        |
|                                                                              |
|   If the "auto init" flag is set, then we initialize PM (and terminate on    |
|   thread completion).                                                        |
|                                                                              |
| Notes:                                                                       |
|   1. The argument's use count is incremented for the duration of this        |
|      function to prevent it from being deleted due to IThread destructors    |
|      being called in other threads.                                          |
------------------------------------------------------------------------------*/
void IStartedThread :: initThread ( IStartedThread *aThread )
  {
  aThread->id = IThread::currentId();
  aThread->tib(); // Set pTIB member.

  IMODTRACE_DEVELOP( IString( "IStartedThread::initThread(" ) +
                     IString( aThread->id.asUnsigned() ) +
                     IString( " - " ) +
                     IString::d2x( IString( (int)aThread )) +
                     IString( ')' ) );

  aThread->pmStartedHere = aThread->autoInitPM;
  // Initialize GUI if we need to...
  if ( aThread->pmStartedHere )
    IThread::current().initializeGUI( aThread->queueSize );
  // If priority is other than "default", set it via API...
  if ( aThread->priorityClass != IApplication::noChange
       ||
       aThread->priorityLevel != -1 )
    IThread::current().setPriority( aThread->priorityClass,
                                    aThread->priorityLevel );
  // Dispatch user function...
  aThread->fnObj->run();
  // Terminate GUI if we initialized it...
  if ( aThread->pmStartedHere )
    IThread::current().terminateGUI();
  // Terminate thread...
  aThread->end();
  }

/*------------------------------------------------------------------------------
| IStartedThread::end                                                          |
|                                                                              |
| Reset the object to an "unstarted" state.  This function can only be called  |
| if this->startedHere is true.                                                |
------------------------------------------------------------------------------*/
void IStartedThread :: end ( )
  {
  IStartedThread::flushCache();
  this->startedHere = this->started = false;
  if ( this->fnObj )
    {
    this->fnObj->removeRef();
    this->fnObj = 0;
    }
  this->pTIB = 0;

  //----------------------------------------------
  // DEFECT 7079 : incorporate development change
  //----------------------------------------------
    // Make sure id is invalid now
#ifdef IC_PM
    id = IThreadId(-1);
#endif
#ifdef IC_WIN
    id = IThreadId(0);
#endif
  //----------------------------------------------

  IStartedThread::remove( *this );
  }

/*------------------------------------------------------------------------------
| IStartedThread::tib                                                          |
|                                                                              |
| Return pointer to system TIB structure.  If zero on entry, then this member  |
| is set.  This function can only be called for the current thread (if not,    |
| then 0 may be returned).                                                     |
------------------------------------------------------------------------------*/
tib_t *IStartedThread :: tib ( )
  {
  if ( !this->pTIB )
    {
#ifdef IC_PM
    PTIB
      ptib;
    PPIB
      ppib;
    DosGetInfoBlocks( &ptib, &ppib );
    if ( this->id == ptib->tib_ptib2->tib2_ultid )
      this->pTIB = ptib;
#endif
#ifdef IC_NOTYET
//need to implement this for windowsnt
#endif
    }
  return this->pTIB;
  }

/*------------------------------------------------------------------------------
| IStartedThread::variable                                                     |
|                                                                              |
| Examine each element in the vars array.  If one with a matching key is found,|
| return the value associated with it, otherwise return null.                  |
------------------------------------------------------------------------------*/
IString IStartedThread :: variable ( const IString &key ) const
  {
  unsigned
    index = 0;

  while ( index < this->numVars )
    if ( vars[ index ].key == key )
      return vars[ index ].value;
    else
      index++;

  return IString();
  }

/*------------------------------------------------------------------------------
| IStartedThread::setVariable                                                  |
|                                                                              |
| Search for a matching key.  If found, set the associated value to the new    |
| one. If no matching key is found and there is room in the array, fill it in  |
| with the argument key/value pair.                                            |
------------------------------------------------------------------------------*/
IStartedThread &IStartedThread :: setVariable ( const IString &key,
                                                const IString &value )
  {
  unsigned
    index = 0;
  {
  IResourceLock
    lock( IStartedThread::lock() );
  while ( index < this->numVars )
    if ( vars[ index ].key == key )
      {
      vars[ index ].value = value;
      break;
      }
    else
      index++;
  if ( index == this->numVars
       &&
       index < varSize )
    {
    vars[ index ].key = key;
    vars[ index ].value = value;
    this->numVars++;
    }
  else if (index >= varSize)
    {
    ITHROWLIBRARYERROR( IC_THREAD_VARIABLE_LIMIT_EXCEEDED,
                        IErrorInfo::invalidRequest,
                        IException::recoverable );
    }
  }
  return *this;
  }

/*------------------------------------------------------------------------------
| IStartedThread::lock                                                         |
|                                                                              |
| If the pointed-to private resource hasn't been initialized, do it.           |
|                                                                              |
| Return the private resource.                                                 |
------------------------------------------------------------------------------*/
IPrivateResource &IStartedThread :: lock ( )
  {
  if ( !IStartedThread::pResource )
    IStartedThread::pResource = new IPrivateResource;
  return *IStartedThread::pResource;
  }

/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
|                                                                              |
| The role of the IThread constructors is twofold:                             |
|   1. They must locate the appropriate IStartedThread instance,               |
|      creating one if necessary, set the "thread" member to point             |
|      to this instance, and bump the IStartedThread use count.                |
|   2. Start the OS/2 thread (if necessary).                                   |
------------------------------------------------------------------------------*/
IThread :: IThread ( const IThreadId &tid )
  : thread( IStartedThread::locate( tid ) ),
    fThreadData( 0 )
  {
  }

IThread :: IThread ( )
  : thread( IThread::newStartedThread() ),
    fThreadData( 0 )
  {
  }

IThread :: IThread ( const IThread &aThread )
  : thread( aThread.startedThread() ),
    fThreadData( 0 )
  {
  this->thread->addRef();
  }

/*------------------------------------------------------------------------------
| IThread::newStartedThread                                                    |
------------------------------------------------------------------------------*/
IStartedThread *IThread::newStartedThread ( )
  {
  return new IStartedThread;
  }

/*------------------------------------------------------------------------------
| IThread::asString                                                            |
------------------------------------------------------------------------------*/
IString IThread :: asString ( ) const
  {
  IString result( "IThread(" );
  result += IString( this->id().asUnsigned() );
  result += ")";
  return result;
  }

/*------------------------------------------------------------------------------
| IThread::asDebugInfo                                                         |
------------------------------------------------------------------------------*/
IString IThread :: asDebugInfo ( ) const
  {
  IString result( "IThread(" );
  result += Inherited::asDebugInfo();
  result += ",thread=";
  result += IString( (unsigned long) this->startedThread() );
  result += "->";
  result += this->startedThread()->asDebugInfo();
  result += ")";
  return result;
  }

/*------------------------------------------------------------------------------
| IThread::~IThread                                                            |
|                                                                              |
| The destructor must remove the reference to the associated IStartedThread.   |                                                         |
------------------------------------------------------------------------------*/
IThread :: ~IThread ( )
  {
  if ( this->thread )
     this->thread->removeRef();
//delete fThreadData; when it is defined.
  }

/*------------------------------------------------------------------------------
| IThread::operator =                                                          |
|                                                                              |
| Attach this thread to aThread's associated started thread.                   |
------------------------------------------------------------------------------*/
IThread &IThread::operator = ( const IThread &aThread )
  {
  if ( aThread.startedThread() )
     aThread.startedThread()->addRef();
  if ( this->thread )
     this->thread->removeRef();
  this->thread = aThread.startedThread();
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::startedThread                                                       |
|                                                                              |
| Returns the thread member.                                                   |
------------------------------------------------------------------------------*/
IStartedThread *IThread :: startedThread ( ) const
  {
  return this->thread;
  }

/*------------------------------------------------------------------------------
| IThread::currentId                                                           |
|                                                                              |
| Construct an IThreadId from the system info blocks.                          |
------------------------------------------------------------------------------*/
IThreadId IThread :: currentId ( )
  {
#ifdef IC_PM
  PTIB
    ptib;
  PPIB
    ppib;
  DosGetInfoBlocks( &ptib, &ppib );
  return IThreadId( ptib->tib_ptib2->tib2_ultid );
#endif
#ifdef IC_WIN
  return IThreadId( (IHandle::Value)GetCurrentThreadId());
#endif
  }

/*------------------------------------------------------------------------------
| IThread::prepareToStart                                                      |
|                                                                              |
| Check that thread is not currently started (if so, throw an exception).      |
| Otherwise, initialize the IStartedThread object appropriately and            |
| return its address.                                                          |
------------------------------------------------------------------------------*/
IStartedThread *IThread :: prepareToStart (
                                    const IReference<IThreadFn> &aFnObjRef,
                                    Boolean                      autoInitGUI )
  {
  IMODTRACE_DEVELOP( "IThread::prepareToStart" );

  IStartedThread
   *p = this->startedThread();

  // If thread is already started, throw exception.
  if ( p->isStarted() )
    {
    ITHROWLIBRARYERROR( IC_THREAD_STARTED,
                        IErrorInfo::invalidRequest,
                        IException::recoverable );
    }

  if ( !p->inArray )
    IStartedThread::add( *p );

  p->startedHere = true;
  p->fnObj       = &(*aFnObjRef);
  p->fnObj->addRef();
  p->autoInitPM  = autoInitGUI;

  return p;
  }

/*------------------------------------------------------------------------------
| IThread::threadStarted                                                       |
|                                                                              |
| Check the return code from beginthread (see ithreads.cpp).  If an error      |
| occurred, throw exception, else set thread id.                               |
------------------------------------------------------------------------------*/
void IThread :: threadStarted( IStartedThread *p, int rc )
  {
  if ( rc == -1 )
    {
    p -> end();
    ITHROWSYSTEMERROR(rc, "DosCreateThread", IErrorInfo::outOfSystemResource,
        IException::recoverable);
    }
  else
#ifdef IC_WIN
    p->id = (void*)rc;
#else
    p->id = rc;
#endif
  }

/*------------------------------------------------------------------------------
| IThread::stop                                                                |
|                                                                              |
| Issue DosKillThread.  If it fails, throw an exception.                       |
------------------------------------------------------------------------------*/
void IThread :: stop ( )
  {
#ifdef IC_PM
  unsigned long rc = DosKillThread( this->id() );
  if ( rc != NO_ERROR )
#endif
#ifdef IC_WIN
  unsigned long rc = TerminateThread((HANDLE)this->id().asUnsigned(),0);
  //0 might not be the correct exitcode in all cases
  if ( !rc )
#endif
    {
    ITHROWSYSTEMERROR( rc,
                       "DosKillThread",
                       IErrorInfo::accessError,
                       IException::recoverable );
    }
  else
    this->startedThread() -> end();
  }

/*------------------------------------------------------------------------------
| ICurrentThread::~ICurrentThread                                              |
|                                                                              |
| Empty dtor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
ICurrentThread :: ~ICurrentThread ( )
  {
  }

/*------------------------------------------------------------------------------
| ICurrentThread::suspend                                                      |
|                                                                              |
| Issue DosSuspendThread.  If it fails, throw an exception.                    |
------------------------------------------------------------------------------*/
void ICurrentThread :: suspend ( )
  {
  if ( this->isGUIInitialized() )
    {
    ITHROWLIBRARYERROR( IC_SUSPEND_PM_THREAD,
                        IErrorInfo::invalidRequest,
                        IException::recoverable );
    }
  else
    this->IThread::suspend();
  }

/*------------------------------------------------------------------------------
| IThread::suspend                                                             |
|                                                                              |
| Issue DosSuspendThread.  If it fails, throw an exception.                    |
------------------------------------------------------------------------------*/
void IThread :: suspend ( )
  {
#ifdef IC_PM
  unsigned long rc = DosSuspendThread( this->id() );
  if ( rc != NO_ERROR )
#endif
#ifdef IC_WIN
  unsigned long rc = SuspendThread( (HANDLE)this->id().asUnsigned() );
  if ( rc == 0xFFFFFFFF )
#endif
    {
    ITHROWSYSTEMERROR( rc,
                       "DosSuspendThread",
                       IErrorInfo::accessError,
                       IException::recoverable );
    }
  }

/*------------------------------------------------------------------------------
| IThread::resume                                                              |
|                                                                              |
| Issue DosResumeThread.  If it fails, throw an exception.                     |
------------------------------------------------------------------------------*/
void IThread :: resume ( )
  {
#ifdef IC_PM
  unsigned long rc = DosResumeThread( this->id() );
  if ( rc != NO_ERROR )
#endif
#ifdef IC_WIN
  unsigned long rc = ResumeThread( (HANDLE)this->id().asUnsigned() );
  if ( rc == 0xFFFFFFFF )
#endif
    {
    ITHROWSYSTEMERROR( rc,
                       "DosResumeThread",
                       IErrorInfo::accessError,
                       IException::recoverable );
    }
  }

/*------------------------------------------------------------------------------
| IThread::id                                                                  |
|                                                                              |
| Return the id from the associated started thread.                            |
------------------------------------------------------------------------------*/
IThreadId IThread :: id ( ) const
  {
  return this->startedThread()->id;
  }

/*------------------------------------------------------------------------------
| IThread::stackSize                                                           |
|                                                                              |
| Return the stkSize member of the referenced IStartedThread.                  |
------------------------------------------------------------------------------*/
unsigned long IThread :: stackSize ( ) const
  {
#ifdef IC_PM
  tib_t *ptib = this->startedThread()->tib();
  if ( ptib )
    {
    return (char*)ptib->tib_pstacklimit - (char*)ptib->tib_pstack;
    }
  else
    return this->startedThread()->stkSize;
#endif
#ifdef IC_WIN
#ifdef IC_NOTYET
//Need to implement this for windows nt
#endif
  return this->startedThread()->stkSize; // Temp for now
#endif
  }

/*------------------------------------------------------------------------------
| IThread::setStackSize                                                        |
|                                                                              |
| Set the stkSize member of the referenced IStartedThread.                     |
------------------------------------------------------------------------------*/
IThread &IThread :: setStackSize ( unsigned long newStkSize )
  {
  this->startedThread() -> stkSize = newStkSize;
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::autoInitGUI                                                         |
|                                                                              |
| Return the autoInitPM member of the referenced IStartedThread.               |
------------------------------------------------------------------------------*/
Boolean IThread :: autoInitGUI ( ) const
  {
  return this->startedThread() -> autoInitPM;
  }

/*------------------------------------------------------------------------------
| IThread::setAutoInitGUI                                                      |
|                                                                              |
| Set the autoInitPM member of the referenced IStartedThread.                  |
------------------------------------------------------------------------------*/
IThread &IThread :: setAutoInitGUI ( Boolean newFlag )
  {
  this->startedThread() -> autoInitPM = newFlag;
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::stopProcessingMsgs                                                  |
|                                                                              |
| If the thread is in a message polling loop, post a WM_QUIT, else, throw an   |
| invalid request exception.                                                   |
------------------------------------------------------------------------------*/
IThread &IThread :: stopProcessingMsgs ( )
  {
  IMODTRACE_DEVELOP( "ICurrentThread::stopProcessingMsgs" );

  IStartedThread
   *p = this->startedThread();

  if ( p->inMsgLoop )
    {
#if 0  // IC_NOTYET - Causes sporatic traps in Extra - HT
    // Allow some amount of clean up on the windows.  These windows
    // otherwise would not do WM_DESTROY processing, since WM_QUIT
    // would shut down any further messages.
    IWindowList* winList = this->windowList();
    if ( winList )
    {
       IWindowList::Cursor cursor( *winList );
       for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
       {         // Send a message to all windows.
          IWindow* window = winList->elementAt( cursor );
          window->sendEvent( IC_UM_CLOSE );
       }
    }
#endif

#ifdef IC_WIN
    // IC_NOTYET - This code is temporary until we implement
    // a portable version of postEvent for Threads - DAB
    // PostThreadMessage( p->id, WM_QUIT, (WPARAM)0, (LPARAM)0 );
    PostQuitMessage(0);
#else
    p->hmq.postEvent( WM_QUIT );
#endif
    }
  else
    {
    ITHROWLIBRARYERROR( IC_THREAD_NOT_PROCESSING_MSGS,
                        IErrorInfo::invalidRequest,
                        IException::recoverable );
    }
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::priorityClass                                                       |
|                                                                              |
|  Query the actual thread class.  If not available (i.e., the thread isn't    |
|  started) return the priority class that's been set).                        |
------------------------------------------------------------------------------*/
IApplication::PriorityClass IThread :: priorityClass ( ) const
  {
  IStartedThread
   *p = this->startedThread();
  if ( p->isStarted() )
    {
#ifdef IC_PM
    tib_t
     *t = p->tib();
    if ( t )
      {
      IApplication::PriorityClass result;
      switch ( t->tib_ptib2->tib2_ulpri / 256 )
#endif
#ifdef IC_WIN   // IC_NOTYET
    DWORD t = GetPriorityClass( p);
    if ( t )
    {
      IApplication::PriorityClass result;
      switch ( t )
#endif
        {
        case 0:
          result = IApplication::noChange;
          break;
        case 1:
          result = IApplication::idleTime;
          break;
        case 2:
          result = IApplication::regular;
          break;
        case 3:
          result = IApplication::timeCritical;
          break;
        case 4:
          result = IApplication::foregroundServer;
          break;
        }
      return result;
      }
    }
  return p->priorityClass;
  }

/*------------------------------------------------------------------------------
| IThread::priorityLevel                                                       |
|                                                                              |
| Query the actual thread priority.  If not available (i.e., the thread        |
| isn't started) return either the priority that's been set, or, 0).           |
------------------------------------------------------------------------------*/
unsigned IThread :: priorityLevel ( ) const
  {
  IStartedThread
   *p = this->startedThread();
  if ( p->isStarted() )
    {
#ifdef IC_PM
    tib_t
     *t = p->tib();
    if ( t )
      return t->tib_ptib2->tib2_ulpri % 256;
#endif
#ifdef IC_WIN   // IC_NOTYET
    DWORD t = GetPriorityClass( p);
    return t;
#endif
    }
  return ( p->priorityLevel == -1 ) ? 0 : p->priorityLevel;
  }

/*------------------------------------------------------------------------------
| IThread::adjustPriority                                                      |
|                                                                              |
| Adjust the thread priority level by the given delta.                         |
------------------------------------------------------------------------------*/
IThread &IThread :: adjustPriority ( int delta )
  {
  IStartedThread
   *p = this->startedThread();
  p->priorityLevel = delta;

#ifdef IC_PM
  if ( p->isStarted() )
    {
#ifdef IC_WIN
    unsigned long
      rc = SetThreadPriority( (HANDLE)(unsigned long)p->id, delta );
    if ( !rc  )
#else
    unsigned long
      rc = DosSetPriority( PRTYS_THREAD, PRTYC_NOCHANGE, delta, p->id );
    if ( rc != NO_ERROR )
#endif
      {
      ITHROWSYSTEMERROR( rc,
                         "DosSetPriority",
                         IErrorInfo::accessError,
                         IException::recoverable );
      }
    }
#endif
#ifdef IC_NOTYET
//don't know how to set it to a delta in windowsnt
#endif
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::setPriority                                                         |
|                                                                              |
| Set the priority class member of the referenced IStartedThread.              |
| In addition, issue DosSetPriority (in case the thread is started).           |
------------------------------------------------------------------------------*/
IThread &IThread :: setPriority ( IApplication::PriorityClass priorityClass,
                                  unsigned                    priorityLevel )
  {

  if ( priorityClass == IApplication::noChange )
    priorityClass = this->priorityClass();

  IStartedThread
   *p = this->startedThread();
  p->priorityClass = priorityClass;
  p->priorityLevel = priorityLevel;
  if ( p->isStarted() )
    {
#ifdef IC_PM
    unsigned long
#endif
#ifdef IC_WIN
    int
#endif
     pclass;
    switch ( priorityClass )
      {
      case IApplication::idleTime:
        pclass = PRTYC_IDLETIME;
        break;
      case IApplication::regular:
        pclass = PRTYC_REGULAR;
        break;
      case IApplication::timeCritical:
        pclass = PRTYC_TIMECRITICAL;
        break;
      case IApplication::foregroundServer:
        pclass = PRTYC_FOREGROUNDSERVER;
        break;
      }
#ifdef IC_PM
    unsigned long
      rc = DosSetPriority( PRTYS_THREAD,
                           pclass,
                           priorityLevel,
                           p->id );
    if ( rc != NO_ERROR )
#endif
#ifdef IC_WIN
    unsigned long
      rc = SetThreadPriority( (HANDLE)p->id.asUnsigned(),pclass);
    if (!rc)
#endif
      {
      ITHROWSYSTEMERROR( rc,
                         "DosSetPriority",
                         IErrorInfo::accessError,
                         IException::recoverable );
      }
    }
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::queueSize                                                           |
|                                                                              |
| Returns the queue size from the associated started thread                    |
------------------------------------------------------------------------------*/
long IThread :: queueSize ( ) const
  {
  return this->startedThread()->queueSize;
  }

/*------------------------------------------------------------------------------
| IThread::setQueueSize                                                        |
|                                                                              |
| Set the queue size of the associated started thread                          |
------------------------------------------------------------------------------*/
IThread &IThread :: setQueueSize ( long newQSize )
  {
  this->startedThread()->queueSize = newQSize;
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::messageQueue                                                        |
|                                                                              |
| Return the hmq. First try the stored hmq (if there is one), else return 0 .  |
| This non-const member function will adjust the stored HMQ if it is           |
| found to be invalid.                                                         |
------------------------------------------------------------------------------*/
IMessageQueueHandle IThread :: messageQueue ( )
  {
  IStartedThread
   *p = this->startedThread();

  IMessageQueueHandle hmq = p->hmq;

#ifdef IC_PM
  if ( p->hmq )
    {
    MQINFO info;
    if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
      {
      p->hmq = 0; // No longer valid.
      hmq    = 0;
      }
    }
#endif
#ifdef IC_NOTYET
//don't know how to do this in windows nt
#endif
  return hmq;
  }

/*------------------------------------------------------------------------------
| IThread::messageQueue                                                        |
|                                                                              |
| Return the hmq. First try the stored hmq (if there is one), else return 0 .  |
------------------------------------------------------------------------------*/
IMessageQueueHandle IThread :: messageQueue ( ) const
  {
  IStartedThread
   *p = this->startedThread();

  IMessageQueueHandle hmq = p->hmq;

#ifdef IC_PM
  if ( p->hmq )
    {
    MQINFO info;
    if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
      hmq = 0;
    }
#endif
#ifdef IC_NOTYET
//don't know how to do this in windows nt
#endif

  return hmq;
  }

/*------------------------------------------------------------------------------
| IThread::windowList                                                          |
|                                                                              |
| Returns the window list from the associated started thread                   |
------------------------------------------------------------------------------*/
IWindowList *IThread :: windowList ( ) const
{
  return this->startedThread()->windowList;
}

/*------------------------------------------------------------------------------
| IThread::setWindowList                                                       |
|                                                                              |
| Sets the window list for the associated start thread                         |
------------------------------------------------------------------------------*/
IThread &IThread :: setWindowList ( IWindowList *list )
  {
  this->startedThread()->windowList = list;
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::isStarted                                                           |
|                                                                              |
| Returns true if the associated start thread has been started                 |
------------------------------------------------------------------------------*/
Boolean IThread :: isStarted ( ) const
  {
  return this->startedThread()->isStarted();
  }

/*------------------------------------------------------------------------------
| ICurrentThread::ICurrentThread                                               |
|                                                                              |
| Initialize referenced IStartedThread to zero.                                |
|                                                                              |
| Notes:                                                                       |
|   1. During the course of constructing this object, an IStartedThread        |
|      corresponding to thread 1 (the main application thread) is              |
|      created.  This IStartedThread is never used directly, however.          |
|   2. This constructor is protected to prevent creation of additional         |
|      objects of this class.  The only instance is the static                 |
|      currentThread object within this module (see above).                    |
------------------------------------------------------------------------------*/
ICurrentThread :: ICurrentThread ( )
#ifdef IC_WIN
  : IThread( (void*)1 )
#else
  : IThread( 1 )
#endif
  {
  }

/*------------------------------------------------------------------------------
| ICurrentThread::anchorBlock                                                  |
|                                                                              |
| Return an IAnchorBlockHandle constructed from the current thread's HAB.      |
------------------------------------------------------------------------------*/
IAnchorBlockHandle ICurrentThread :: anchorBlock ( ) const
  {
  IStartedThread
   *p = this->startedThread();

  IAnchorBlockHandle
    hab( 0 );

#ifdef IC_PM
  if ( !p->hab )
  {
    if ( ( hab = WinInitialize( 0 ) ) != 0 )
       p->hab = hab;
    else
    {
      ITHROWGUIERROR2( "WinInitialize",
                       IErrorInfo::accessError,
                       IException::unrecoverable );
    }
  }
  else
  {
    hab = p->hab;
  }
#endif
#ifdef IC_WIN
//don't need to do anything here since hab is initialized to 0
#endif
#ifdef IC_NOTYET
//don't know if this is good enough
#endif
  return( hab );
  }

/*------------------------------------------------------------------------------
| ICurrentThread::isGUIInitialized                                              |
|                                                                              |
| Return true if PM has been initialized for this thread.                      |
------------------------------------------------------------------------------*/
Boolean ICurrentThread :: isGUIInitialized ( ) const
  {
#ifdef IC_PM
  if ( this->anchorBlock() != 0
       &&
       this->messageQueue() != 0 )
    return true;
  else
    return false;
#else
  // Function is effectively a no-op since no setup needs to be done
  return true;
#endif
  }

/*------------------------------------------------------------------------------
| ICurrentThread::messageQueue                                                 |
|                                                                              |
| Return the hmq. First try the stored hmq ( IThread handles this)             |
| (if there is one), else try HMQ_CURRENT.                                     |
| Return 0 if no message queue is stored and HMQ_CURRENT is invalid.           |
------------------------------------------------------------------------------*/
IMessageQueueHandle ICurrentThread :: messageQueue ( )
  {
  IMessageQueueHandle hmq = this->Inherited::messageQueue();
  if ( !hmq )
    {
#ifdef IC_PM
    MQINFO info;
    hmq = HMQ_CURRENT;
    if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
      hmq = 0;
#endif
#ifdef IC_WIN
    hmq=(void*)1;
#endif
#ifdef IC_NOTYET
//I know this fix for windownt isn't correct.
#endif
    }
  return hmq;
  }

/*------------------------------------------------------------------------------
| ICurrentThread::messageQueue                                                 |
|                                                                              |
| Return the hmq. First try the stored hmq ( IThread handles this)             |
| (if there is one), else try HMQ_CURRENT.                                     |
| Return 0 if no message queue is stored and HMQ_CURRENT is invalid.           |
------------------------------------------------------------------------------*/
IMessageQueueHandle ICurrentThread :: messageQueue ( ) const
  {
  IMessageQueueHandle hmq = this->Inherited::messageQueue();
  if ( !hmq )
    {
#ifdef IC_PM
    MQINFO info;
    hmq = HMQ_CURRENT;
    if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
      hmq = 0;
#endif
#ifdef IC_WIN
    hmq=(void*)1;
#endif
#ifdef IC_NOTYET
//I know this fix for windownt isn't correct.
#endif
    }
  return hmq;
  }

/*------------------------------------------------------------------------------
| ICurrentThread::exit                                                         |
|                                                                              |
| Terminate the current thread with the given return code.  The thread is      |
| terminated via DosExit.                                                      |
------------------------------------------------------------------------------*/
void ICurrentThread :: exit ( unsigned long rc )
  {
  this->startedThread()->end();

  if ( id().asUnsigned() == 1 )
    ::exit( rc );
  else
    {
  #ifdef IC_PM
    DosExit( EXIT_THREAD, rc );
  #endif
  #ifdef IC_WIN
    ExitThread(rc);
  #endif
    }
  }

/*------------------------------------------------------------------------------
| ICurrentThread::sleep                                                        |
|                                                                              |
| Suspend the current thread for the given number of milliseconds.  The thread |
| is suspended via DosSleep.                                                   |
------------------------------------------------------------------------------*/
ICurrentThread &ICurrentThread :: sleep ( unsigned long msecs )
  {
  ISLEEP( msecs );
  return *this;
  }

/*------------------------------------------------------------------------------
| ICurrentThread::waitFor                                                      |
|                                                                              |
| Issue DosWaitThread on the argument thread's id.                             |
------------------------------------------------------------------------------*/
ICurrentThread &ICurrentThread :: waitFor ( const IThread &anotherThread )
  {
  IMODTRACE_DEVELOP( "ICurrentThread::waitFor" );
  IThreadId tid = anotherThread.id();

  IASSERTPARM( tid.asUnsigned() != 1 );
  IASSERTPARM( tid != IThread::current().id() );
  if ( tid )
    {
    unsigned long
#ifdef IC_PM
      rc = DosWaitThread( (PTID)&tid, DCWW_WAIT );
    if ( rc != NO_ERROR && rc != ERROR_INVALID_THREADID )
#endif
#ifdef IC_WIN
      rc = WaitForSingleObject((HANDLE)tid.asUnsigned(),INFINITE);
    if ( rc == 0xFFFFFFFF)
#endif
      {
      ITHROWSYSTEMERROR( rc,
                         "DosWaitThread",
                         IErrorInfo::accessError,
                         IException::recoverable );
      }
    }
  return *this;
  }

/*------------------------------------------------------------------------------
| ICurrentThread::waitForAllThreads                                            |
|                                                                              |
| Wait for any thread till there are no more...                                |
------------------------------------------------------------------------------*/
ICurrentThread &ICurrentThread :: waitForAllThreads ( )
{
  Boolean loopFlag = true;
  IThreadId tid;
  IMODTRACE_DEVELOP( "ICurrentThread::waitForAllThreads" );
  IASSERTPARM( this->id().asUnsigned() == 1 );
  while ( loopFlag )
  {
    /******************************************************************/
    /* Have the current thread wait until the next thread in the      */
    /* process ends                                                   */
    /******************************************************************/
    tid = 0;
#ifdef IC_PM
    if ( DosWaitThread( (PTID)&tid, DCWW_WAIT ) != NO_ERROR )
      loopFlag = false;
#endif
#ifdef IC_WIN
    //IC_NOTYET - Can't do this in windows nt??
    loopFlag = false;
#endif
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentThread::waitForAnyThread                                             |
|                                                                              |
| Issue DosWaitThread for thread id 0 and return the thread actually           |
| terminating.  Throw an exception in case of errors.                          |
------------------------------------------------------------------------------*/
IThreadId ICurrentThread :: waitForAnyThread ( )
  {
  IMODTRACE_DEVELOP( "ICurrentThread::waitForAnyThread" );
  IThreadId tid = 0;
#ifdef IC_PM
  unsigned long
    rc = DosWaitThread( (PTID)&tid, DCWW_WAIT );
  if ( rc != NO_ERROR )
  {
     ITHROWSYSTEMERROR( rc,
                       "DosWaitThread",
                       IErrorInfo::accessError,
                       IException::recoverable );
  }
#endif
  return IThreadId( tid );
#ifdef IC_NOTYET
//can't do this in windows nt
#endif
  }

/*------------------------------------------------------------------------------
| ICurrentThread :: id                                                         |
|                                                                              |
| Return the current thread's id.                                              |
------------------------------------------------------------------------------*/
IThreadId ICurrentThread :: id ( ) const
  {
  return IThread::currentId();
  }

/*------------------------------------------------------------------------------
| ICurrentThread::remainingStack                                               |
|                                                                              |
| Approximate stack remaining as the distance between the stack base           |
| (from the tib structure) and the current stack pointer (approximately        |
| the address of this function's argument).                                    |
------------------------------------------------------------------------------*/
unsigned long ICurrentThread :: remainingStack ( ) const
  {
#ifdef IC_PM
  tib_t
   *tib = this->startedThread()->tib();
  void
   *base = tib->tib_pstack,
   *cur  = &tib;
  return (char*)cur - (char*)base;
#endif
#ifdef IC_WIN
//#ifdef __BORLANDC__
// return stackavail();
//#else
 return 1000L;
//#endif
#endif
  }

/*------------------------------------------------------------------------------
| ICurrentThread::processMsgs                                                  |
|                                                                              |
| Loop, issuing WinGetMessage/WinDispatchMsg, until a WM_QUIT is received. If  |
| the quit has a non-NULLHANDLE hwnd in mp2, then it probably came from the    |
| task list so instead of quitting, post a WM_SYSCOMMAND+SC_CLOSE to the       |
| window in question.                                                          |
------------------------------------------------------------------------------*/
void ICurrentThread :: processMsgs ( )
{
  IMODTRACE_DEVELOP( "ICurrentThread::processMsgs" );

  IStartedThread
   *p = this->startedThread();
  if ( ! this->isGUIInitialized() )
    this->initializeGUI();

  IAnchorBlockHandle
    hab = this->anchorBlock();
  QMSG
    msg;

  p->inMsgLoop++;          //signal start of the loop
  ITRACE_DEVELOP( "Starting message loop - Threads in message loop = " +
                  IString( p->inMsgLoop ) );

  while ( true )
  {
    if ( IGETMSG( hab, &msg, 0, 0, 0 ) )
    {
#ifdef IC_WIN
      //Need to add a call to TranslateMessage for Windows nt
      TranslateMessage( &msg );
#endif
      IDISPATCHMSG( hab, &msg );
    }
#ifdef IC_NOTYET
//might need to add a call to translateMessage for windows nt???
#endif
    else
    {
#ifdef IC_PM
      if ( msg.mp2 != NULL )
      {
        IWindowHandle frame( (unsigned long)( msg.mp2 ) );
        if ( frame.isValid() )
        {     // (A modal frame closed via the task list will
              // have its now invalid window handle here.)
           frame
            .postEvent( WM_SYSCOMMAND,
                        SC_CLOSE,
                        IEventData( CMDSRC_OTHER, false ) );
        }
      }
      else
#endif
#ifdef IC_WIN
#ifdef IC_NOTYET
      // This special code does not appear to be needed in Windows
      // at least I can't find it documented as necessary anywhere.
      if ( msg.lParam != NULL )
      {
        IWindowHandle frame( (void*)( msg.lParam ) );
        if ( frame.isValid() )
        {
           frame
            .postEvent( WM_SYSCOMMAND,
                        SC_CLOSE,
                        IEventData( 0 ) );
        }
      }
      else
#endif
#endif
        break;
    } // end getmsg
  } // end while

  p->inMsgLoop--;          //signal end of the loop
  ITRACE_DEVELOP( "Ending message loop - Threads in message loop = " +
                  IString( p->inMsgLoop ) );
}

/*------------------------------------------------------------------------------
| ICurrentThread::initializeGUI                                                |
|                                                                              |
| Initialize the thread for PM by calling WinInitialize (in                    |
| ICurrentThread::anchorBlock) and WinCreateMsgQueue.                          |
| If an error occurs, throw an exception.                                      |
|                                                                              |
| Notes:                                                                       |
|   1. Increment the use count for the referenced IStartedThread (so the       |
|      thread-related hab/hmq are retained till terminateGUI).                 |
------------------------------------------------------------------------------*/
void ICurrentThread :: initializeGUI ( long queueSize )
{
  IMODTRACE_DEVELOP( "ICurrentThread::initializeGUI" );
#ifdef IC_PM
  IStartedThread
   *p = this->startedThread();
  IAnchorBlockHandle hab = this->anchorBlock();

// Moved WinInitialize logic to anchorBlock() function

  IMessageQueueHandle hmq = this->messageQueue();
  if ( !hmq )
  {
    if ( !queueSize )
      queueSize = this->queueSize();

    if ( ( hmq = WinCreateMsgQueue( hab, queueSize ) ) != 0 )
    {
       p->hmq = hmq;
       p->addRef();
    }
    else
      // WinCreateMsqQueue failed, throw exception...
    {
      ITHROWGUIERROR2( "WinCreateMsgQueue",
                       IErrorInfo::accessError,
                       IException::unrecoverable );
    }
  }
#else
// In Windows NT, a message queue is automatically created
// for each thread so no extra work needs to be done.
#endif
}

/*------------------------------------------------------------------------------
| ICurrentThread::terminateGUI                                                  |
|                                                                              |
| Issue WinDestroyMsgQueue and WinTerminte, as necessary.                      |
------------------------------------------------------------------------------*/
void ICurrentThread :: terminateGUI ( )
{
  IMODTRACE_DEVELOP( "ICurrentThread::terminateGUI" );
#ifdef IC_PM
  IStartedThread
   *p = this->startedThread();

  if ( this->messageQueue() )
  {
     WinDestroyMsgQueue( this->messageQueue() );
     if ( p->hmq )
     {             // Created and stored in initializeGUI.
        p->hmq = 0;
        p->removeRef();
     }
  }

  if ( this->anchorBlock() )
     WinTerminate( this->anchorBlock() );
#else
// Message queue deleted automatically in Windows NT so no-op
#endif
}

/*------------------------------------------------------------------------------
| IThread__watcher                                                             |
------------------------------------------------------------------------------*/
void _System IThread__watcher ( unsigned long );

/*------------------------------------------------------------------------------
| IStartedThread::watch                                                        |
|                                                                              |
| This function is called when a non-IThread thread is encountered in          |
| IStartedThread::startedThread.  The first time it is called it launches      |
| a secondary thread to watch for thread terminations.  This thread will       |
| see if terminating threads are non-IThread and if so, call end() to          |
| clean them up and call removeRef() to free the IStartedThread resource       |
| (if no other references exist).                                              |
|                                                                              |
| The thread watching is done by the function IThread__watcher (see below).    |
|                                                                              |
| Notes:                                                                       |
|   1. Must be called with resources locked.                                   |
------------------------------------------------------------------------------*/
void IStartedThread :: watch ( )
  {
  IMODTRACE_DEVELOP( "IStartedThread::watch" );
#ifdef IC_PM
  TID tid;
#endif
  if ( !IStartedThread::watching )
    {
    unsigned long
#ifdef IC_PM
      rc = DosCreateThread( &tid,
                            IThread__watcher,
                            0,
                            0,
                            16384 );
    if ( rc )
#endif
#ifdef IC_WIN
     rc, tempId;

     rc = (unsigned long)CreateThread( 0,
                                       16384,
                                       (LPTHREAD_START_ROUTINE)IThread__watcher,
                                       0,
                                       0,
                                       &tempId);
     if ( !rc )
#endif
      {
      ITHROWSYSTEMERROR( rc,
                         "DosCreateThread",
                         IErrorInfo::accessError,
                         IException::unrecoverable );
      }
    else
      {                                     // 9868
      IStartedThread::watching = true;
      IThreadWatcher::tid = tid;            // 9868
      }                                     // 9868
    }
  }

/*------------------------------------------------------------------------------
| IThread__watcher                                                             |
|                                                                              |
| This function watches for thread terminations.  When a thread ends, the      |
| list of started threads is searched for this thread.  If found, then we      |
| remove it from the list and call end().  Also, the reference is count        |
| is decremented (twice if PM had been started).                               |
------------------------------------------------------------------------------*/
void _System IThread__watcher ( unsigned long )
{
  Boolean
    watching = true;
  unsigned long
    rc = 0;

  while ( watching && rc == 0 )
  {
    IThreadId tid = 0;
    ITRACE_DEVELOP( "Waiting for a thread to end..." );
#ifdef IC_PM
    rc = DosWaitThread( (PTID)&tid, DCWW_WAIT );
#endif
#ifdef IC_WIN
    rc = 1;
#endif
#ifdef IC_NOTYET
//  don't know how to implement this in windows nt
#endif
    // Check to see if a thread really ended...
    if ( rc == 0 )
    {
       ITRACE_DEVELOP( "Thread " + IString( tid.asUnsigned() ) + " ended." );
       IResourceLock
         lock( IStartedThread::lock() );
       unsigned
         index = 0;

       // Check if the application managed to start another thread
       // with the same thread ID.
       Boolean
         reusedId = IStartedThread::isTIDValid( tid ),
         threadEnded = false;
       IStartedThread* restartedThread = 0;

       // Kill this thread unless no more not "startedHere" threads.
       watching = false;
       // Examine all started threads...
       while ( index < IStartedThread::arraySize )
       {
          IStartedThread
           *t = IStartedThread::threads[ index ];
          // Make sure slot is filled...
          if ( t )
          {
             // See if this is the entry for the thread that just ended.
             if ( t->id == tid )
             {        // Thread IDs match.
                if ( reusedId )
                {     // The ended thread has been replaced with
                      // another one with the same thread ID.  Be
                      // sure not to delete/end the wrong one.
                   if ( restartedThread )
                   {
                      if ( t->fCount > restartedThread->fCount )
                      {    // Keep the newest IStartedThread.
                         IStartedThread* swap = t;
                         t = restartedThread;     // End the old one.
                         restartedThread = swap;  // Keep the new one.
                      }
                      // Else t, restartedThread already ok.
                   }
                   else
                   {       // Decide later whether to end this one.
                      restartedThread = t;
                   }
                }

                if ( ! t->startedHere  &&
                     t != restartedThread )
                {     // Need to "end" the IStartedThread.
                   if ( restartedThread )
                   {
                      ITRACE_DEVELOP( "\tEnding oldest thread with same ID." );
                   }
                   ITRACE_DEVELOP( "\tRemoving from IStartedThread array!" );
                   t -> end();
                   threadEnded = true;
                }
             }
             else if ( ! t->startedHere )
             {     // If not started here, keep watching...
                watching = true;
             }
          }
          index++;
       }  // endwhile

       if ( restartedThread  &&
            threadEnded == false  &&
            ! restartedThread->startedHere )
       {  // Found only one IStartedThread with the ID to delete,
          // even though another thread with the same ID has been
          // started.  The IStartedThread could either be the one to
          // end (if it was not started via IThread) or the newly
          // started thread (if the ended thread was created via
          // IThread, and the new one wasn't but is now wrappered
          // with an IThread).
          ITRACE_DEVELOP( "\tOnly one of two started threads found..." );
          ITRACE_DEVELOP( "\tRemoving from IStartedThread array!" );
          restartedThread->end();
       }
    }
    else
    {
       ITRACE_DEVELOP( "Error watching threads, rc=" + IString( rc ) );
    }
  }

  ITRACE_DEVELOP( "No more threads to watch." );
  IStartedThread::watching = false;
}

/*------------------------------------------------------------------------------
| ICurrentThread::startedThread                                                |
|                                                                              |
| Returns the result of looking up the current TID.                            |
------------------------------------------------------------------------------*/
IStartedThread *ICurrentThread :: startedThread ( ) const
  {
  IStartedThread
   *result = IStartedThread::locate( this->id() );
  result -> removeRef();
  return result;
  }

/*------------------------------------------------------------------------------
| IThread::inGUISession                                                         |
|                                                                              |
| Query the current session type and return whether it is PM.                  |
------------------------------------------------------------------------------*/
Boolean IThread :: inGUISession ( )
  {
#ifdef IC_PM
  PTIB
    ptib;
  PPIB
    ppib;
  DosGetInfoBlocks( &ptib, &ppib );
  if ( ppib->pib_ultype == SSF_TYPE_PM )
    return true;
  else
    return false;
#endif
#ifdef IC_WIN
  return true;
#endif
#ifdef IC_NOTYET
//added a temporary hard code of true for windows nt
#endif
  }

/*------------------------------------------------------------------------------
| IThread::variable                                                            |
|                                                                              |
| Dispatch this request to the corresponding started thread.                   |
------------------------------------------------------------------------------*/
IString IThread :: variable ( const IString &key ) const
  {
  return this-> startedThread() -> variable( key );
  }

/*------------------------------------------------------------------------------
| IThread::setVariable                                                         |
|                                                                              |
| Dispatch this request to the corresponding started thread.                   |
------------------------------------------------------------------------------*/
IThread &IThread :: setVariable ( const IString &key,
                                  const IString &value )
  {
  this-> startedThread() -> setVariable( key, value );
  return *this;
  }

/*------------------------------------------------------------------------------
| IThread::defaultAutoInitGUI                                                  |
|                                                                              |
| Return static default value.                                                 |
------------------------------------------------------------------------------*/
Boolean IThread :: defaultAutoInitGUI ( )
  {
  if ( IThread::dfltAutoInitGUI == -1 )
    IThread::dfltAutoInitGUI = IThread::inGUISession();
  return IThread::dfltAutoInitGUI;
  }

/*------------------------------------------------------------------------------
| IThreadFn::IThreadFn                                                         |
|                                                                              |
| Empty ctor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
IThreadFn :: IThreadFn ( )
  {
  }

/*------------------------------------------------------------------------------
| IThreadFn::~IThreadFn                                                        |
|                                                                              |
| Empty dtor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
IThreadFn :: ~IThreadFn ( )
  {
  }

/*------------------------------------------------------------------------------
| IThread::current                                                             |
|                                                                              |
| Returns the current thread                                                   |
------------------------------------------------------------------------------*/
ICurrentThread &IThread :: current ( )
  {
  if ( !IThread::pCurrent )
    IThread::pCurrent = new ICurrentThread;
  return *IThread::pCurrent;
  }

/*------------------------------------------------------------------------------
| IThread::inMsgLoop                                                           |
|                                                                              |
| Get the receiver's IStartedThread and query its inMsgLoop flag.              |
------------------------------------------------------------------------------*/
Boolean IThread :: inMsgLoop ( ) const
  {
  return( this->startedThread()->inMsgLoop );
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::Cursor                                                      |
|                                                                              |
| Set data members:                                                            |
|   all      - set from constructor argument                                   |
|   created  - set to 0 (which is always invalid)                              |
------------------------------------------------------------------------------*/
IThread::Cursor :: Cursor ( Boolean allThreads )
  : all( allThreads ),
    iteratedAll( false ),
    created( 0 ),
    fCursorData( 0 )
  {
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::~Cursor                                                     |
------------------------------------------------------------------------------*/
IThread::Cursor :: ~Cursor ( )
  {
//delete fCursorData; when it is defined.
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::setToFirst                                                  |
|                                                                              |
| Point to first element of started thread array and reset this cursor's       |
| time stamp.                                                                  |
------------------------------------------------------------------------------*/
Boolean IThread::Cursor :: setToFirst ( )
  {
  this->created = IThread::Cursor::timeStamp;
  this->cursor  = (unsigned)-1;
  this->iteratedAll = false;
  return this -> advance();
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::setToNext                                                   |
|                                                                              |
| Advance cursor and return whether it's still valid.                          |
------------------------------------------------------------------------------*/
Boolean IThread::Cursor :: setToNext ( )
  {
  return this -> advance();
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::isValid                                                     |
|                                                                              |
| A cursor is valid iff its timestamp equals the static one (i.e., no          |
| threads have been started/stopped since it was reset).                       |
------------------------------------------------------------------------------*/
Boolean IThread::Cursor :: isValid ( ) const
  {
  return this->created == IThread::Cursor::timeStamp;
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::invalidate                                                  |
|                                                                              |
| Invalidate the cursor by setting it's creation "timestamp" to an invalid     |
| value.                                                                       |
------------------------------------------------------------------------------*/
void IThread::Cursor :: invalidate ( )
  {
  this->created = 0;
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::threadId                                                    |
|                                                                              |
| Check validity and return TID of started thread at cursor.                   |
------------------------------------------------------------------------------*/
IThreadId IThread::Cursor :: threadId ( ) const
  {
  IASSERTPARM( this->isValid() );
  return IStartedThread::threads[ this->cursor ] -> id;
  }

/*------------------------------------------------------------------------------
| IThread::Cursor::advance                                                     |
|                                                                              |
| Advance cursor to next IThread.  Skip non-PM if required.  If we run off     |
| the end, invalidate the cursor.                                              |
------------------------------------------------------------------------------*/
Boolean IThread::Cursor :: advance ( )
  {
  ICritSec
    lock;
  if ( this->isValid() )
    {
    for ( this->cursor++;
          this->cursor < IStartedThread::arraySize;
          this->cursor++ )
      {
      IStartedThread
       *p = IStartedThread::threads[ this->cursor ];
      if ( p && p->id )
        if ( this->all
             ||
             p->windowList )
          break;
      }
    if ( this->cursor == IStartedThread::arraySize )
      {
      this->iteratedAll = true;   // Flag no more items.
      this->invalidate();
      }
    }
  return this->isValid();
  }
