/*****************************************************************************
  Module : DataType.h
  Contents : Holds data type declarations for robust
  Robust makes use of a number of different data types. All these declarations
  are used by the numerics portion of the program. The Parameters structure
  is also included, as it store stae information about a program.
*****************************************************************************/
#ifndef DATATYPES
#define DATATYPES

#include <stdlib.h>

//The possible types of data held by a FailureDataSet object
enum DataType { Empty, Failure_Time, Failure_Interval, SRGModel };

//The possible data types that could be held by a FailureDataSet or
//CoverageDataSet object. Used by Robust when retaining state information.
//Not all model types are used by the program
enum modtype { Exponential, Logarithmic, Power, DelayedS, Littlewood,
	       Static, Coverage, Mixed, User_Defined, NotPresent };

/****************************************************************************
  Structure : FailureDataPoint
  Purpose : Encapsulates a single data point in a failure data set. This
  is primarly a helper class for FailureDataSet.
**************************************************************************/
struct FailureDataPoint
{
  double count;
  float time,interval;
  double intensity;
};

/****************************************************************************
  Class : FailureDataSet
  Purpose : Encapsulates a set of failure data. This data can be from a model,
  actual project data, or a smoothed version of that data.
****************************************************************************/

class FailureDataSet
{
  void FillInterval(void);    //Method to fill interval data
  void FillTime(void);        //Method to fill time data
  void FillIntensity(void);   //Compute intensity of the data set
public:
  int Size;                          //Number of data points
  char* FileName;                    //Name of file that held original data
  DataType Type;                     //Type of data held
  FailureDataPoint* Failure_Data;    //Pointer to array of failure data
  
  FailureDataSet(void);      //Constructor
  ~FailureDataSet(void);     //Destructor
  FailureDataPoint& operator[](int);
  int OpenFile(char*);       //Method to load data from a file
  int SaveFile(char* = NULL);//Method to save data back to a file
  int SetSize(int);           //Method to change the number of data points
  void Shrink(void);          //Method to shirnk Failure_Data to size
  int Fill(void);             //Method to autofill data
};

/****************************************************************************
  Class : StaticMetrics
  Purpose : This class encapsulates a set of static metrics. This should
  probably be defined as a struct, but g++ seems to have trouble with
  member functions for structs.
****************************************************************************/

class StaticMetrics
{
public:
  float Constant;            //Experimentally determined constant
  float Structure;          //The complexity of the project
  float Phase;               //The development phase of the project
  float Prog_Team;           //Average programmer experince, in years
  float Maturity;            //Maturity of the development process
  float CPUrate;             //Testing CPU instruction rate (MIPS)
  float Instructions;        //Average object instructions per line
  float Size;                //Code size in KLOC
  float Time;		     //Time to run the model for
  double Alpha;              //These are parameters supplied by the user
  int EstType;

  StaticMetrics(void);
  double DefectDensity(void);
};

/*******************************************************************
  Class : FailureDataPoint
  Purpose : Encapsulates a single data point in a coverage data set.
  This is primarly a helper class for CoverageDataSet.
*******************************************************************/
class CoverageDataPoint
{
 public:
  double faults; //The number of faults found
  double branch; //These remaining four members are percentage of
  double block; //of coverage obtained when a particular fault was found
  double p_use;
  double c_use;

  double map(int); //returns the value of the corresponding enumerable
};

/**********************************************************************
  Class : CoverageDataSet
  Purpose : CoverageDataSet encapsulates coverage information about a
  program.
**********************************************************************/
class CoverageDataSet {
public:
  int Size; //The number of data points available
  CoverageDataPoint* cov_data; //Array of data points
  int avail[5]; //boolean array telling if an enumerable is available
  char* FileName; //The name of the file the data was read from, if any

  CoverageDataSet(int = 0); //Constructor
  ~CoverageDataSet(void); //Destructor
  int OpenFile(char*); //Routine to read in a coverage data set from CDS file
  int Shrink(void); //Reduce memory used to only what is necessary
};

/**************************************************************************
  Sturcture : Parameters
  Purpose : Parameters maintains sate information about the robust program.
  It stores information about the last modeling function used, and what
  data sets are currently being displayed, and the parameters used by the
  models, ect... It is a convient place to put variables without having to
  make them global. (Note: All V programs have global access theApp, a
  pointer to the application class. The robust version of this class contains
  the only instance of this variable, so they are in fact global.)
**************************************************************************/

struct Parameters {
  char title[16];
  char mname[24];
  double beta0,beta1; //Parameters used to fit SRGM models
  double alpha0,alpha1,alpha2; //Parameters used to fit coverage models
  double cntbias,intenbias; //Bias of the models
  double mre; // Error of the SRGM model
  modtype mtype; //Type of model currently in use
  int recalibrated; //Wheather or not the model is recalibrated
  int recalibrate; //Tells whether or not new models should be recalibrated
  int stabilize; //Tells whether or not new models should be stabilized
  double laplace; //The trend present in the data
  int data1,data2; //The data sets on display in the text windows of robust
  //Pointers to the modeling functions currently in use
  void (*modfunc)(double, double, double, double&, double&);
  int (*fitfunc)(FailureDataSet*, double&, double&);
  void (*predfunc)(double, double, double&, double&, double&);
  int (*statfunc)(StaticMetrics,double&,double&);
};
#endif






