/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * debug.c ---	analyze compiled code
 * (duz 26Aug93)
 */

#include <ctype.h>
#include <string.h>

#include "config.h"
#include "term.h"
#include "forth.h"
#include "support.h"
#include "compiler.h"


/*****************************************************************************/
/* decompiler								     */
/*****************************************************************************/

#ifdef WRONG_SPRINTF		/* provision for buggy sprintf (SunOS) */
# define SPRFIX(X) strlen(X)
#else
# define SPRFIX(X) X
#endif

static int locals;

Xt *
decompile_word (Xt *ip, char *p, Decomp *d)
{
  static Decomp default_style = { SKIPS_NOTHING, 0, 0, 0, 0 };
  Xt xt = *ip++;
  Semant *s;
  char *nfa, buf [80];

  s = to_semant (xt);
  *d = s ? s->decomp : default_style;
  if (*xt == literal_execution_)
    {
      strcpy (p, str_dot (*(Cell *)ip, buf + sizeof buf, BASE));
      return ++ip;
    }
  if (*xt == locals_bar_execution_)
    {
      int i;

      locals += *(Cell *)ip;
      p += SPRFIX (sprintf (p, "LOCALS| "));
      for (i = locals; --i >= 0; )
	p += SPRFIX (sprintf (p, "<%c> ", 'A' - 1 + locals - i));
      p += SPRFIX (sprintf (p, "| "));
      return ++ip;
    }
  if (*xt == local_execution_)
    {
      sprintf (p, "<%c> ", 'A' + 1 + locals - *(Cell *)ip);
      return ++ip;
    }
  if (*xt == to_local_execution_)
    {
      sprintf (p, "TO <%c> ", 'A' + 1 + locals - *(Cell *)ip);
      return ++ip;
    }
  if (*xt == plus_to_local_execution_)
    {
      sprintf (p, "+TO <%c> ", 'A' + 1 + locals - *(Cell *)ip);
      return ++ip;
    }
  if (s == NULL)
    {
      nfa = to_name (xt);
      sprintf (p, *nfa & IMMEDIATE ? "POSTPONE %.*s " : "%.*s ",
	       *nfa & 0x1F, nfa + 1);
      return ip;
    }
  else
    nfa = s->name;
  switch (d->skips)
    {
    case SKIPS_CELL:
    case SKIPS_OFFSET:
      INC (ip, Cell);
    default:
      sprintf (p, "%.*s ", *nfa & 0x1F, nfa + 1);
      return ip;
    case SKIPS_DCELL:
      sprintf (p, "%s. ",
	       str_d_dot_r (*(dCell *)ip, buf + sizeof buf, 0, BASE));
      INC (ip, dCell);
      return ip;
    case SKIPS_FLOAT:
#if FLOAT_ALIGN > CELL_ALIGN
      if (!DFALIGNED (ip))
	ip++;
#endif
      sprintf (p, "%g ", *(double *)ip);
      INC (ip, double);
      return ip;
    case SKIPS_STRING:
      sprintf (p, "%.*s %.*s\" ",
	       *nfa & 0x1F, nfa + 1,
	       (int)*(Byte *)ip, (Byte *)ip + 1);
      SKIP_STRING;
      return ip;
    case SKIPS_2STRINGS:
      {
	Byte *s1 = (Byte *)ip;
	SKIP_STRING;
	sprintf (p, "%.*s %.*s %.*s ",
		 *nfa & 0x1F, nfa + 1, (int)*s1, s1 + 1,
		 (int)*(Byte *)ip, (Byte *)ip + 1);
	SKIP_STRING;
	return ip;
      }
    }
}

void
decompile (Xt *ip)
{
  char buf [0x80];
  Seman2 *s;
  Decomp d;
  int indent = 4, nl = 0;

  cr_();
  start_question_cr_();
  spaces (indent);
  locals = 0;
  for (;;)
    {
      s = (Seman2 *)to_semant (*ip);
      ip = decompile_word (ip, buf, &d);
      indent += d.ind_bef;
      if ((!nl && d.cr_bef) || OUT + strlen (buf) >= cols)
	{
	  if (question_cr ())
	    break;
	  nl = 1;
	}
      if (nl)
	{
	  spaces (indent);
	  nl = 0;
	}
      outs (buf);
      spaces (d.space);
      indent += d.ind_aft;
      if (d.cr_aft)
	{
	  if (question_cr ())
	    break;
	  nl = 1;
	}
      if (s == &semicolon_semantics)
	break;
    }
}


/*****************************************************************************/
/* debugger								     */
/*****************************************************************************/

static int debugging, level, maxlevel;

char
category (pCode p)
{
  if (p == colon_runtime
   || p == debug_colon_runtime)		return ':';
  if (p == create_runtime)		return 'v';
  if (p == constant_runtime)		return 'c';
  if (p == sysvar_runtime)		return 's';
  if (p == sysconst_runtime)		return 'c';
  if (p == vocabulary_runtime)		return 'w';
  if (p == does_defined_runtime
   || p == debug_does_defined_runtime)	return 'd';
  if (p == marker_runtime)		return 'm';
  /* must be primitive */		return 'p';
}

static void
display (Xt *ip)
{
  Decomp style;
  char buf [80];
  int indent = maxlevel * 2;
  int width = 32 - indent;
  int depth = sys.s0 - sp, i;

  for (i = 0; i < depth; i++)
    {
      outf ("%10ld ", (long)sp [i]);
      if (OUT + 11 >= cols)
	break;
    }
  cr_();
  decompile_word (ip, buf, &style);
  outf ("%*s%c %-*.*s", indent, "", category (**ip), width, width, buf);
}

static void
interaction (Xt *ip)
{
  int c;

  for (;;)
    {
      outs ("> ");
      c = getekey ();
      c = c < ' ' ? c + '@' : toupper (c);
      backspace_();
      backspace_();
      switch (c)
	{
	default:
	  bing ();
	  continue;
	case EKEY_kd:
	case 'M':
	case 'J':
	case 'X':
	  return;
	case EKEY_kr:
	case 'D':
	  maxlevel++;
	  return;
	case EKEY_kl:
	case 'S':
	  maxlevel--;
	  return;
	case 'Q':
	  debugging = 0;
	  tHrow (THROW_QUIT);
	case 'H':
	case '?': outf ("\nArrow down, 'X', CR:\texecute word"
			"\nArrow right, 'D':\tsingle step word"
			"\nArrow left, 'S':\tfinish word"
			" w/o single stepping"
			"\n'Q'\t\t\tQUIT"
			"\n'?', 'H':\t\tthis message");
	  display (ip);
	}
    }
}

static void			/* modfied inner interpreter for */
single_step (void)		/* single stepping */
{
  spaces (34 - OUT);
  for (;;)
    {
      if (level <= maxlevel)
	{
	  maxlevel = level;
	  display (ip);
	  interaction (ip);
	}
      w = *ip++;
      if (*w == colon_runtime ||
	  *w == debug_colon_runtime ||
	  *w == does_defined_runtime ||
	  *w == debug_does_defined_runtime)
	level++;
      else if (*w == semicolon_execution_ ||
	       *w == locals_exit_execution_)
	if (--level < 0)
	  {
	    (**w) ();
	    return;
	  }
      (**w) ();
    }
}

void
debug_colon_runtime (void)
{
  colon_runtime ();
  if (!debugging)
    {
      debugging = 1;
      locals = 0;
      level = maxlevel = 0;
      single_step ();
      debugging = 0;
    }
}

void
debug_does_defined_runtime (void)
{
  does_defined_runtime ();
  if (!debugging)
    {
      debugging = 1;
      locals = 0;
      level = maxlevel = 0;
      single_step ();
      debugging = 0;
    }
}

Code (debug)
{
  Xt xt;

  tick (&xt);
  if (*xt == debug_colon_runtime ||
      *xt == debug_does_defined_runtime)
    return;
  if (*xt == colon_runtime)
    *xt = debug_colon_runtime;
  else if (*xt == does_defined_runtime)
    *xt = debug_does_defined_runtime;
  else
    tHrow (THROW_ARG_TYPE);
}

Code (no_debug)
{
  Xt xt;

  tick (&xt);
  if (*xt == debug_colon_runtime)
    *xt = colon_runtime;
  else if (*xt == debug_does_defined_runtime)
    *xt = does_defined_runtime;
  else
    tHrow (THROW_ARG_TYPE);
}


LISTWORDS (debug) =
{
  CO ("DEBUG",		debug),
  CO ("NO-DEBUG",	no_debug)
};
COUNTWORDS (debug, "Debugger words");
