; struct.lsp -- structures for OS2XLISP 
; Andrew Schulman 11-June-1988

;======================================================================
; helper routines

; write non-recursive version later
; this interprets 0 as 4 (string pointers)
(define (sum lst)
    (cond
        ((null lst)
            0)
        (t
            (+
                (if (zerop (car lst)) 4 (car lst))
                (sum (cdr lst))))))

; write non-recursive version later             
; can flatten assoc-lists into property-lists
(define (flatten lst)
    (cond
        ((null lst)
            nil)
        ((atom lst)
            (list lst))
        (t
            (append
                (flatten (car lst))
                (flatten (cdr lst))))))

; write non-recursive version later             
(define (make-list length init)
    ;;; (print (sp)) ;;; check stack 
    (if (zerop length)
        nil
        (cons init (make-list (1- length) init))))
            
(define (conv s)
    (case s
        ((0 1 2 4 8)        s)
        ((str string)       0)
        ((byte char)        1)
        ((word int)         2)
        ((long fixnum ptr)  4)
        ((float double)     8)
        (t (error "Bad structure element type"))))
            
(define (convert-template template flag)
    (mapcar
        (lambda (size)
            (cond
                ((and flag (listp size))
                    ; convert size, keep field name
                    (list (conv (car size)) (cadr size)))
                ((listp size)
                    ; else turn any lists into size
                    (conv (car size)))
                (t
                    ; else turn retval directive into size
                    (conv size))))
        template))

;======================================================================
(define (make-struct template &optional data)
    (let*
        ((template (convert-template template nil))
         (str (make-string 32 (sum template)))  
         (offset 0))
        (if data
            (if (not (= (length template) (length data)))
                (error "MAKE-STRUCT: template list/data list mismatch"))
        ; else
            (setf data (make-list (length template) 0)))
        (mapcar
            (lambda (size info)
                ;;; (format stdout "~A ~A\n" size info) ;;; debugging
                (if (zerop size) (setf size 4))
                (poke (+ ^str offset) info size)
                (setf offset (+ offset size)))
            template
            data)
        str))
            
;;; example
;;; (define str
;;;     (make-struct
;;;         '(word  long    word    long)       ; template
;;;         '(111   2       3       666)))      ; data

(define (unpack-struct template str)
    (let
        ((template (convert-template template t))
         (offset 0)
         (s 0)
         (info 0)
         (addr (if (eq 'STRING (type-of str)) ^str str)))
        (mapcar
            (lambda (size)
                (setf s (if (listp size) (car size) size))
                (prog2
                    (setf info
                        (let
                            ((a (+ addr offset)))
                            (if (and (zerop s) (not (zerop (peek a 4))))
                                (peek (peek a 4) 0)
                                (peek a s))))
                    (if (listp size)
                        (list (cadr size) info)
                        info)
                    (setf offset
                        (+ offset (if (zerop s) 4 s)))))
            template)))

;;; note how combination of (mapcar) and (prog2) takes care of setting
;;; up return list

;;; example
;;; (unpack-struct '(2 4 2 4) str) ===> (111 2 3 666)

;======================================================================
;;; example -- also shows optional field names in structure template

;(define PIDINFO
;    '((word PID)
;      (word TID)
;      (word PIDPAR)))
;(define pi (make-struct PIDINFO))
;(if (call (getprocaddr doscalls "DOSGETPID") pi t)
;    (print (unpack-struct PIDINFO pi)))

;;; (unpack-struct) can be used on any area of memory
;(define (show-ldt-info-seg)
;    (unpack-struct '(2 2 2 2 2 2 2 1) (mk-fp ldt 0)))
        
;;; note that (make-list) can be used to create monotonous templates
;;; (for arrays, rather than for heterogeneous structures):
;(define (show-ldt-info-seg)
;    (unpack-struct (make-list 7 2) (mk-fp ldt 0)))

;(define (show-gdt-info-seg)
;    (unpack-struct
;        '(4 4 1 1 1 1 2 2 1 1 2 1 1 1 1 1 1 1 1 2 1 1 2 2 2)
;        (mk-fp gdt 0)))

;======================================================================
; now the fieldname can be returned as part of the list:  want to make a
; Lisp property list, but for now we have association lists.  E.g.:
;  (define ldt-struct
;   '((word PID)
;     (word PARENT)
;     (word PRIORITY)
;     (word THREAD)
;     (word SESSION)
;     (word SUBSESSION)
;     (word FOREGROUND?)
;     (byte REALMODE)))
; (define ldt-info (unpack-struct ldt-struct (mk-fp ldt 0)))
; > (assoc 'pid ldt-info)
; (PID 15)
; > (assoc 'foreground? ldt-info)
; (FOREGROUND? 65535)

