/* 
 * tclOS2Env.c --
 *
 *	This file defines the TclSetSystemEnv function.
 *
 * Copyright (c) 1996-1998 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#include "tclOS2Int.h"

/* peek into EMX's DLL */
extern char **_org_environ;

/*
 *----------------------------------------------------------------------
 *
 * TclSetSystemEnv --
 *
 *	Set an environment variable, or remove it from the environment if
 *	the value is NULL.
 *
 * Results:
 *	0 on sucess, -1 on failure.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
int
TclSetSystemEnv( const char *variable, const char *value )
{
    char *string;
    int ret = 0;

    if (variable == (const char *)NULL) {
        errno = EINVAL;
        return -1;
    }
    if (value == (const char *)NULL) {
        /*
         * Since EMX doesn't offer a special putenv syntax to remove an env.
         * variable (like IBM VA C++, giving just the name without '=' and the
         * value), we have to modify the environ global array ourselves.
         * This is based on code nicked from EMX, \emx\src\lib\app\putenv.c.
         */
        char *s, **p;
        int len, env_size;

        /* Find the variable */
        p = _environ;
        len = strlen(variable);
        env_size = 0;
        if (p != NULL) {
            while (*p != NULL) {
                s = *p;
                if (strncmp (s, variable, len) == 0 &&
                    (s[len] == 0 || s[len] == '=')) {
                    break;
                }
                ++p; ++env_size;
            }
        }
        /* If exists, then update environ */
        if (_environ == _org_environ) {
            /* This is the original runtime environment, make a copy */
            char **ptr = p;
            int complete = 0;

            while (*ptr != NULL) {
                ++ptr; ++complete;
            }
            _environ = malloc ((complete+2) * sizeof (char *));
            if (_environ == NULL) return -1;
            if (complete != 0)
              memcpy (_environ, _org_environ, complete * sizeof (char *));
            _environ[complete+1] = NULL;
            p = _environ + (p - _org_environ);
        }
        if (p != NULL && *p != NULL) {
            /* found */
            /* copy all following entries one position to the top */
            while (*p != NULL) {
                *p = *(p+1);
                ++p; ++env_size;
            }
            /* realloc so the last entry falls off */
            _environ = realloc (_environ, (env_size+1) * sizeof (char *));
            if (_environ == NULL) return -1;
        }
    } else {
        string= malloc(strlen(variable) +
                       1 +			/* = sign */
                       ((value == (const char *)NULL) ? 0 : strlen(value)) +
                       1);			/* \0 termination */
        if (string == (char *)NULL) return -1;
        ret = sprintf(string, "%s=%s", variable,
                      (value == (const char *)NULL) ? "" : value);
        if (ret == 0 || ret == EOF) {
            free(string);
            return -1;
        }
        ret= putenv(string);
        /* String may NOT be freed or reused */
    }
    return ret;
}
