/* 
 * tkOS2Image.c --
 *
 *	This file contains routines for manipulation full-color of images.
 *
 * Copyright (c) 1996-2000 Illya Vaes
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#include "tkOS2Int.h"

static int             DestroyImage _ANSI_ARGS_((XImage* data));
static unsigned long   ImageGetPixel _ANSI_ARGS_((XImage *image, int x, int y));
static int             PutPixel (XImage *image, int x, int y,
			    unsigned long pixel);

/*
 *----------------------------------------------------------------------
 *
 * DestroyImage --
 *
 *      This is a trivial wrapper around ckfree to make it possible to
 *      pass ckfree as a pointer.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Deallocates the image.
 *
 *----------------------------------------------------------------------
 */

int
DestroyImage(imagePtr)
     XImage *imagePtr;          /* image to free */
{
    if (imagePtr) {
        if (imagePtr->data) {
            ckfree((char*)imagePtr->data);
        }
        ckfree((char*)imagePtr);
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * ImageGetPixel --
 *
 *      Get a single pixel from an image.
 *
 * Results:
 *      Returns the 32 bit pixel value.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

unsigned long
ImageGetPixel(image, x, y)
    XImage *image;
    int x, y;
{
    unsigned long pixel = 0;
    unsigned char *srcPtr = &(image->data[(y * image->bytes_per_line)
            + ((x * image->bits_per_pixel) / NBBY)]);

#ifdef VERBOSE
    printf("ImageGetPixel %x (%d,%d)\n", image, x, y);
#endif
/*
    switch (image->bits_per_pixel) {
        case 32:
        case 24:
            pixel = RGB(srcPtr[2], srcPtr[1], srcPtr[0]);
            break;
        case 16:
            pixel = RGB(((((WORD*)srcPtr)[0]) >> 7) & 0xf8,
                    ((((WORD*)srcPtr)[0]) >> 2) & 0xf8,
                    ((((WORD*)srcPtr)[0]) << 3) & 0xf8);
            break;
        case 8:
            pixel = srcPtr[0];
            break;
        case 4:
            pixel = ((x%2) ? (*srcPtr) : ((*srcPtr) >> 4)) & 0x0f;
            break;
        case 1:
            pixel = ((*srcPtr) & (0x80 >> (x%8))) ? 1 : 0;
            break;
    }
*/
    pixel = RGB(srcPtr[0], srcPtr[1], srcPtr[2]);
    return pixel;
}

/*
 *----------------------------------------------------------------------
 *
 * PutPixel --
 *
 *	Set a single pixel in an image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
PutPixel(image, x, y, pixel)
    XImage *image;
    int x, y;
    unsigned long pixel;
{
    unsigned char *destPtr;

#ifdef VERBOSE
    printf("PutPixel %x image %x (%d,%d) bytes_p_l %d, bits_p_p %d\n", pixel,
           image, x, y, image->bytes_per_line, image->bits_per_pixel);
#endif
    destPtr = &(image->data[(y * image->bytes_per_line)
              + ((x * image->bits_per_pixel) / NBBY)]);

/*
    rc = GpiQueryLogColorTable(globalPS, 0L, pixel, 1, &pixel);
*/
    destPtr[0] = destPtr[1] = destPtr[2] = 0;
    destPtr[0] = GetRValue(pixel);
    destPtr[1] = GetGValue(pixel);
    destPtr[2] = GetBValue(pixel);
#ifdef VERBOSE
    printf(" pixel now %x, RGB (%x,%x,%x) destPtr %x\n", pixel, destPtr[0],
           destPtr[1], destPtr[2], (ULONG)destPtr);
#endif

/*
    LONG *destPtr;

    destPtr = (LONG *) &(image->data[(y * image->bytes_per_line)
              + ((x * image->bits_per_pixel) / NBBY)]);
    *destPtr = pixel;
#ifdef VERBOSE
    printf("PutPixel %x image %x (%d,%d) bytes_p_l %d, bits_p_p %d: %x\n",
           pixel, image, x, y, image->bytes_per_line, image->bits_per_pixel,
           *(destPtr));
#endif
*/

    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * XCreateImage --
 *
 *	Allocates storage for a new XImage.
 *
 * Results:
 *	Returns a newly allocated XImage.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

XImage *
XCreateImage(display, visual, depth, format, offset, data, width, height,
	bitmap_pad, bytes_per_line)
    Display* display;
    Visual* visual;
    unsigned int depth;
    int format;
    int offset;
    char* data;
    unsigned int width;
    unsigned int height;
    int bitmap_pad;
    int bytes_per_line;
{
    XImage* imagePtr = (XImage *) ckalloc(sizeof(XImage));

    if (imagePtr) {
        imagePtr->width = width;
        imagePtr->height = height;
        imagePtr->xoffset = offset;
        imagePtr->format = format;
        imagePtr->data = data;
        imagePtr->byte_order = MSBFirst;
        imagePtr->bitmap_unit = 32;
        imagePtr->bitmap_bit_order = MSBFirst;
        imagePtr->bitmap_pad = bitmap_pad;
        imagePtr->bits_per_pixel = 24;
        imagePtr->depth = depth;

        /*
         * Bitmap_pad must be on 4-byte boundary.
         */

#define LONGBITS    (sizeof(LONG) * 8)

        bitmap_pad = (bitmap_pad + LONGBITS - 1) / LONGBITS * LONGBITS;
#ifdef VERBOSE
    printf("XCreateImage bpp %d, depth %d, pad %d (was %d)\n",
           imagePtr->bits_per_pixel, imagePtr->depth, imagePtr->bitmap_pad,
           bitmap_pad);
#endif

        /*
         * Round to the nearest bitmap_pad boundary.
         */

        if (bytes_per_line) {
            imagePtr->bytes_per_line = bytes_per_line;
        } else {
            imagePtr->bytes_per_line = (((depth * width)
                    + (bitmap_pad - 1)) >> 3) & ~((bitmap_pad >> 3) - 1);
        }

        imagePtr->red_mask = 0;
        imagePtr->green_mask = 0;
        imagePtr->blue_mask = 0;

        imagePtr->f.put_pixel = PutPixel;
        imagePtr->f.get_pixel = ImageGetPixel;
        imagePtr->f.destroy_image = DestroyImage;
        imagePtr->f.create_image = NULL;
        imagePtr->f.sub_image = NULL;
        imagePtr->f.add_pixel = NULL;
    }
    
    return imagePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * XGetImage --
 *
 *      This function copies data from a pixmap or window into an
 *      XImage.
 *
 * Results:
 *      Returns a newly allocated image containing the data from the
 *      given rectangle of the given drawable.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

XImage *
XGetImage(display, d, x, y, width, height, plane_mask, format)
    Display* display;
    Drawable d;
    int x;
    int y;
    unsigned int width;
    unsigned int height;
    unsigned long plane_mask;
    int format;
{
    TkOS2Drawable *todPtr = (TkOS2Drawable *)d;
    XImage *imagePtr;
    BITMAPINFO2 infoBuf;

#ifdef VERBOSE
    printf("XGetImage\n");
#endif

    if ((todPtr->type != TOD_BITMAP) || (todPtr->bitmap.handle == NULLHANDLE)
            || (format != XYPixmap) || (plane_mask != 1)) {
        panic("XGetImage: not implemented");
    }


    imagePtr = XCreateImage(display, NULL, 1, XYBitmap, 0, NULL,
            width, height, 32, 0);
    imagePtr->data = ckalloc(imagePtr->bytes_per_line * imagePtr->height);

    infoBuf.cbFix = 20;
    infoBuf.cx = width;
    infoBuf.cy = height;
    infoBuf.cPlanes = 1;
    infoBuf.cBitCount = 1;
    infoBuf.ulCompression = BCA_UNCOMP;

    rc = GpiQueryBitmapBits(todPtr->bitmap.hps, 0L, (LONG)height,
                            (PBYTE)imagePtr->data, &infoBuf);

    return imagePtr;
}
