/******************************************************************
 * WILDCARD.CMD
 *
 * This program calls the various REXX external functions provided in the FILEREXX DLL.
 *******************************************************************/


/* The FileLoadFuncs loads all the rest of the REXX functions in the FILEREXX DLL. */
/* So, we don't have to make calls to RxFuncAdd to add each one of those functions. Of */
/* course, in order to call FileLoadFuncs, we have to add that one. */
CALL RxFuncAdd 'FileLoadFuncs', 'FILEREXX', 'FileLoadFuncs'
CALL FileLoadFuncs

/* Let's assume that this script reads in some existing file, manipulates it, and writes
out the new changes to a file.	The script wants the user to type the name of the file to
read in, as an arg on the command line when this script is run.  This must be an existing
file.  Furthermore, the user can optionally supply the name of the file to write out the new
changes to.  If he doesn't supply this, then it defaults to the same name as the existing
file (ie, overwrites it).  The script wants to allow the user to use wildcards for the source
and destination filenames.  For example, if he supplied the following args&colon.

*.c  c:\temp\*.bak

then this should read in each file (in the current directory) that ends with a :hp9..c:ehp9.
extension, perform an operation upon it, and write it out in the C&colon.\temp directory,
changing the file's extension to :hp9..bak:ehp9..  Here's how to implement the script.
*/

/* Get the fromname and toname that the user typed as args on the command line. */
arg fromname toname

/* Separate fromname into separate pieces. Set the flag to check whether it's a file or
directory. This script only wants to perform work upon files.
*/
err = FileGetPath('Source', fromname, 'CHK')
IF err = 0 THEN DO

    /* Now remake the fromname from its pieces. This will ensure that, if the user specified
	a dir for the source, our fromname will end with a '\' character, which ensures that
	FileMatchFile enumerates all of the files within that directory */
    fromname = Source.0||Source.1||Source.2||Source.3

    /* If toname was not supplied, default to the same as fromname.
    */
    IF toname = '' THEN DO
	toname = fromname
    END

    ELSE DO
    /* Let's also chop up the toname into pieces, checking if it's a directory, and remake
	the toname from its pieces. This will ensure that, if the user specified a dir for the
	toname, our toname will end with a '\' character, which ensures that FileEditName
	enumerates all of the matching files to that directory */
	err = FileGetPath('File', toname, 'CHK')
	IF err = 0 THEN toname = File.0||File.1||File.2||File.3

	/* Error */
	ELSE DO
	    SAY 'ERROR checking "' toname '":' err
	    SIGNAL OutOfHere
	END
    END

    /* Since the fromname can contain wildcards, we'll pass it to FileMatchFile so that we
	can enumerate each and every match. If fromname doesn't contain any wildcards
	(ie, it's simply the name of a specific file), this will only loop once anyway on that
	one fromname.  Note that we limit the search to read/write files, readonly files,
	and files with the archive bit set.  We ignore subdirectories, system, and hidden
	files since this script doesn't care about those.
    */

    /* Initially set handle to 0 */
    Handle = 0

    DO UNTIL Handle = 0

	/* Locate the next match to fromname */
	err = FileMatchFile('File', 'Handle', fromname, 'RDO|ARC')

	/* No error? (ie, another match) */
	IF err = 0 THEN DO

	    /* Since the toname can contain wildcards, we need to apply it as a template to the matched
		source (ie, minus the drive and directory of source, and using the drive and dir of the toname).
	    */
	    NewFile = FileEditName(File, toname, 'SKIPDIR')
	    IF NewFile <> '' THEN DO

		/* Add the original source drive and directory to the matched source */
		File = Source.0||Source.1||File

		/* Here, we would read in the file (name is in File), operate upon it, and write it back out using the
		    filename in NewFile.  Just for testing purposes, we'll just write out these 2 filenames.
		*/
		SAY '============================================='
		SAY 'Source =' File
		SAY 'New file =' NewFile
	    END

	    /* An error occurred during creating the NewFile name */
	    ELSE SAY 'ERROR making new filename.'

	END

	/* An error occurred during matching the source */
	ELSE IF err <> 18 THEN SAY 'ERROR locating "' fromname '":' err

    END
END

/* Error */
ELSE SAY 'ERROR checking "' fromname '":' err

OutOfHere:

/* =============================================================================== */
/* FileDropFuncs: This unloads all of the functions in the FILEREXX DLL. This is not necessary, and
    we could otherwise leave it open for some other REXX script */
CALL FileDropFuncs
