/* rxImgSize */

#define  INCL_DOS
#define  INCL_REXXSAA
#define  __REXXSAA_H_INCLUDED
#define  VAR_RX
#define INCL_PM
#define INCL_DOSERRORS   /* DOS error values */
#include <os2.h>

#include <stdio.h>
#include <ctype.h>
#include <stddef.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "gbm.h"
#include "gbmmem.h"
#include "gbmscale.h"
#include "gbmtool.h"

/* own replacement for string.h */
#include "var.hpp"

#ifdef __cplusplus
extern "C" {
#endif

    typedef struct FILTER_NAME_TABLE_DEF
    {
        char * name;   /* filter name */
        GBM_SCALE_FILTER filter; /* filter type */
    };

    static const int GBM_SCALE_FILTER_SIMPLE = 0;

    static FILTER_NAME_TABLE_DEF FILTER_NAME_TABLE [] =
    { //{ "simple"         , GBM_SCALE_FILTER_SIMPLE },
        { "nearestneighbor", GBM_SCALE_FILTER_NEARESTNEIGHBOR },
        { "bilinear"       , GBM_SCALE_FILTER_BILINEAR        },
        { "bell"           , GBM_SCALE_FILTER_BELL            },
        { "bspline"        , GBM_SCALE_FILTER_BSPLINE         },
        { "mitchell"       , GBM_SCALE_FILTER_MITCHELL        },
        { "lanczos"        , GBM_SCALE_FILTER_LANCZOS         },
        { "blackman"       , GBM_SCALE_FILTER_BLACKMAN        },
        { "catmullrom"     , GBM_SCALE_FILTER_CATMULLROM      },
        { "quadratic"      , GBM_SCALE_FILTER_QUADRATIC       },
        { "gaussian"       , GBM_SCALE_FILTER_GAUSSIAN        },
        { "kaiser"         , GBM_SCALE_FILTER_KAISER          }
    };

    const int FILTER_NAME_TABLE_LENGTH = sizeof( FILTER_NAME_TABLE ) / sizeof( FILTER_NAME_TABLE[0] );

#include <rexxsaa.h>


    /*********************************************************************/
    /* .DLL version                                                 */
    /*********************************************************************/
#define IMG_VERSION_REXX     "1.02"

    /*********************************************************************/
    /* Numeric Return calls                                              */
    /*********************************************************************/
#define  INVALID_ROUTINE 40            /* Raise Rexx error           */
#define  VALID_ROUTINE    0            /* Successful completion      */


    /*********************************************************************/
    /*  Declare all exported functions as REXX functions.                */
    /*********************************************************************/
    APIRET APIENTRY __export rxImgLoadFuncs( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr );
    APIRET APIENTRY __export rxImgDropFuncs( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr );

    APIRET APIENTRY __export rxImgVersion( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr );
    APIRET APIENTRY __export rxImgReSize( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr );
    APIRET APIENTRY __export rxImgCapture( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr );

#ifdef __cplusplus
}
#endif

/*********************************************************************/
/* RxFncTable                                                        */
/*   Array of names of the REXXUTIL functions.                       */
/*   This list is used for registration and deregistration.          */
/*********************************************************************/

static PSZ RxFncTable[] =
{
    "rxImgLoadFuncs",
    "rxImgDropFuncs",
    "rxImgVersion",
    "rxImgReSize",
    "rxImgCapture"
};

BOOL isBetween( int pre, int test, int post ) {
    return pre < test & test < post;
}

BOOL QueryExist( var &fName ) {

    HFILE  hfFileHandle   = 0L;     /* Handle for file being manipulated */
    ULONG ulAction        = 0;      /* Action taken by DosOpen */

    int rc = DosOpen( fName,                    /* File path name */
                     &hfFileHandle,                  /* File handle */
                     &ulAction,                      /* Action taken */
                     0L,                           /* File primary allocation */
                     0L,    /* File attribute */
                     OPEN_ACTION_FAIL_IF_NEW |
                     OPEN_ACTION_OPEN_IF_EXISTS,  /* Open an existing file only   */
                     OPEN_FLAGS_NOINHERIT | OPEN_ACCESS_READONLY |
                     OPEN_SHARE_DENYNONE,
                     0L );
    if ( rc == 0 )
        rc = DosClose( hfFileHandle );      /* Close the file  (check rc in real life) */

    return ( ulAction == FILE_EXISTED );                            /* No extended attribute */
}

static int getFilterIndex( var &s, const struct FILTER_NAME_TABLE_DEF *table, const int tableLength )
{
    for ( int n = 0; n < tableLength; n++ )
        if ( s.cmp( table[n].name ) > 0, FALSE )
            return n;

    return GBM_SCALE_FILTER_SIMPLE;
}

static BOOL isGrayscalePalette( const GBMRGB *gbmrgb, const int entries )
{
    if ( ( entries > 0 ) && ( entries <= 0x100 ) )
    {
        for ( int i = 0; i < entries; i++ )
            if ( ( gbmrgb[i].r != gbmrgb[i].g ) || ( gbmrgb[i].r != gbmrgb[i].b ) || ( gbmrgb[i].g != gbmrgb[i].b ) )
                return FALSE;
        return TRUE;
    }
    return FALSE;
}

static BOOL expandTo24Bit( GBM *gbm, GBMRGB *gbmrgb, BYTE **ppbData, int *dataLen )
{
    int stride = ( ( gbm->w * gbm->bpp + 31 ) / 32 ) * 4;
    int new_stride = ( ( gbm->w * 3 + 3 ) & ~3 );
    int bytes, y;
    BYTE *pbDataNew;

    if ( gbm->bpp == 24 )
        return TRUE;

    bytes = new_stride * gbm->h;

    pbDataNew = ( BYTE * )gbmmem_malloc( bytes );
    if ( pbDataNew == NULL )
        return FALSE;

    for ( y = 0; y < gbm -> h; y++ )
    {
        BYTE *src = *ppbData + y * stride;
        BYTE *dest = pbDataNew + y * new_stride;
        int   x;

        switch ( gbm -> bpp )
        {
        case 1:
            {
                BYTE  c;
                for ( x = 0; x < gbm -> w; x++ )
                {
                    if ( ( x & 7 ) == 0 )
                        c = *src++;
                    else
                        c <<= 1;

                    *dest++ = gbmrgb [(c & 0x80) != 0].b;
                    *dest++ = gbmrgb [(c & 0x80) != 0].g;
                    *dest++ = gbmrgb [(c & 0x80) != 0].r;
                }
            }
            break;

        case 4:
            for ( x = 0; x + 1 < gbm -> w; x += 2 )
            {
                BYTE c = *src++;

                *dest++ = gbmrgb [c >> 4].b;
                *dest++ = gbmrgb [c >> 4].g;
                *dest++ = gbmrgb [c >> 4].r;
                *dest++ = gbmrgb [c & 15].b;
                *dest++ = gbmrgb [c & 15].g;
                *dest++ = gbmrgb [c & 15].r;
            }

            if ( x < gbm -> w )
            {
                BYTE c = *src;

                *dest++ = gbmrgb [c >> 4].b;
                *dest++ = gbmrgb [c >> 4].g;
                *dest++ = gbmrgb [c >> 4].r;
            }
            break;

        case 8:
            for ( x = 0; x < gbm -> w; x++ )
            {
                BYTE c = *src++;

                *dest++ = gbmrgb [c].b;
                *dest++ = gbmrgb [c].g;
                *dest++ = gbmrgb [c].r;
            }
            break;
        }
    }

    gbmmem_free( *ppbData );
    *ppbData = pbDataNew;
    *dataLen = bytes;

    gbm->bpp = 24;

    return TRUE;
}

/*********************************************************************/
/* Function:  rxImgVersion                                           */
/*                                                                   */
/* Syntax:    version = rxImgVersion()                               */
/*                                                                   */
/* Return:    Version number: "major.minor"                          */
/*********************************************************************/

APIRET APIENTRY __export rxImgVersion( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr )
{
    var version, opts;
    if ( numargs > 0 )
        if ( args[0].strptr != NULL )
            opts.cpy( args[0].strptr );

    if ( numargs == 0 ) {
        Gbm_init();
        version.cat( "rxImgSize: version " ).cat( IMG_VERSION_REXX ).cat( ", GBM: version " ).cat( ( Gbm_version() / 100.0 ) );
        Gbm_deinit();
    } else {
        if ( opts.wordBeginsWith( "R", " ", FALSE )  )
            version.cpy( IMG_VERSION_REXX );
        else if ( opts.wordBeginsWith( "G", " ", FALSE ) ) {
            Gbm_init();
            version.cpy( ( Gbm_version() / 100.0 ) );
            Gbm_deinit();
        }
    }

    /* build return string ( "rxImgSize: v. major.minor GBM: v. major.minor" )*/
    version.set( retstr );

    return VALID_ROUTINE;  /* no error on call */
}

/********************************************************************************/
/* Function:  rxImgReSize                                                       */
/*                                                                              */
/* Syntax:    rc = rxImgReSize( srcfName, tgtfName, w, h, opt, opt_in, opt_out )*/
/*                                                                              */
/* Params:    string  in - Path and Name of file to resize                      */
/*                                                                              */
/*( optional )string  in - Path and Name of file to save to                     */
/*                         source by default                                    */
/*                                                                              */
/*( optional )string  in - Width of target file                                 */
/*                                                                              */
/*( optional )string  in - Height of target file                                */
/*                                                                              */
/*( optional )string  in - Scaling options                                      */
/*                         To keep aspect ratio use "aspect"                    */
/*                         Quality scaling using one of the algorithms          */
/*                         "simple" ( default ), "nearestneighbor", "bilinear", */
/*                         "bell", "bspline",  "mitchell", "lanczos"            */
/*                                                                              */
/*( optional )string  in - Options for reading image                            */
/*                         see documentation for each image format              */
/*                                                                              */
/*( optional )string  in - Options for writing image                            */
/*                         Se documentation for each image format               */
/*                                                                              */
/* Return:    "0" ( successful ) or a number indicating the error               */
/* Note:      Image info will be returned when the only parameter is a valid   */
/*            source file name in the form "width height bitmap_pixel_depth"    */
/*                                                                              */
/********************************************************************************/

APIRET APIENTRY __export rxImgReSize( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr )
{

    var retval;

    if ( numargs < 1 )
    {
        fprintf( stderr, "Syntax:  rc = rxImgReSize( srcfName, tgtfName, w, h, opt, opt_in, opt_out )\n" );
        fprintf( stderr, "Example: rc = rxImgReSize( 'C:\OS2\BITMAP\GMARBLE.BMP',, 120 )\n" );
        retval.cpy( "22" ).set( retstr );
        return VALID_ROUTINE;
    }

    var source_FileName, target_FileName, source_opts, target_opts;
    int   width = -1, height = -1, color_depth = -1, filterIndex = GBM_SCALE_FILTER_SIMPLE;
    int   fd, ft_src, ft_dst, i, stride, stride2, flag, rc;
    GBMFT    gbmft;
    GBM      gbm;
    GBMRGB   gbmrgb[0x100];
    BOOL  aspect = FALSE;
    BOOL  qualityScalingEnabled = FALSE;
    BOOL  isGrayscale = FALSE;
    BYTE    *data, *data2;

    if ( numargs > 0 )
        if ( args[0].strptr != NULL )
            source_FileName.cpy( args[0].strptr );

    if ( !QueryExist( source_FileName ) )
    {
        retval.cpy( "2" ).set( retstr );
        return VALID_ROUTINE;
    }

    if ( numargs > 1 )
        if ( args[1].strptr != NULL )
            target_FileName.cpy( args[1].strptr );
        else target_FileName.cpy( source_FileName );
    else target_FileName.cpy( source_FileName );

    if ( numargs > 2 )
        if ( args[2].strptr != NULL )
            width = atoi( args[2].strptr );

    if ( numargs > 3 )
        if ( args[3].strptr != NULL )
            height = atoi( args[3].strptr );

    if ( numargs > 4 )
        if ( args[4].strptr != NULL ) {
            aspect = ( source_opts.cpy( args[4].strptr ).contains( "aspect", 0, FALSE ) > 0 );
            filterIndex = getFilterIndex( args[4].strptr, FILTER_NAME_TABLE, FILTER_NAME_TABLE_LENGTH );
        }

    if ( filterIndex == -1 )
        filterIndex = GBM_SCALE_FILTER_SIMPLE;

    if ( ( width > 0 && height < 1 ) || ( width < 1 && height > 0 ) ) /* Preserve aspect ratio unless otherwise stated */
        aspect = TRUE;

    if ( filterIndex != GBM_SCALE_FILTER_SIMPLE )
        qualityScalingEnabled = TRUE;

    /* processing */
    Gbm_init();

    if ( Gbm_guess_filetype( source_FileName, &ft_src ) != GBM_ERR_OK )
    {
        retval.cpy( "11" ).set( retstr );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    if ( Gbm_guess_filetype( target_FileName, &ft_dst ) != GBM_ERR_OK )
    {
        retval.cpy( "11" ).set( retstr );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    if ( ( fd = Gbm_io_open( source_FileName, GBM_O_RDONLY ) ) == -1 )
    {
        retval.cpy( "5" ).set( retstr );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    if ( ( rc = Gbm_read_header( source_FileName, fd, ft_src, &gbm, source_opts ) ) != GBM_ERR_OK )
    {
        Gbm_io_close( fd );
        retval.cpy( "11" ).set( retstr );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    if ( numargs == 1 )
    {
        retval.cpy( gbm.w ).cat( " " ).cat( gbm.h ).cat( " " ).cat( gbm.bpp ).set( retstr );
        Gbm_io_close( fd );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    /* check for color depth supported by algorithms */
    switch ( gbm.bpp )
    {
    case 64:
    case 48:
    case 32:
    case 24:
    case 8:
    case 4:
    case 1:
        break;
    default:
        {
            Gbm_io_close( fd );
            retval.cpy( "11" ).set( retstr );
            Gbm_deinit();
            return VALID_ROUTINE;
        }
    }

    if ( aspect && ( width > 0 ) && ( height > 0 ) )
    {
        if ( ( ( float ) width / gbm.w ) <= ( ( float ) height / gbm.h ) )
            height = ( gbm.h * width ) / gbm.w;
        else
            width = ( gbm.w * height ) / gbm.h;
    }
    else
    {
        if ( width < 1 )
        {
            if ( aspect )
                width = ( gbm.w * height ) / gbm.h;
            else
                width = gbm.w;
        }
        if ( height < 1 )
        {
            if ( aspect )
                height = ( gbm.h * width ) / gbm.w;
            else
                height = gbm.h;
        }
    }

    if ( !isBetween( 0, width, 10001 ) && !isBetween( 0, height, 10001 ) )
    {
        Gbm_io_close( fd );
        retval.cpy( "384" ).set( retstr );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    if ( ( rc = Gbm_read_palette( fd, ft_src, &gbm, gbmrgb ) ) != GBM_ERR_OK )
    {
        Gbm_io_close( fd );
        retval.cpy( "22" ).set( retstr );
        Gbm_deinit();
        return VALID_ROUTINE;
    }

    stride = ( ( ( gbm.w * gbm.bpp + 31 ) / 32 ) * 4 );
    if ( ( data = ( BYTE * )gbmmem_malloc( stride * gbm.h ) ) == NULL )
    {
        gbmmem_free(data);
        Gbm_io_close( fd );
        Gbm_deinit();
        retval.cpy( "8" ).set( retstr );
        return VALID_ROUTINE;
    }

    if ( ( rc = Gbm_read_data( fd, ft_src, &gbm, data ) ) != GBM_ERR_OK )
    {
        gbmmem_free( data2 );
        gbmmem_free( data );
        Gbm_io_close( fd );
        Gbm_deinit();
        retval.cpy( "30" ).set( retstr );
        return VALID_ROUTINE;
    }

    color_depth = gbm.bpp;

    if ( qualityScalingEnabled )
    {
        if ( gbm.bpp <= 8 )
            isGrayscale = isGrayscalePalette( gbmrgb, 1 << gbm.bpp );

        if ( ( gbm.bpp <= 8 ) && ( !isGrayscale ) )
        {
            /* convert to 24bpp */
            if ( !expandTo24Bit( &gbm, gbmrgb, &data, &stride ) )
            {
                Gbm_io_close( fd );
                retval.cpy( "22" ).set( retstr );
                return VALID_ROUTINE;
            }
            color_depth = 24;
        }
        if ( isGrayscale )
            color_depth = 8;
    }

    stride2 = ( ( ( width * color_depth + 31 ) / 32 ) * 4 );
    if ( ( data2 = ( BYTE * )gbmmem_malloc( stride2 * height ) ) == NULL )
    {
        gbmmem_free( data );
        Gbm_io_close( fd );
        Gbm_deinit();
        retval.cpy( "8" ).set( retstr );
        return VALID_ROUTINE;
    }

    Gbm_io_close( fd );

    if ( qualityScalingEnabled )
    {
        if ( isGrayscale )
            rc = gbm_quality_scale_gray( data , gbm.w, gbm.h, gbm.bpp, gbmrgb,
                                        data2, width, height, gbmrgb,
                                        FILTER_NAME_TABLE[filterIndex].filter );
        else
            rc = gbm_quality_scale_bgra( data , gbm.w, gbm.h,
                                        data2, width, height, color_depth,
                                        FILTER_NAME_TABLE[filterIndex].filter );
    }
    else
        rc = gbm_simple_scale( data, gbm.w, gbm.h, data2, width, height, gbm.bpp );

    gbmmem_free( data );

    if ( rc != GBM_ERR_OK )
    {
        gbmmem_free( data2 );
        Gbm_deinit();
        retval.cpy( "13" ).set( retstr );
        return VALID_ROUTINE;
    }

    if ( ( fd = Gbm_io_create( target_FileName, GBM_O_WRONLY ) ) == -1 )
    {
        gbmmem_free( data2 );
        Gbm_deinit();
        retval.cpy( "29" ).set( retstr );
        return VALID_ROUTINE;
    }

    Gbm_query_filetype( ft_dst, &gbmft );

    switch ( color_depth )
    {
    case 64:   flag = GBM_FT_W64;  break;
    case 48:   flag = GBM_FT_W48;  break;
    case 32:   flag = GBM_FT_W32;  break;
    case 24:   flag = GBM_FT_W24;  break;
    case 8:    flag = GBM_FT_W8;   break;
    case 4:    flag = GBM_FT_W4;   break;
    case 1:    flag = GBM_FT_W1;   break;
    default:   flag = 0;           break;
    }

    if ( ( gbmft.flags & flag ) == 0 )
    {
        Gbm_io_close( fd );
        Gbm_deinit();
        retval.cpy( "120" ).set( retstr );
        return VALID_ROUTINE;
    }

    gbm.w = width;
    gbm.h = height;
    gbm.bpp = color_depth;

    if ( ( rc = Gbm_write( target_FileName, fd, ft_dst, &gbm, gbmrgb, data2, target_opts ) ) != GBM_ERR_OK )
    {
        Gbm_io_close( fd );
        gbmmem_free( data2 );
        Gbm_deinit();
        retval.cpy( "29" ).set( retstr );
        return VALID_ROUTINE;
    }

    Gbm_io_close( fd );
    gbmmem_free( data2 );

    Gbm_deinit();

    retval.cpy( "0" ).set( retstr );

    return VALID_ROUTINE;  /* no error on call */
}

HWND title2Hwnd( var *window, BOOL similar = FALSE ) {

#define MEMPOOL  40000   /* Size of local memory pool area */

    HAB      hab        = NULLHANDLE;
    ULONG    cbItems    = 0, i,           /* Number of items in list */
        ulBufSize  = 0;            /* Size of buffer for information */
    PVOID    pvBase     = NULL;         /* Pointer to local memory pool */
    PSWBLOCK pswblk     = NULL;         /* Pointer to information returned */

    /* Allocate a large block of memory (uncommitted) and make it available for suballocation. This allows the system to commit memory only when it is actually needed.       */
    if( DosAllocMem( &pvBase, MEMPOOL, PAG_READ | PAG_WRITE ) != NO_ERROR ) return NULL;
    if( DosSubSetMem( pvBase, DOSSUB_INIT | DOSSUB_SPARSE_OBJ, MEMPOOL ) != NO_ERROR ) return NULL;

    /* Determine the number of items in the list and calculate the size of the buffer needed.                          */
    cbItems = WinQuerySwitchList( hab, NULL, 0 );
    ulBufSize = ( cbItems * sizeof( SWENTRY ) ) + sizeof( HSWITCH );

    /* Allocate the buffer from our memory pool */

    if( DosSubAllocMem( pvBase, (PVOID *) &pswblk, ulBufSize ) != NO_ERROR ) return NULL;

    /* Call WinQuerySwitchList again to fill our buffer with information */
    cbItems = WinQuerySwitchList( hab, pswblk, ulBufSize );

    PSWENTRY pswentry = pswblk->aswentry; /* pointer to current entry         */
    for ( i = 0; i < pswblk->cswentry; i++, pswentry++ ) {
        if ( window->cmp( pswentry->swctl.szSwtitle, FALSE ) == 0 )
            return pswentry->swctl.hwnd;
    }
    if ( similar )
        for ( ; i > 0; i--, pswentry-- ) {
            if ( window->isWithin( pswentry->swctl.szSwtitle, 0, FALSE ) > 0 )
                return pswentry->swctl.hwnd;
        }
    return NULL;
}

/*********************************************************************/
/* Function:  rxImgCapture                                           */
/*                                                                   */
/* Syntax:    CALL rxImgCapture 'Desktop', 1, 1, 240, 320, 'myfile.png' */
/*                                   <left>, <bottom>, <right>, <top>*/
/*            CALL rxImgCapture 'Desktop', 'Interior', 'C:\Myfile.bmp' */
/*            CALL rxImgCapture 'Desktop', 'Window' (Default)        */
/*            CALL rxImgCapture 'MyAppWindow'                        */
/*            CALL rxImgCapture 'Current'                            */
/*            CALL rxImgCapture 'Select', 'Clipboard' (Future) */
/*                                                                   */
/* Return:    0 on success, other on failure                             */
/*********************************************************************/

APIRET APIENTRY __export rxImgCapture( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr )
{
    HWND hwndFocus;
    PRECTL prcl;
    HAB hab;
    var window, opts, retval;
    static PSZ pszData[4] = { "Display", NULL, NULL, NULL };
    SIZEL   sizlPage = {0, 0};
    POINTL  aptl[3];
    LONG    alData[2];
    SWP     swp;
    SHORT   sWidth, sHeight;
    BOOL    coords = FALSE;

    if ( numargs > 0 )
        if ( args[0].strptr != NULL )
            window.cpy( args[0].strptr ).upper();

    if ( numargs > 1 )
        if ( args[1].strptr != NULL )
            if( retval.cpy( args[1].strptr ).isInteger() )
                prcl->xLeft = retval.toInt();
            else if( retval.cmp( "INTERIOR", FALSE ) == 0 )
                opts = retval;
            else if( retval.cmp( "WINDOW", FALSE ) == 0 );

    if ( numargs > 2 )
        if ( args[2].strptr != NULL )
            if( retval.cpy( args[2].strptr ).isInteger() ) {
                prcl->yBottom = retval.toInt();
                coords = TRUE;
            }

    if ( numargs > 3 )
        if ( args[3].strptr != NULL )
            if( retval.cpy( args[3].strptr ).isInteger() ) {
                prcl->xRight = retval.toInt();
                coords = TRUE;
            }

    if ( numargs > 4 )
        if ( args[4].strptr != NULL )
            if( retval.cpy( args[4].strptr ).isInteger() ) {
                prcl->yTop = retval.toInt();
                coords = TRUE;
            }

    var target_FileName, target_opts;
    if ( args[numargs-1].strptr != NULL && numargs > 1 )
        target_FileName.cpy( args[ numargs - 1 ].strptr );
    else
        target_FileName.cpy( "temporary.bmp" );
    if( target_FileName.cmp( '.', FALSE ) != 1 )
        target_FileName.cat( ".bmp" );

    if( window.cmp( "DESKTOP", FALSE ) == 0 )
        hwndFocus = HWND_DESKTOP;
    else if( window.cmp( "CURRENT", FALSE ) == 0 )
        hwndFocus = WinQueryFocus( HWND_DESKTOP );
    else if( window.cmp( "FOCUS", FALSE ) == 0 ) {
        POINTL   tmp_ptl;
        WinQueryPointerPos( HWND_DESKTOP, &tmp_ptl );
        hwndFocus = WinWindowFromPoint ( HWND_DESKTOP, &tmp_ptl, FALSE );
    } else {
        /* List windows */
        if( ( hwndFocus = title2Hwnd( &window, TRUE ) ) == NULL )
        {
            retval.cpy( window ).cat( " not found." ).set( retstr );
            return VALID_ROUTINE;
        }
    }
    if( opts.cmp( "INTERIOR", FALSE ) == 0 ) {
        HWND hwndOld;
        hwndOld = hwndFocus;
        if( !( hwndFocus = WinWindowFromID( hwndOld, FID_CLIENT ) ) )
            if( !( hwndFocus = WinQueryWindow( hwndOld, QW_BOTTOM ) ) )
                hwndFocus = hwndOld;
    } 

    // take the whole "window" if no rectangle has been specified
    if ( !coords ) {
        WinQueryWindowPos( hwndFocus, &swp );
        sWidth  = swp.cx;
        sHeight = swp.cy;
    }
    else
    {
        sWidth  = prcl->xRight - prcl->xLeft + 1;
        sHeight = prcl->yTop - prcl->yBottom + 1;
    }
    /* create the memory device context and presentation space so they are compatible with the screen device context and presentation space */
    HDC hdcMem = DevOpenDC( hab, OD_MEMORY, "*", 4, PDEVOPENDATA( pszData ), 0 );
    HPS hpsMem = GpiCreatePS( hab, hdcMem, &sizlPage, PU_PELS | GPIT_MICRO | GPIA_ASSOC | GPIF_DEFAULT );

    // determine the device's plane/bit-count format
    GpiQueryDeviceBitmapFormats( hpsMem, 2, alData );

    BITMAPINFO2 bmp;
    HBITMAP hbmNew, hbmOld;

    /* load the BITMAPINFOHEADER2 and BITMAPINFO2 structures. The sWidth and sHeight fields specify the width and height of the destination rect. */
    bmp.cbFix           = sizeof( BITMAPINFOHEADER2 );
    bmp.cx              = sWidth;
    bmp.cy              = sHeight;
    bmp.cPlanes         = alData[0];
    bmp.cBitCount       = min( 24, alData[1] );

    bmp.ulCompression   = BCA_UNCOMP;
    bmp.cxResolution    = 0;
    bmp.cyResolution    = 0;
    bmp.cclrUsed        = 0;
    bmp.cclrImportant   = 0;

    bmp.usUnits         = BRU_METRIC;
    bmp.usReserved      = 0;
    bmp.usRecording     = BRA_BOTTOMUP;
    bmp.usRendering     = BRH_NOTHALFTONED;
    bmp.cSize1          = 0;
    bmp.cSize2          = 0;
    bmp.ulColorEncoding = BCE_RGB;
    bmp.ulIdentifier    = 0;

    // create a bitmap that is compatible with the display
    if( ( hbmNew = GpiCreateBitmap( hpsMem, (BITMAPINFOHEADER2 *)&bmp, 0L, NULL, NULL ) ) == NULL )
    {
        retval.cpy( "12" ).set( retstr );
        return VALID_ROUTINE;
    }

    // associate the bitmap and the memory presentation space
    hbmOld = GpiSetBitmap( hpsMem, hbmNew );

    /* Copy the screen to the bitmap. */
    aptl[0].x = 0;
    aptl[0].y = 0;
    aptl[1].x = sWidth;
    aptl[1].y = sHeight;

    if ( !coords )
    {
        aptl[2].x = 0;
        aptl[2].y = 0;
    }
    else
    {
        aptl[2].x = prcl->xLeft;
        aptl[2].y = prcl->yBottom;
    }

    HPS hpsTemp;

    if( WinIsWindow( hab, hwndFocus ) || ( hwndFocus == HWND_DESKTOP ) )
        hpsTemp = WinGetPS( hwndFocus );
    else
    {
        GpiDeleteBitmap( hbmNew );
        DevCloseDC( hdcMem );
        return VALID_ROUTINE;
    }

    GpiBitBlt( hpsMem, hpsTemp, 3L, aptl, ROP_SRCCOPY, BBO_IGNORE );

    int   fd, ft_dst, stride2, rc;
    BYTE    *data2;
    GBMFT    gbmft;
    GBM      gbm;
    GBMRGB gbmrgb[0x100];

    /* processing */
    Gbm_init();

    gbm.w = sWidth;
    gbm.h = sHeight;
    gbm.bpp = bmp.cBitCount;
    stride2 = ( ( ( gbm.w * gbm.bpp + 31 ) / 32 ) * 4 );
    if ( ( data2 = ( BYTE * )gbmmem_malloc( stride2 * gbm.h ) ) == NULL )
    {
        gbmmem_free( data2 );
        Gbm_io_close( fd );
        retval.cpy( "8" ).set( retstr );
        return VALID_ROUTINE;
    }

    GpiQueryBitmapBits( hpsMem, 0L, (LONG) bmp.cy, data2, &bmp );

    if ( Gbm_guess_filetype( target_FileName, &ft_dst ) != GBM_ERR_OK )
    {
        retval.cpy( "11" ).set( retstr );
        return VALID_ROUTINE;
    }

    if ( ( fd = Gbm_io_create( target_FileName, GBM_O_WRONLY ) ) == -1 )
    {
        gbmmem_free( data2 );
        retval.cpy( "29" ).set( retstr );
        return VALID_ROUTINE;
    }

    Gbm_query_filetype( ft_dst, &gbmft );

    if ( ( rc = Gbm_write( target_FileName, fd, ft_dst, &gbm, gbmrgb, data2, target_opts ) ) != GBM_ERR_OK )
    {
        Gbm_io_close( fd );
        gbmmem_free( data2 );
        retval.cpy( "29" ).set( retstr );
        return VALID_ROUTINE;
    }

    Gbm_io_close( fd );
    gbmmem_free( data2 );

    Gbm_deinit();

    // re-associate the previous bitmap and the memory presentation space
    GpiSetBitmap( hpsMem, hbmOld );

    GpiDeleteBitmap( hbmNew );
    WinReleasePS( hpsTemp );
    GpiAssociate( hpsMem, (HDC)NULL );
    GpiDestroyPS( hpsMem );
    DevCloseDC( hdcMem );

    retval.cpy( "0" ).set( retstr );

    return VALID_ROUTINE;  /* no error on call */
}

/*********************************************************************/
/* Function:  rxImgLoadFuncs                                         */
/*                                                                   */
/* Syntax:    call rxImgLoadFuncs                                    */
/*                                                                   */
/* Params:    none                                                   */
/*                                                                   */
/* Return:    null string                                            */
/*********************************************************************/

APIRET APIENTRY __export rxImgLoadFuncs( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr )
{
    retstr->strlength = 0; /* return a null string result */

    for ( int i = 0; i < sizeof( RxFncTable ) / sizeof( PSZ ); i++ )
        RexxRegisterFunctionDll( RxFncTable[i], "RXIMGSZE", RxFncTable[i] );

    return VALID_ROUTINE;  /* no error on call */
}

/*********************************************************************/
/* Function:  rxImgDropFuncs                                         */
/*                                                                   */
/* Syntax:    call rxImgDropFuncs                                    */
/*                                                                   */
/* Params:    none                                                   */
/*                                                                   */
/* Return:    null string                                            */
/*********************************************************************/

APIRET APIENTRY __export rxImgDropFuncs( char* name, ULONG numargs, RXSTRING args[], char* queuename, RXSTRING *retstr )
{

    retstr->strlength = 0;           /* return a null string result*/

    for ( int i = 0; i < sizeof( RxFncTable ) / sizeof( PSZ ); i++ )
        RexxDeregisterFunction( RxFncTable[i] );

    return VALID_ROUTINE;  /* no error on call */
}
