/*****************************************************************************/
/*
 *  copydbg.c - copy IBM debug data to separate file
 *  $Id: copydbg.c,v 1.2 2020/04/13 06:26:04 Steven Exp $
 *
 *  Parts of this file are
 *    Copyright (c) 2019 Steven Levine and Associates, Inc.
 *
 * Build with VAC 3.65
 *   icc /Ss /Ti copydbg.c copydbg.def
 *   copydbg copydbg.exe
 *   lxlite -X- -B- copydbg.exe
 */
/*****************************************************************************/

#include <stdio.h>
#include <io.h>
#include <stdlib.h>
#include <fcntl.h>
#include <share.h>                      // SH_DENYNO
#include <string.h>
#include <ctype.h>                      // islower
#include <sys\stat.h>                   // S_IREAD

/* ------------------------------------------------------------------------ */
/* Last 8 bytes of 16:16 or 32 file when IBM style debugging data is present */
_Packed struct  _eodbug
{
   unsigned short dbug;          	/* 'NB' signature */
   unsigned short ver;           	/* version        */
   unsigned long dfaBase;        	/* size of debug data */
} eodbug;

#define DBUGSIG         0x424E          // NB

#define TRANSFERSIZE 0x8000
char Buffer[TRANSFERSIZE];

main(int argc, char **argv,char ** envp)
{
   int ModuleFile;
   int DbgFile;
   int TotalSize;
   int NewOffset=0;
   int savedErrno;
   unsigned long lfaBase;
   char *pszFileName;
   char *pszExt;

   if (argc != 2) {
      puts("Syntax is copydbg filename.exe");
      puts("       or copydbg filename.dll");
      puts("which will generate a file named filename.dbg containing");
      puts("the debug data, if any present in input file");
      exit(0);
   } /* endif */

   pszFileName = argv[1];

   printf("\nReading %s\n",pszFileName);
   ModuleFile = sopen(pszFileName,O_RDONLY|O_BINARY,SH_DENYNO);
   if (ModuleFile == -1) {
      savedErrno = errno;
      perror("Cannot access input file");
      exit(savedErrno);
   }

   /* Check if any debug data attached to executable */
   if ((TotalSize=lseek(ModuleFile,-8L,SEEK_END))==-1) {
      savedErrno = errno;
      perror("Cannot seek to debug data signature");
      exit(savedErrno);
   }

   TotalSize+=8;                        // Include NB0x and offset in size

   if (read(ModuleFile,(void *)&eodbug,8)==-1) {
      savedErrno = errno;
      perror("Cannot read debug data signature");
      exit(savedErrno);
   }

   if (eodbug.dbug!=DBUGSIG) {
      puts("Input file contains no debug data.");
      exit(99);
   }

   pszExt=pszFileName+strlen(pszFileName)-3;
   strcpy(pszExt, islower(*pszExt) ? "dbg" : "DBG");

   printf("Writing debug data to %s\n",pszFileName);

   DbgFile=open(pszFileName,O_CREAT | O_WRONLY | O_TRUNC | O_BINARY,S_IREAD | S_IWRITE);
   if (DbgFile==-1) {
      savedErrno = errno;
      perror("Cannot create output file");
      exit(savedErrno);
   }

   if ((lfaBase=lseek(ModuleFile,-eodbug.dfaBase,SEEK_END))==-1L) {
      savedErrno = errno;
      perror("Cannot seek to start of debug data");
      exit(savedErrno);
   }

   TotalSize-=lfaBase;                  // Calc debug data data size

   while (TotalSize>0) {
      int ReadBytes;
      int written;
      ReadBytes=read(ModuleFile,Buffer,min(TRANSFERSIZE,TotalSize));
      if (ReadBytes>0) {
         TotalSize-=ReadBytes;
         written=write(DbgFile,(void *)Buffer,ReadBytes);
	 if (written<=0)
	    perror("Cannot write to output file");
      } /* endif */
   } /* endwhile */

   close(DbgFile);
   close(ModuleFile);

   return 0;
} // main
