
/* PMM Sample Source "C" File ***********************************************
 *
 * PROGRAM NAME: PMMSAM
 * -------------
 *  Presentation Manager Standard Window "C" Sample Program
 *
 * WHAT THIS PROGRAM DOES:
 * -----------------------
 *  This program displays a standard window containing the text "Hello".
 *  The action bar contains a single choice, Options.
 *  The Options pull-down contains three choices that each
 *  paint a different string in the window.
 *  A PMM trace session is created.  All window messages generated
 *  in this program are traced and written to a file
 *
 * WHAT THIS PROGRAM DEMONSTRATES:
 * -------------------------------
 *  This program demonstrates how to create and display a standard window,
 *  how to use resources defined in a resource script file,
 *  and how to create and use a PMM trace session
 *
 * WHAT YOU NEED TO COMPILE THIS PROGRAM:
 * --------------------------------------
 *
 *  REQUIRED FILES:
 *  ---------------
 *
 *    PMMSAM.C       - Source code
 *    PMMSAM.H       - Application header file
 *    PMMSAM.MAK     - Make file for this program
 *    PMMSAM.LNK     - Linker automatic response file
 *    PMMSAM.DEF     - Module definition file
 *    PMMSAM.RC      - Resource file
 *    PMMSAM.ICO     - Icon file
 *
 *    OS2.H          - Presentation Manager include file
 *    STRING.H       - String handling function declarations
 *
 *    PMMPOS2.H      - PMM header file
 *    PMM.H          - PMM header file
 *
 * WHAT FILES NEED TO BE MODIFIED TO USE PMM IN THIS PROGRAM:
 * ----------------------------------------------------------
 *
 *    PMMSAM.C
 *    PMMSAM.MAK
 *    PMMSAM.DEF
 *
 *  REQUIRED LIBRARIES:
 *  -------------------
 *
 *    OS2386.LIB     - Presentation Manager OS/2 library
 *    DDE4MBS.LIB    - Presentation Manager OS/2 library
 *    PMM.LIB        - Presentation Manager Monitor library
 *
 *  REQUIRED PROGRAMS:
 *  ------------------
 *
 *    A "C" Compiler
 *    A Linker
 *    A Resource Compiler
 *
 ***************************************************************************/

/* Conditional compilation. Activate/deactivate PMM easily */
#define USE_PMM         1

#define INCL_WIN
#define INCL_GPI
#define INCL_DOSPROCESS

/* Include this file immediately before including "os2.h" */
#if USE_PMM
   #include <pmmpos2.h>
#endif
#include <os2.h>                        /* PM header file                */
#include "pmmsam.h"                     /* Resource symbolic identifiers */
/* Include the "main" PMM file */
#if USE_PMM
   #include <pmm.h>
#endif
/* Include header files as needed */
#include <string.h>

#define BEEP_WARN_FREQ          60      /* Frequency of warning beep    */
#define BEEP_WARN_DUR           100     /* Duration of warning beep     */

/* Function prototypes */
INT main( VOID );
MRESULT EXPENTRY PMMSampleWindowProc( HWND hwnd, ULONG msg,
                                      MPARAM mp1, MPARAM mp2 );

/**************************************************************************
 *
 *  Name       : main()
 *
 *  Description: Initializes the process for OS/2 PM services, creates
 *               a PMM trace session and process the application message
 *               queue until a WM_QUIT message is received.  It then
 *               destroys all created resources and terminates.
 *
 *  Concepts   : - obtains anchor block handle and creates message
 *                 queue
 *               - creates a trace session and starts tracing
 *               - creates the main frame window which creates the
 *                 main client window
 *               - polls the message queue via Get/Dispatch Msg loop
 *               - upon exiting the loop destroys the trace session,
 *                 frees all allocated resources and exits
 *
 *  API's      :   DosBeep
 *                 WinInitialize
 *                 WinCreateMsgQueue
 *                 PMMCreateTraceSession
 *                 PMMStartTracing
 *                 WinRegisterClass
 *                 WinCreateStdWindow
 *                 WinGetMsg
 *                 WinDispatchMsg
 *                 WinDestroyWindow
 *                 PMMStopTracing
 *                 PMMDestroyTraceSession
 *                 WinDestroyMsgQueue
 *                 WinTerminate
 *
 *  Parameters :  [none]
 *
 *  Return     :  zero (0) - if successful execution completed
 *                non-zero - if error
 *
 *************************************************************************/
INT main( VOID )
{
   HAB  hab;                            /* PM anchor block handle        */
   HMQ  hmq;                            /* Message queue handle          */
   QMSG qmsg;                           /* Message from message queue    */
   HWND hwndFrame;                      /* Frame window handle           */
   HWND hwndClient;                     /* Client area window handle     */
   ULONG flCreate;                      /* Window creation control flags */
#if USE_PMM
   HPMMSESSION hpmms;
   PMMRET pmmr;
#endif

   hab = WinInitialize( 0 );
   if ( !hab )
   {
      DosBeep( BEEP_WARN_FREQ, BEEP_WARN_DUR );
      return 1; /* Error */
   }

   hmq = WinCreateMsgQueue( hab, 0 );
   if ( !hmq )
   {
      WinTerminate( hab );
      DosBeep( BEEP_WARN_FREQ, BEEP_WARN_DUR );
      return 2; /* Error */
   }

   /* Return codes from PMM are not tested in this example,
   they should be tested */

   /* Create a trace session and start tracing */
#if USE_PMM
   pmmr = PMMCreateTraceSession( &hpmms, hab, "pmmsam.log" );
   pmmr = PMMStartTracing( hpmms );
#endif

   if ( !WinRegisterClass(              /* Register window class        */
        hab,                            /* Anchor block handle          */
        (PSZ) "PMMSampleClass",         /* Window class name            */
        (PFNWP) PMMSampleWindowProc,    /* Address of window procedure  */
        CS_SIZEREDRAW,                  /* Class style                  */
        0 ) )                           /* No extra window words        */
   {
      WinDestroyMsgQueue( hmq );
      WinTerminate( hab );
      DosBeep( BEEP_WARN_FREQ, BEEP_WARN_DUR );
      return 3; /* Error */
   }

   flCreate = FCF_STANDARD;

   if ( ( hwndFrame = WinCreateStdWindow(
      HWND_DESKTOP,                     /* Desktop window is parent     */
      WS_VISIBLE,                       /* STD. window styles           */
      &flCreate,                        /* Frame control flags          */
      "PMMSampleClass",                 /* Client window class name     */
      "PMM Sample",                     /* Window text                  */
      0,                                /* No special class style       */
      (HMODULE) 0L,                     /* Resource is in .EXE file     */
      ID_WINDOW,                        /* Frame window identifier      */
      &hwndClient                       /* Client window handle         */
      ) ) == 0L )
   {
      WinDestroyMsgQueue( hmq );
      WinTerminate( hab );
      DosBeep( BEEP_WARN_FREQ, BEEP_WARN_DUR );
      return 4; /* Error */
   }

   while( WinGetMsg( hab, &qmsg, 0L, 0, 0 ) )
      WinDispatchMsg( hab, &qmsg );

   WinDestroyWindow( hwndFrame );

   /* Stop tracing and destroy the trace session */
#if USE_PMM
   pmmr = PMMStopTracing( hpmms );
   pmmr = PMMDestroyTraceSession( hpmms );
#endif

   WinDestroyMsgQueue( hmq );

   WinTerminate( hab );

   return 0; /* Success */
} /* End of main */

/**************************************************************************
 *
 *  Name       : PMMSampleWindowProc
 *
 *  Description: The window procedure associated with the client area in
 *               the standard frame window.  It processes all messages
 *               either sent or posted to the client area, depending on
 *               the message command and parameters.
 *
 *  Concepts   :
 *
 *  API's      :   WinInvalidateRegion
 *                 WinPostMsg
 *                 WinDefWindowProc
 *                 WinBeginPaint
 *                 GpiSetColor
 *                 GpiSetBackColor
 *                 GpiSetBackMix
 *                 GpiCharStringAt
 *                 WinEndPaint
 *
 *  Parameters :  hwnd = window handle
 *                msg = message code
 *                mp1 = first message parameter
 *                mp2 = second message parameter
 *
 *  Return     :  depends on message sent
 *
 *************************************************************************/
MRESULT EXPENTRY PMMSampleWindowProc( HWND hwnd, ULONG msg,
                                      MPARAM mp1, MPARAM mp2 )
{
   static CHAR szString[20];

   switch( msg )
   {
   case WM_CREATE:
      /*
         Window initialization is performed here in WM_CREATE processing
      */
      strcpy( szString, "Hello" );
      break;
   case WM_COMMAND:
      /*
         When the user chooses option 1, 2, or 3 from the Options pull-
         down, the text string is set to 1, 2, or 3, and
         WinInvalidateRegion sends a WM_PAINT message.
         When Exit is chosen, the application posts itself a WM_CLOSE
         message.
      */
   {
      USHORT command;                /* WM_COMMAND command value     */
      command = SHORT1FROMMP(mp1);   /* Extract the command value    */
      switch( command )
      {
      case ID_OPTION1:
         strcpy( szString, "Option 1" );
         WinInvalidateRegion( hwnd, 0L, FALSE );
         break;
      case ID_OPTION2:
         strcpy( szString, "Option 2" );
         WinInvalidateRegion( hwnd, 0L, FALSE );
         break;
      case ID_OPTION3:
         strcpy( szString, "Option 3" );
         WinInvalidateRegion( hwnd, 0L, FALSE );
         break;
      case ID_EXITPROG:
         WinPostMsg( hwnd, WM_CLOSE, (MPARAM) 0, (MPARAM) 0 );
         break;
      default:
         return WinDefWindowProc( hwnd, msg, mp1, mp2 );
      }
      break;
   }
   case WM_ERASEBACKGROUND:
      /*
        Return TRUE to request PM to paint the window background
        in SYSCLR_WINDOW.
      */
      return (MRESULT) TRUE;
   case WM_PAINT:
      /*
        Window contents are drawn here in WM_PAINT processing
      */
   {
      HPS    hps;                       /* Presentation Space handle    */
      RECTL  rc;                        /* Rectangle coordinates        */
      POINTL pt;                        /* String screen coordinates    */
                                        /* Create a presentation space  */
      hps = WinBeginPaint( hwnd, 0L, &rc );
      pt.x = 50; pt.y = 50;             /* Set the text coordinates,    */
      GpiSetColor( hps, CLR_NEUTRAL );         /* colour of the text,   */
      GpiSetBackColor( hps, CLR_BACKGROUND );  /* its background and    */
      GpiSetBackMix( hps, BM_OVERPAINT );      /* how it mixes,         */
                                               /* and draw the string...*/
      GpiCharStringAt( hps, &pt, (LONG) strlen( szString ), szString );
      WinEndPaint( hps );                      /* Drawing is complete   */
      break;
   }
   case WM_CLOSE:
      /*
        This is the place to put your termination routines
      */
      /* Cause termination */
      WinPostMsg( hwnd, WM_QUIT, (MPARAM) 0, (MPARAM) 0 );
      break;
   default:
      /*
        Everything else comes here.  This call MUST exist
        in your window procedure
      */
      return WinDefWindowProc( hwnd, msg, mp1, mp2 );
   }
   return (MRESULT) 0;
} /* End of PMMSampleWindowProc */

/* End of PMM Sample Source C File ****************************************/

