/*
 *      HTEXT.CMD - HyperText/2 - V2.01 - C. Langanke 2001 - 2007
 *                                        A. Schnellbacher 2008 - 2015
 *
 *    Syntax: htext <input_file> [<ipf_file>] [<inf_or_hlp_file>] [rch_file]
 *                  [/Verbose] [/Noview] [/Hlp] [/Rch[:define_prefix]]
 *                  [/Subpages]
 *
 *    By default, HyperText/2 creates an IPF and INF or HLP file with the
 *    same name as the input file, but with extensions .ipf and .inf or .hlp.
 *    Optionally you can specify the IPF and INF/HLP file to override the
 *    default names (and directories).
 *
 *    Notes:
 *    - In order to specify the resulting INF or HLP file, you have also
 *      to specify the IPF file.
 *    - In order to specify a resource header file, you have also to specify
 *      the IPF and INF/HLP file.
 *    - If you specify the IPF file and no INF file, the INF file name
 *      will match the (path)name of the IPF file.
 *    - If not specified otherwise, the target files are created in the
 *      directory of the source file.
 *    - If the INF file can be successfully generated, the INF file viewer
 *      is started to show it, unless the parameter /Noview is specified.
 */
/* The first comment is used as online help text */

 SIGNAL ON HALT NAME Halt;

 env       = 'OS2ENVIRONMENT';
 TRUE      = (1 = 1);
 FALSE     = (0 = 1);
 CrLf      = '0d0a'x;
 Redirection = '1>NUL 2>&1';
 '@ECHO OFF';
 rcx = SETLOCAL();

 TitleLines = SUBSTR( SourceLine(2), 9)''CrLf ||,
              SUBSTR( SourceLine(3), 9 + LENGTH( '.CMD'));
 PARSE VAR TitleLines CmdName'.CMD 'Info;
 Title = CmdName Info;

 /* some OS/2 error codes */
 ERROR.NO_ERROR           =   0;
 ERROR.INVALID_FUNCTION   =   1;
 ERROR.FILE_NOT_FOUND     =   2;
 ERROR.PATH_NOT_FOUND     =   3;
 ERROR.ACCESS_DENIED      =   5;
 ERROR.NOT_ENOUGH_MEMORY  =   8;
 ERROR.INVALID_FORMAT     =  11;
 ERROR.INVALID_DATA       =  13;
 ERROR.NO_MORE_FILES      =  18;
 ERROR.WRITE_FAULT        =  29;
 ERROR.READ_FAULT         =  30;
 ERROR.SHARING_VIOLATION  =  32;
 ERROR.GEN_FAILURE        =  31;
 ERROR.INVALID_PARAMETER  =  87;
 ERROR.ENVVAR_NOT_FOUND   = 204;

 GlobalVars = 'Title CmdName env TRUE FALSE CrLf Redirection ERROR.';

 /* load RexxUtil */
 CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs';
 CALL SysLoadFuncs;

 /* some defaults */
 GlobalVars        = GlobalVars 'Flag. Valid. IpfSymbol. HtmlSymbol. CSymbolList' ||,
                                ' Char. Pages. Bitmap. If. Launch. List.' ||,
                                ' PendingNewPar ListPendingNewPar LineCount' ||,
                                ' StartLine EndLine NextLines SourceFile' ||,
                                ' TargetFile PrevCommand Doc.';

 rc                = ERROR.NO_ERROR;

 Flag.fHelp        = FALSE;
 Flag.fDebug       = FALSE;
 Flag.fVerbose     = FALSE;
 Flag.fNoView      = FALSE;
 Flag.fHelpFile    = FALSE;
 Flag.fRchFile     = FALSE;
 Flag.fSubPages    = FALSE;
 Flag.fAutoLinkAll = FALSE;
 Flag.fFullPaths   = FALSE;


 SourceFile        = '';
 TargetFile        = '';
 InfFile           = '';
 IpfDefaultOptions = '-c:850';
 RchFile           = '';
 RchDefineStem     = '';
 LineCount         = 0;
 StartLine         = '';
 EndLine           = '';
 NextLines         = '';
 PendingNewPar     = '';
 ListPendingNewPar = '';
 PrevCommand       = '';

 IPFC._EXEC        = 'IPFC.EXE';
 IPFC._EnvVar      = 'IPFC';

 Valid._Colors     = 'DEFAULT BLACK BLUE RED PINK GREEN CYAN YELLOW BROWN DARKGRAY' ||,
                     ' DARKBLUE DARKRED DARKPINK DARKGREEN DARKCYAN PALEGRAY';
 Valid._Fonts      = 'DEFAULT HELV TMS_RMN COURIER SYSTEM_MONOSPACED SYSTEM_PROPORTIONAL SYSTEM_VIO';

 /* used by MakeIPFLine */
 Char._EscapeChars = '*#_';
 Char._Attrs       = 'BOLD ITALIC UNDERLINED';
 Char._LastAttrs   = '';
 Char._LinkChars   = '[]';  /* unused */
 Char._Urls        = 'HTTP HTTPS FTP MAILTO IRC';

 /* ttfont attributes */
 Char._DefaultTtAttr     = 'facename=Courier size=18x12';
 Char._TtAttr            = Char._DefaultTtAttr;

 /* list attributes */
 Char._ListAttrOrdered   = ':li.|:ol compact.|:eol.|:ol compact.|:eol.';
 Char._ListAttrUnordered = ':li.|:ul compact.|:eul.|:ul compact.|:eul.';
 Char._ListAttrSimple    = ':li.|:sl compact.|:esl.|:sl compact.|:esl.';
 Char._ListAttrBreaks    = '.br'CrLf'|'CrLf'|.br|:ul compact.|:eul.';
 Char._ListAttrNoList    = '';

 /* symbols for conversion may not include escape chars above! */
 IpfSymbol._Chars        = '&:';
 IpfSymbol._Strings      = '&amp. &colon.';
 IpfSymbol._Col1Chars    = '.';
 IpfSymbol._Col1Strings  = '&per.';
 IpfSymbol._Space        = '&rbl.';  /* rbl means required blank */

 CSymbolList        = '';

 Pages.             = '';
 Pages.0            = 0;
 Pages._MaxLevel    = 0;
 Pages._ClearLevels = 1;
 Pages._ResetNumber = '0.0.0.0.0.0.';
 Pages._CurNumber   = Pages._ResetNumber;
 Pages._fTextMode   = FALSE;
 Pages._fMonospaceMode = FALSE;

 Doc._ListOfLinks   = '';

 /* set default values for env vars if not already set */
 NameList = 'WEBBMP EMAILBMP   INFBMP   NOTE'
 ValList  = 'ns.bmp nsmail.bmp book.bmp *Note:*'
 DO i = 1 TO WORDS( NameList)
    Name = WORD( NameList, i);
    Val  = WORD( ValList, i);
    IF (VALUE( Name,, env) = '') THEN rcx = VALUE( Name, Val, env);
 END;
 /* check for bitmap files only once at first usage */
 Bitmap.                 = '';
 Bitmap._fBitmapsChecked = FALSE;

 /* special launchname to allow patch of INF files */
 /* created by HyperText/2 through HTLAUNCH.EXE    */
 Launch._Exec       = 'nEtScApE.eXe';

 If._fIfOpen        = FALSE;
 If._fIfElse        = FALSE;
 If._fIncludeSource = FALSE;

 /* process config via env var */
 Flag.fFullPaths = (WORDPOS( TRANSLATE( VALUE( HTWRITEFULLPATHS,, env)), '1 TRUE ON') > 0);

 /* read commandline parameters */
 PARSE ARG Parms;
 Rest = Parms;
 DO WHILE (LENGTH( Rest) > 0)
    Rest = STRIP( Rest);
    IF LEFT( Rest, 1) = '"' THEN
       PARSE VAR Rest '"'ThisParm'"' Rest;
    ELSE
       PARSE VAR Rest ThisParm Rest;
    PARSE VAR ThisParm ThisTag':'ThisValue;
    ThisTag = TRANSLATE( ThisTag);

    SELECT
       WHEN (POS( ThisTag, '/DEBUG') = 1) THEN
          Flag.fDebug = TRUE;

       WHEN (POS( ThisTag, '/VERBOSE') = 1) THEN
          Flag.fVerbose = TRUE;

       WHEN (POS( ThisTag, '/NOVIEW') = 1) THEN
          Flag.fNoView = TRUE;

       WHEN (POS( ThisTag, '/HLP') = 1) THEN
          Flag.fHelpFile = TRUE;

       WHEN (POS( ThisTag, '/SUBPAGES') = 1) THEN
          Flag.fSubPages = TRUE;

       WHEN (POS( ThisTag, '/RCH') = 1) THEN
       DO
          Flag.fRchFile = TRUE;
          IF (POS( ':', ThisParm) > 0) THEN
             RchDefineStem = ThisValue;
       END;

       WHEN (POS( ThisTag, '/?') = 1) THEN
          Flag.fHelp =TRUE;

       OTHERWISE
       DO
          SELECT
             WHEN (SourceFile = '') THEN
                 SourceFile = ThisParm;

             WHEN (TargetFile = '') THEN
                 TargetFile = ThisParm;

             WHEN (InfFile = '') THEN
                 InfFile = ThisParm;

             WHEN (RchFile = '') THEN
                 RchFile = ThisParm;

             OTHERWISE
             DO
                SAY 'Error: invalid parameters.';
                SAY;
                'PAUSE'
                EXIT( ERROR.INVALID_PARAMETER);
             END;
          END;
       END;
    END;
 END;

 IF (Flag.fVerbose) THEN
    SAY;

 /* always enable subpages for INF files */
 IF \(Flag.fHelpFile) THEN
    Flag.fSubPages = TRUE;

 /* automatically open subpages when dimensions match? */
 /* otherwise autolinks can be created by .SUBLINKs only */
 IF (Flag.fSubPages) THEN
    Flag.fAutoLinkAll = TRUE;

 DO UNTIL (TRUE)

    /* display help */
    IF (Flag.fHelp) THEN
    DO
       rcx = SHowHelp();
       LEAVE;
    END;

    /* check for IPF compiler */
    IF (SysSearchPath( 'PATH', IPFC._Exec) = '') THEN
    DO
       SAY 'Error: IPF compiler not found.';
       SAY;
       'PAUSE'
       rc = ERROR.FILE_NOT_FOUND;
       LEAVE;
    END;

    IF (VALUE( IPFC._EnvVar,,env) = '') THEN
    DO
       SAY 'Error: IPF compiler is not properly installed.';
       SAY '       Environment variable' IPFC._EnvVar 'is missing.';
       SAY;
       'PAUSE'
       rc = ERROR.ENVVAR_NOT_FOUND;
       LEAVE;
    END;

    /* check source file */
    IF (SourceFile = '') THEN
    DO
       SAY 'Error: no sourcefile given.';
       rc = ERROR.INVALID_PARAMETER;
       LEAVE;
    END;
    IF (\FileExist( SourceFile)) THEN
    DO
       SAY 'Error: sourcefile "'SourceFile'" not found.';
       rc = ERROR.FILE_NOT_FOUND;
       LEAVE;
    END;

    /* determine target file */
    IF (TargetFile = '') THEN
    DO
       BaseName = FILESPEC( 'N', SourceFile);
       TargetFile = ChangeExtension( Basename, '.ipf');
    END;

    IF (InfFile = '') THEN
    DO
       BaseName = FILESPEC( 'N', TargetFile);
       IF (Flag.fHelpFile) THEN
          InfFile = ChangeExtension( Basename, '.hlp');
       ELSE
          InfFile = ChangeExtension( Basename, '.inf');
    END;

    IF (Flag.fHelpFile) THEN
       IpfTypeOption = '';
    ELSE
       IpfTypeOption = '/INF';

    IpfOptions = IpfDefaultOptions IpfTypeOption;

    IF (Flag.fRchFile) THEN
    DO
       IF (RchFile = '') THEN
          RchFile = ChangeExtension( Basename, '.rch');
    END;

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    List.0    = 0;
    rcProcess = ProcessSourceFile( SourceFile, TargetFile, 0, '');

    IF (rcProcess > 0) THEN
    DO
       rc = rcProcess;
       SAY 'Aborting due to error, rc =' rc'.';
       LEAVE;
    END;

    /* determine used levels */
    IF (LENGTH( Pages._HToc) > 0) THEN
       DocLevels = Pages._HToc;
    ELSE
    DO
       DocLevels = '';
       DO i = 1 TO Pages._MaxLevel
          DocLevels = DocLevels''i;
       END;
    END;

    /* write header of target file */
    rcx = SysFileDelete( TargetFile);
    rcx = LINEOUT( TargetFile, ':userdoc.');
    rcx = LINEOUT( TargetFile, ':docprof toc='DocLevels'.');

    IF (Pages._DocTitle \= '') THEN
       rcx = LINEOUT( TargetFile, ':title.'MakeIPFLine( Pages._DocTitle, TRUE));

    /* go thru all pages resolving links, sublink lists and write data */
    DO p = 1 TO Pages.0

       SourceWindowAttrs = '';

       /* - - - - - create sublink list - - - - - */

       /* sublink commands supported only in headers, not in footnotes! */
       IF ((Pages.p._Level < 6) & (Pages.p._SublinkPos > -1)) THEN
       DO
          SublinkPercent = Pages.p._SublinkPercent;

          /* check starting point of sublink tree */
          StartPage = p;
          SublinkAnchor = Pages.StartPage._SublinkAnchor;
          IF (SublinkAnchor \= '') THEN
          DO
             StartPage = GetPageFromAnchor( SublinkAnchor);
             IF (StartPage = 0) THEN
             DO
                SAY SourceFile'(***): Anchor' SublinkAnchor 'for .SUBLINK command not found.';
                rc = ERROR.INVALID_DATA;
                EXIT(rc);
             END;
          END;

          /* determine dimensions of pages */
          IF (SublinkPercent \= '') THEN
             Pages.StartPage._Dimensions = ModifyDimensions( Pages.StartPage._Dimensions, 'SOURCE', SublinkPercent);

          /* check values for negative level offset */
          SELECT
             /* show all pages one level above */
             WHEN (Pages.p._SublinkLevels = -1) THEN
             DO
                ThisLevel = Pages.StartPage._Level - 1;
                SubLevel  = 1;
             END;

             OTHERWISE
             DO
                ThisLevel = Pages.StartPage._Level;
                SubLevel  = Pages.StartPage._SublinkLevels;
             END;
          END;

          /* go thru all subpages */
          LastLevel = ThisLevel;
          LinkList  = '';
          DownCount = 0;
          PARSE VAR Pages.StartPage._SublinkType ItemTag'|'StartTag'|'EndTag'|'DownTag'|'UpTag;
          DO s = StartPage + 1 TO Pages.0
             LinkStart = ':link reftype=hd viewport dependent refid='s'.';
             LinkEnd   = ':elink.';

             IF (Pages.s._Alias = '') THEN
                LinkTitle = Pages.s._RefTitle;
             ELSE
                Linktitle = Pages.s._Alias;

             SELECT

                /* ignore footnotes with level 7 */
                WHEN (Pages.s._Level > 6) THEN
                   ITERATE;

                /* only modify dimensions on windows of lower level than desired */
                WHEN ((SubLevel > 0) & (Pages.s._Level - ThisLevel > SubLevel)) THEN
                DO
                   Pages.s._Dimensions = ModifyDimensions( Pages.s._Dimensions, 'TARGET', SublinkPercent);
                   ITERATE;
                END;

                /* level is in range, so add to list if */
                /* step down in list indent when reachin lower level than before */
                WHEN (Pages.s._Level = ThisLevel + 1) THEN
                DO
                   IF (Pages.s._Level < LastLevel) THEN
                   DO
                      LinkList = LinkList''CrLf''UpTag;
                      DownCount = DownCount - 1;
                   END;

                   IF (Pages.p._SublinkType \= '') THEN
                   DO
                      NewLink = ''CrLf''ItemTag''LinkStart''MakeIPFLine( LinkTitle, TRUE)''LinkEnd;
                      LinkList = LinkList''NewLink;
                   END;

                END;

                /* level is in range, so add to list if */
                /* step up in list indent when reachin upper level than before */
                WHEN (Pages.s._Level > ThisLevel) THEN
                DO
                   IF (Pages.s._Level > LastLevel) THEN
                   DO
                      LinkList = LinkList''CrLf''DownTag;
                      DownCount = DownCount + 1;
                   END;

                   IF (Pages.p._SublinkType \= '') THEN
                   DO
                      NewLink = ''CrLf':li.'LinkStart''MakeIPFLine( Linktitle, TRUE)''LinkEnd;
                      LinkList = LinkList''NewLink;
                   END;

                END;

                OTHERWISE LEAVE;
             END;

             /* modify dimensions of subpage, but only  */
             /* when .DI wasn't specified for that page */
             IF (Pages.s._SpecifiedDimensions = '') THEN
             DO
                Pages.s._Dimensions = ModifyDimensions( Pages.s._Dimensions, 'TARGET', SublinkPercent);

                /* add auto-link on the superior panel for the subpanel, */
                /* but do that only when .DI wasn't specified            */
                IF (Pages.p._AutoLink = '' & s = p + 1 & Pages.p._Dimensions \= '') THEN
                DO
                   /* check space for a linked page */
                   PARSE VAR Pages.p._Dimensions _x _y _cx _cy;
                   /* space at the right */
                   Width_x = _x + _cx;
                   IF (Width_x > 80) THEN
                      Width_x = 100;
                   /* space at the bottom */
                   Space_y = _y;
                   IF (Space_y < 20) THEN
                      Space_y = 0;
                   /* create autolink only if there's enough space */
                   IF (Width_x \= 100 | Space_y \= 0) THEN
                      Pages.p._AutoLink = ':link reftype=hd auto viewport dependent refid='s'.';
                END;
             END;

             LastLevel = Pages.s._Level;

          END; /* s = StartPage + 1 TO Pages.0 */

          /* append end tags for open lists */
          DO i = 1 TO DownCount
             IF (LENGTH( UpTag) > 0) THEN
                LinkList = LinkList''CrLf''UpTag;
          END;
          DownCount = 0;

          IF (LinkList \= '') THEN
          DO
             /* make list complete */
             LinkList = StartTag''LinkList''CrLf''EndTag;
             IF (LENGTH( Pages.p._Contents) = 0) THEN
                Pages.p._Contents = LinkList;
             ELSE
                Pages.p._Contents = INSERT( CrLf''LinkList, Pages.p._Contents, Pages.p._SublinkPos);
          END;

       END;
    END;

    /* go thru all pages copying contents of pages */
    DO p = 1 TO Pages.0
       IF (Pages.p._MirrorAnchor \= '') THEN
       DO
          SourcePage = GetPageFromAnchor( Pages.p._MirrorAnchor);
          IF (SourcePage = 0) THEN
          DO
             SAY SourceFile'(***): anchor' Pages.p._MirrorAnchor 'for .MIRROR command not found.';
             rc = ERROR.INVALID_DATA;
             EXIT(rc);
          END;
          ELSE
          DO
             /* mirror contents to this page */
             Pages.p._Contents       = Pages.SourcePage._Contents;

             /* also copy sublink attributes */
             Pages.p._SublinkPercent = Pages.SourcePage._SublinkPercent;
             Pages.p._Dimensions     = Pages.SourcePage._Dimensions;
             Pages.p._AutoLink       = Pages.SourcePage._AutoLink;
          END;
       END;
    END;

    /* reset all attributes previously set */
    Char._LastAttrs   = '';

    /* write data */
    DO p = 1 TO Pages.0

       /* invoke header creation trace */
       IF (Pages.p._HTrace) THEN
       DO
          SAY 'Trace for header' p':'Pages.p._Title;
          TRACE ?i;
       END;

       /* - - - - - - - - - - - - - - - - - - - - */

       /* determine parameters for header */
       HeaderAttributes = '';
       IpfTitle    = MakeIPFLine( Pages.p._Title, TRUE);
       IpfRefTitle = MakeIPFLine( Pages.p._RefTitle, TRUE);
       IF (Flag.fSubPages) THEN
       DO
          Dimensions = Pages.p._Dimensions;
          IF (Pages.p._Dimensions \= '') THEN
          DO
             PARSE VAR Pages.p._Dimensions _x _y _cx _cy;
             HeaderAttributes = STRIP( HeaderAttributes 'x='_x'% y='_y'% width='_cx'% height='_cy'%');

             /* automatically add auto-links if panel dimensions match */
             IF (Flag.fAutoLinkAll & Pages.p._AutoLink = '') THEN
             DO
                /* space at the right */
                Width_x = _x + _cx;
                IF (Width_x > 80) THEN
                   Width_x = 100;
                /* space at the bottom */
                Space_y = _y;
                IF (Space_y < 20) THEN
                   Space_y = 0;
                /* maybe create autolink if there's enough space */
                IF (Width_x \= 100 | Space_y \= 0) THEN
                DO

                   /* go thru all subpages for auto-links */
                   ThisLevel = Pages.p._Level;
                   StartPage = p;
                   IF (Pages.p._AutoLink = '') THEN
                   DO s = StartPage + 1 TO Pages.0
                      PARSE VAR Pages.s._Dimensions s_x s_y s_cx s_cy;

                      SELECT

                         /* ignore footnotes with level 7 */
                         WHEN (Pages.s._Level > 6) THEN
                            ITERATE;

                         /* stop at window of higher level */
                         WHEN (Pages.s._Level < ThisLevel) THEN
                            LEAVE;

                         /* find window with matching dimensions */
                         WHEN (Pages.s._Dimensions = '') THEN
                            ITERATE;

                         WHEN (Width_x > 0 & s_x = Width_x) THEN
                            /* found! */
                            NOP;

                         WHEN (Space_y > 0 & s_cy + s_y = Space_y) THEN
                            /* found! */
                            NOP;

                         OTHERWISE LEAVE;
                      END;

                      /* add link to list if panel with matching dimensions found */
                      Pages.p._AutoLink = ':link reftype=hd auto viewport dependent refid='s'.';
                      LEAVE;
                   END; /* s = StartPage + 1 TO Pages.0 */

                END;
             END;

          END;
       END;
       IF (Pages.p._Clear = 1) THEN
          HeaderAttributes = STRIP( HeaderAttributes 'clear');
       IF (Pages.p._Hide = 1) THEN
          HeaderAttributes = STRIP( HeaderAttributes 'hide');
       IF (Pages.p._Resid \= '') THEN
          HeaderAttributes = STRIP( HeaderAttributes 'res='Pages.p._Resid);

       /* write header or footnote */
       SELECT
          /* create footnote panel */
          WHEN (Pages.p._Level > 6) THEN
          DO
             rcx = LINEOUT( TargetFile, ':fn id='p'.');
             IF (LENGTH( Pages.p._Contents) > 0) THEN
                Pages.p._Contents = Pages.p._Contents''CrLf;
             Pages.p._Contents = Pages.p._Contents':efn.';
          END;

          /* create header panel */
          OTHERWISE
             rcx = LINEOUT( TargetFile, ':h'Pages.p._Level' id='p' 'HeaderAttributes'.'IpfTitle);
       END;

       /* write index data */
       IF (Pages.p._Index) THEN
          rcx = LINEOUT( TargetFile, ':i1.'IpfRefTitle);

       EmptyLineLen = LENGTH( ':p.'CrLf);

       /* for INF file (or HLP file with /Subpages option) write autolink */
       IF (Flag.fSubPages) THEN
       DO
          IF ((Pages.p._AutoLink \= '') & ((Pages.p._SublinkPercent \= '') | Flag.fAutoLinkAll)) THEN
          DO
             /* ---- Workaround for NewView:                                 */
             /* A ":link" line coming after the header would make NewView    */
             /* show an additional line at the top of a page. Reversing the  */
             /* ":link" line with a following :p. line will work around      */
             /* that. Apparently NewView does an internal newline correction */
             /* after the header to add a newline, even when forgotten.      */
             /* BTW: :i1. and :i2. lines don't confuse NewView here.         */

             /* first check for a leading parbreak in the page data */
             IF (WORDPOS( LEFT( Pages.p._Contents, EmptyLineLen), ':p.'CrLf '.br'CrLf) > 0) THEN
             DO
                /* strip it off and write it before the :link line */
                Pages.p._Contents = RIGHT( Pages.p._Contents, LENGTH( Pages.p._Contents) - EmptyLineLen);
                rcx = LINEOUT( TargetFile, ':p.');
             END;
             /* ---- End workaround for NewView */

             rcx = LINEOUT( TargetFile, Pages.p._AutoLink);
          END;
       END;

       /* replace links in contents */
       Pages.p._Contents = ReplaceLinks( Pages.p._Contents);

       /* maybe replace leading linebreaks with parbreaks in contents of page */
       ParbreakCount = 0;
       DO WHILE (WORDPOS( LEFT( Pages.p._Contents, EmptyLineLen), ':p.'CrLf '.br'CrLf) > 0)
          /* strip off and count */
          Pages.p._Contents = RIGHT( Pages.p._Contents, LENGTH( Pages.p._Contents) - EmptyLineLen);
          ParBreakCount = ParBreakCount + 1;
       END;
       /* prepend with parbreaks */
       Pages.p._Contents = COPIES( ':p.'CrLf, ParBreakCount)''Pages.p._Contents;

       /* maybe strip trailing empty parbreaks or linebreaks from contents of page */
       DO WHILE (WORDPOS( RIGHT( Pages.p._Contents, EmptyLineLen), CrLf':p. 'CrLf'.br') > 0)
          Pages.p._Contents = LEFT( Pages.p._Contents, LENGTH( Pages.p._Contents) - EmptyLineLen);
       END;

       /* ensure that page is not empty, otherwise links to this page */
       /* would link to the next page in IBM's VIEW (not in NewView)  */
       IF (Pages.p._Contents = '') THEN
          Pages.p._Contents = ':p.'

       /* write contents of page */
       rcx = LINEOUT( TargetFile, Pages.p._Contents);

    END; /* p = 1 TO Pages.0 */

    rcx = LINEOUT( TargetFile, ':euserdoc.');
    rcx = STREAM( TargetFile, 'C', 'CLOSE');

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    /* write RCH file */
    IF (Flag.fRchFile) THEN
    DO
       BaseName = TRANSLATE( TRANSLATE( FILESPEC( 'N', RchFile)), '_', '.');
       rcx = SysFileDelete( RchFile);
       rcx = LINEOUT( RchFile, '');
       rcx = LINEOUT( RchFile, '/*' FILESPEC( 'N', RchFile) 'generated by HyperText/2 -' DATE() TIME() '*/');
       rcx = LINEOUT( RchFile, '');
       rcx = LINEOUT( RchFile, '#ifndef' BaseName);
       rcx = LINEOUT( RchFile, '#define' BaseName);
       rcx = LINEOUT( RchFile, '');

       /* check all pages for resource ids */
       DO p = 1 TO Pages.0
          IF (Pages.p._Resid = '') THEN ITERATE;
          AnchorList = Pages.p._AnchorList;
          IF (AnchorList = '') THEN
          DO
             SAY;
             SAY SourceFile'(***): Warning: no anchor name specified for panel with resource id' Pages.p._Resid'.';
             ITERATE;
          END;
          DO WHILE (AnchorList \= '')
             PARSE VAR AnchorList ThisAnchorName AnchorList ;
             rcx = LINEOUT( RchFile, '#define' RchDefineStem''LEFT( ThisAnchorName, 40) RIGHT( Pages.p._Resid, 6));
          END;
       END;

       rcx = LINEOUT( RchFile, '');
       rcx = LINEOUT( RchFile, '#endif /*' BaseName '*/');
       rcx = LINEOUT( RchFile, '');
       rcx = STREAM( RchFile, 'C', 'CLOSE');
    END;

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    /* now compile - handle strange rc of IPFC */
    IF (Flag.fVerbose) THEN
    DO
       SAY 'Compiling INF.';
       SAY;
    END;

    TmpFile = SysTempFileName( VALUE('TMP',,env)'\htext.???');
    IF POS( ' ', TargetFile) > 0 THEN
       TargetFile = '"'TargetFile'"';
    IF POS( ' ', InfFile) > 0 THEN
       InfFile = '"'InfFile'"';
    IPFC._Exec IpfOptions TargetFile InfFile '>' TmpFile;

    IF ((rc \= ERROR.NO_ERROR) | (Flag.fVerbose)) THEN
       rcx = ShowIPFCOutput( TmpFile);
    rcx = SysFileDelete( TmpFile);

    IF ((rc = ERROR.NO_ERROR) & (\Flag.fNoView)) THEN
       'START VIEW' InfFile;

 END;

 EXIT( rc);

/* ------------------------------------------------------------------------- */
Halt:
 SAY;
 SAY 'Interrupted by user.';

 EXIT(ERROR.GEN_FAILURE);

/* ------------------------------------------------------------------------- */
ShowHelp: PROCEDURE EXPOSE (GlobalVars)

 PARSE SOURCE . . ThisFile

 SAY;
 SAY Title;
 SAY;

 /* skip header */
 DO i = 1 TO 4
    rcx = LINEIN(ThisFile);
 END;

 /* show help */
 DO WHILE (ThisLine \= ' */')
    ThisLine = LINEIN(Thisfile);
    SAY SUBSTR(ThisLine, 7);
 END;

 /* close file */
 rcx = LINEOUT(Thisfile);

 RETURN('');

/* ------------------------------------------------------------------------- */
LastWord: PROCEDURE
 PARSE ARG String;
 IF (String = '') THEN
    RETURN( '');
 ELSE
    RETURN( WORD( String, WORDS( String)));

/* ------------------------------------------------------------------------- */
FileExist: PROCEDURE
 ARG FileName;

 RETURN( STREAM( Filename, 'C', 'QUERY EXISTS') > '');

/* ------------------------------------------------------------------------- */
DirExist: PROCEDURE
   PARSE ARG Dirname;

   Found.0 = 0;
   rcx = SysFileTree( Dirname, 'Found.', 'DO');

   RETURN( Found.0 > 0);

/* ------------------------------------------------------------------------- */
GetCalldir: PROCEDURE
 PARSE SOURCE . . CallName
 CallDir = FILESPEC( 'Drive', CallName)||FILESPEC( 'Path', CallName);
 RETURN( LEFT( CallDir, LENGTH( CallDir) - 1));

/* ========================================================================= */
GetBitmap: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG BitmapFileSpec, Align;

 FoundFilename = '';
 LinkBitmap    = '';
 CurDir        = STRIP( directory(), 'T', '\');
 CallDir       = GetCallDir();

 DO UNTIL (TRUE)
    IF (BitmapFileSpec = '') THEN
       LEAVE;

    IF (TRANSLATE( BitmapFileSpec) = 'NONE') THEN
       LEAVE;

    /* search given path */
    IF (FileExist( BitmapFileSpec)) THEN
    DO
       FoundFilename = BitmapFileSpec;
       LEAVE;
    END;

    SearchDirList = ,
       CurDir';'CurDir'\bmp;'CurDir'\htext;' ||,           /* current dir and subdir */
       CurDir'\..;'CurDir'\..\bmp;'CurDir'\..\htext;' ||,  /* parent dir of current dir and subdir */
       CallDir';'CallDir'\bmp;'CallDir'\htext;'            /* call dir and subdir */

    /* search file in every dir */
    Rest = SearchDirList
    DO WHILE Rest <> ''
       IF Rest = ';' THEN
          LEAVE;

       PARSE VAR Rest NextDir';'Rest;
       IF NextDir = '' THEN
          ITERATE;

       CheckName = NextDir'\'BitmapFileSpec;
       IF (FileExist( CheckName)) THEN
       DO
          FoundFilename = CheckName;
          LEAVE;
       END;
    END;
    IF (FoundFilename \= '') THEN
       LEAVE;

    /* search in environment vars */

    /* search files here without path */
    Filename = FILESPEC( 'N', BitmapFileSpec);

    /* do not use SysSearchPath, it returns fully qualified pathnames */
    FoundFilename = SearchPath( 'INCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
    FoundFilename = SearchPath( 'HTINCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
 END;

 DO UNTIL (TRUE)
    IF (FoundFileName = '') THEN
       LEAVE;
    IF (Flag.fFullPaths) THEN
       LEAVE;
    /* make filename of found bitmap relative to compile dir */
    CompileDir = directory()
    RelFileName = MakeRelativePath( FoundFileName, CompileDir);
    IF RelFileName = '' THEN
       LEAVE;
    FoundFileName = RelFileName;
 END;

 IF (Align = '') THEN
    Align = 'runin';
 ELSE
    Align = 'align='Align;

 SELECT
    /* 'NONE' must be handled different from not found */
    WHEN (TRANSLATE( Filename) = 'NONE') THEN
       LinkBitmap = 'NONE';
    WHEN (FoundFilename \= '') THEN
       LinkBitmap = ":artwork name='"FoundFilename"' "Align".";
    OTHERWISE
       SAY SourceFile'(***): Warning: link bitmap "'Filename'" not found';
 END;

 RETURN( LinkBitmap);

/* ========================================================================= */
/* check for bitmaps */
GetBitmaps: PROCEDURE EXPOSE (GlobalVars)
 Bitmap._Web   = GetBitmap( VALUE( 'WEBBMP',,   env));
 Bitmap._Email = GetBitmap( VALUE( 'EMAILBMP',, env));
 Bitmap._Inf   = GetBitmap( VALUE( 'INFBMP',,   env));
 Bitmap._fBitmapsChecked = TRUE;
 RETURN;

/* ========================================================================= */
GetFullFileName: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG FSpec;

 SELECT
    WHEN (POS( ':\', FSpec) > 0) THEN
       FullFSpec = FSpec;
    WHEN (POS( '\', FSpec) = 0) THEN
    DO
       CurDir = DIRECTORY();
       FullFSpec = STRIP( CurDir, 'T', '\')'\'FSpec;
    END;
 OTHERWISE
    /* not very smart: FSpec must already exist here */
    FullFSpec = STREAM( FSpec, 'C', 'QUERY EXISTS');
    IF FullFSpec = '' THEN
       FullFSpec = FSpec;
 END;

 RETURN (FullFSpec);

/* ========================================================================= */
MakeRelativePath: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG ThisFSpec, BaseFSpec;
 RelFSpec = ThisFSpec;

 /* handle dir name as BaseFSpec parameter */
 IF DirExist( BaseFSpec) THEN
    BaseFSpec = BaseFSpec'\*';

 /* check parameters for fully qualified format */
 SELECT
    WHEN (SUBSTR( ThisFSpec, 2, 1) = ':') THEN
    DO
       ThisFDrive = FILESPEC( 'D', ThisFSpec)
       ThisFPath  = FILESPEC( 'P', ThisFSpec)
       ThisFName  = FILESPEC( 'N', ThisFSpec)
    END
 OTHERWISE
    SAY SourceFile'(***): Warning: filename not fully qualified: 'ThisFSpec;
    RETURN( '');
 END;

 SELECT
    WHEN (SUBSTR( BaseFSpec, 2, 1) = ':') THEN
    DO
       BaseFDrive = FILESPEC( 'D', BaseFSpec)
       BaseFPath  = FILESPEC( 'P', BaseFSpec)
       BaseFName  = FILESPEC( 'N', BaseFSpec)  /* unused */
    END
 OTHERWISE
    SAY SourceFile'(***): Warning: filename not fully qualified: 'BaseFSpec;
    RETURN( '');
 END;

 /* remove equal parts of the paths and build the relative pathname */
 DO UNTIL (TRUE)
    IF (BaseFDrive <> ThisFDrive) THEN
       LEAVE;
    RelFSpec = '';
    ThisRest   = ThisFPath;
    ThisNext   = '';
    BaseRest = BaseFPath;
    BaseNext = '';
    fNextDifferent = 0;
    cParentDir = 0;
    DO FOREVER
       PARSE VAR ThisRest ThisNext'\'ThisRest;
       PARSE VAR BaseRest BaseNext'\'BaseRest;
       /* say 'BaseNext = 'BaseNext', ThisNext = 'ThisNext; */
       IF TRANSLATE( ThisNext) <> TRANSLATE( BaseNext) & \fNextDifferent THEN
          fNextDifferent = 1;
       /* say 'fNextDifferent = 'fNextDifferent; */
       IF fNextDifferent THEN
       DO
          cParentDir = cParentDir + 1;
          IF ThisNext <> '' THEN
             RelFSpec = RelFSpec''ThisNext'\'
       END;
       IF (BaseRest = '') THEN
          LEAVE;
    END;
    RelFSpec = RelFSpec''ThisRest;
    RelFSpec = COPIES( '..\', cParentDir)''RelFSpec''ThisFName;
 END;

 RETURN( RelFSpec);

/* ========================================================================= */
DumpDimensions:
 PARSE ARG x y cx cy;
 RETURN( RIGHT( x, 3) RIGHT( y, 3) RIGHT( cx, 3) RIGHT( cy, 3));

/* ========================================================================= */
ModifyDimensions: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Dimensions, Type, Modifier;

 IF (Modifier = '') THEN
    RETURN( Dimensions);

 fSource = FALSE;
 fTarget = FALSE;

 IF (Dimensions = '') THEN
    Dimensions = '0 0 100 100';

 Type = TRANSLATE( Type);
 SELECT
    WHEN (POS( Type, 'SOURCE') = 1) THEN fSource = TRUE;
    WHEN (POS( Type, 'TARGET') = 1) THEN fTarget = TRUE;
    OTHERWISE RETURN Dimensions;
 END;

 PARSE VAR Dimensions x y cx cy;
 PARSE VAR Modifier Direction +1 Percentage;
 Direction = TRANSLATE( Direction);

/* CALL CHAROUT, Type ':' Modifier '-' DumpDimensions( Dimensions) ' -> '; */

 SELECT
    WHEN (Direction = 'V') THEN
    DO
       IF (fSource) THEN
          cx = Percentage;

       IF (fTarget) THEN
       DO
          x  = x  + Percentage;
          cx = cx - Percentage;
       END;
    END;

    WHEN (Direction = 'H') THEN
    DO
       IF (fSource) THEN
       DO
          IF (cy \= 100) THEN
             y =  cy - Percentage;
          ELSE
             y  = 100 + y - Percentage;
          cy = Percentage;
       END;

       IF (fTarget) THEN
       DO
          y  = 0;
          cy = cy - Percentage;
       END;
    END;
    OTHERWISE NOP;
 END;
 Dimensions = x y cx cy;

/* SAY DumpDimensions( Dimensions); */

 RETURN Dimensions;

/* ========================================================================= */
ChangeExtension: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename, NewExtension;

 /* prepend NewExtension with the leading dot if not specified */
 IF (LEFT( NewExtension, 1) \= '.') THEN
    NewExtension = '.'NewExtension;

 BaseName   = FILESPEC( 'N', Filename);
 BaseExtPos = LASTPOS( '.', BaseName);
 IF (BaseExtPos < 2) THEN
    NewName = FileName''NewExtension;
 ELSE
    NewName = SUBSTR( FileName, 1,,
                      LENGTH( Filename) - LENGTH( BaseName) + BaseExtPos - 1)''NewExtension;

 RETURN( NewName);

/* ========================================================================= */
MakeIPFLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG OldStr, fIsHtextMode, fEscapeCol1Chars, fIgnoreLinks;

 /* make 3rd param optional - only useful for standard and text lines */
 IF (fEscapeCol1Chars <> TRUE) THEN
    fEscapeCol1Chars = FALSE;

 /* make 4th param optional - used when called by ReplaceLinks only */
 IF (fIgnoreLinks <> FALSE) THEN
    fIgnoreLinks = TRUE;

 NewStr = '';
 fLinkOpen = FALSE;
 i = 0;

 /* now check for IPF symbols */
 DO WHILE (OldStr \= '')
    i = i + 1;
    PARSE VAR OldStr char +1 OldStr;
    NextChar = LEFT( OldStr, 1);
    OverNextChar = SUBSTR( OldStr, 2, 1);

    IF (fIsHtextMode) THEN
    SELECT
       /* handle double bracket */
       WHEN ((char = '[') & (NextChar = '[')) THEN
       DO
          NewStr = NewStr''char;
          OldStr = SUBSTR( OldStr, 2);
       END;
       WHEN ((char = ']') & (NextChar = ']')) THEN
       DO
          NewStr = NewStr''char;
          OldStr = SUBSTR( OldStr, 2);
       END;

       /* don't process links here, because links will  */
       /* be processed later for the entire page        */
       WHEN (char = '[' & fIgnoreLinks) THEN
       DO
          NewStr = NewStr''char;
          fLinkOpen = TRUE;
          ITERATE;
       END;
       WHEN (char = ']' & fIgnoreLinks) THEN
       DO
          NewStr = NewStr''char;
          fLinkOpen = FALSE;
          ITERATE;
       END;
       WHEN (fLinkOpen & fIgnoreLinks) THEN
       DO
          NewStr = NewStr''char;
          ITERATE;
       END;

       OTHERWISE NOP;
    END;

    ELSE  /* for non-htext mode */
    DO
       IF (POS( char, '[]') > 0) THEN
       DO
          /* double link escape chars temporarily here, because */
          /* links will be processed later for the entire page  */
          NewStr = NewStr''char''char;
          ITERATE;
       END;
    END;

    /* check for IPF special char */
    CharIndex = POS( char, IpfSymbol._Chars);
    IF (i = 1 & fEscapeCol1Chars) THEN
       Col1CharIndex = POS( char, IpfSymbol._Col1Chars);
    ELSE
       Col1CharIndex = 0;

    /* replace HTEXT escape chars in HyperText/2 mode */
    IF (fIsHtextMode) THEN
       EcharIndex = POS( char, Char._EscapeChars);
    ELSE
       EcharIndex = 0;

    SELECT
       WHEN (CharIndex > 0) THEN
          /* replace with IPF symbols */
          NewStr = NewStr''WORD( IpfSymbol._Strings, CharIndex);

       WHEN (Col1CharIndex > 0) THEN
          /* replace with IPF symbols if in column 1 */
          NewStr = NewStr''WORD( IpfSymbol._Col1Strings, Col1CharIndex);

       WHEN ((EcharIndex > 0) & (char = NextChar)) THEN
       DO
          /* replace HTEXT escape chars - skip char if given twice */
          NewStr = NewStr''char;
          OldStr = SUBSTR( OldStr, 2);
       END;

       WHEN (EcharIndex > 0) THEN
       DO
          /* any attribute active? - end it first */
          IF (Char._LastAttrs \= '') THEN
          DO
             IpfAttr = GetTextAttr( Char._LastAttrs);
             PARSE VAR IpfAttr StartAttr'|'EndAttr;
             NewStr = NewStr''EndAttr;
          END;

          /* check for switched attribute */
          NewAttr = WORD( Char._Attrs, EcharIndex);
          /* take care for double escape char attributes... */
          NextEcharIndex = POS( NextChar, Char._EscapeChars);
          /* ...but ignore doubled escape chars here */
          IF ((NextEcharIndex > 0) & (NextChar \= OverNextChar)) THEN
          DO
             NewAttr = NewAttr WORD( Char._Attrs, NextEcharIndex);
             OldStr = SUBSTR( OldStr, 2);
          END

          /* change attribute(s) in saved attribute list */
          DO w = 1 to WORDS( NewAttr)
             Attr = WORD( NewAttr, w);
             AttrPos = WORDPOS( Attr, Char._LastAttrs);
             IF (AttrPos = 0) THEN
                /* add attribute */
                Char._LastAttrs = Char._LastAttrs Attr;
             ELSE
                /* remove attribute */
                Char._LastAttrs = STRIP( DELWORD( Char._LastAttrs, AttrPos, 1));
          END;

          /* start with new attribute */
          IF (Char._LastAttrs \= '') THEN
          DO
             IpfAttr = GetTextAttr( Char._LastAttrs);
             PARSE VAR IpfAttr StartAttr'|'EndAttr;
             NewStr = NewStr''StartAttr;
          END;

       END;

       OTHERWISE
          NewStr = NewStr''char;
    END;
 END;

 RETURN( NewStr);

/* ========================================================================= */
/* unused */
MakeUrlLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG OldStr;

 NewStr = '';

 /* now check for IPF symbols */
 DO WHILE (OldStr \= '')
    PARSE VAR OldStr char +1 OldStr;
    DVal = C2D( char);
    /* examine char */
    IF ((DVal <= 32) | (DVal > 127)) THEN
       Char = '%'D2X( DVal);

    NewStr = NewStr''char;
 END;

 RETURN( NewStr);

/* ========================================================================= */
/* Called by ReplaceLinks to process URLs and INF file links */
MakeExternalLink: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Link, LinkTag, LinkData, LinkBitmap, LinkText;

 fLinkTextCopied = FALSE;
 /* check for default bitmap files only once at first usage */
 IF \(Bitmap._fBitmapsChecked) THEN
    CALL GetBitmaps;

 DO UNTIL (TRUE)

    SELECT

       WHEN (WORDPOS( LinkTag, Char._Urls) > 0) THEN
       DO
          PARSE VAR Link LinkSource Rest;
          LinkSource = STRIP( LinkSource);
          Rest       = STRIP( Rest);

          Program = Launch._Exec;
          IF (LinkTag = 'MAILTO') THEN
          DO
             DefaultLinkBitmap = Bitmap._Email;
             DefaultLinkText   = LinkData;
          END;
          ELSE
          DO
             DefaultLinkBitmap = Bitmap._Web;
             DefaultLinkText   = LinkSource;
          END;

          SELECT
             /* LinkBitmap is here already converted   */
             /* to IPF by GetBitmap, except for 'NONE' */
             WHEN (TRANSLATE( LinkBitmap) = 'NONE') THEN
                NOP;
             WHEN (LinkBitmap = '') THEN
                LinkBitmap = DefaultLinkBitmap;
             OTHERWISE NOP;
          END;

          /* if no link text is specified, use */
          /* either rest or link data          */
          IF (LinkText = '') THEN
          DO
             IF (Rest \= '') THEN
                LinkText = Rest;
             ELSE
             DO
                LinkText = DefaultLinkText;
                fLinkTextCopied = TRUE;
             END;
          END;
       END;

       WHEN (WORDPOS( LinkTag, 'INF') > 0) THEN
       DO
          Program = 'view.exe';
          SELECT
             WHEN TRANSLATE( LinkBitmap) = 'NONE' THEN
                NOP;
             WHEN LinkBitmap = '' THEN
                LinkBitmap = Bitmap._Inf;
             OTHERWISE NOP;
          END;

          /* take care for spaces in source filenames */
          SELECT
             WHEN (LEFT( LinkData, 1) = '"') THEN
                PARSE VAR LinkData '"'LinkSource'"' Rest;
             WHEN (LEFT( LinkData, 1) = "'") THEN
                PARSE VAR LinkData "'"LinkSource"'" Rest;
             OTHERWISE
                PARSE VAR LinkData LinkSource Rest;
          END;
          Rest = STRIP( Rest);

          /* search string has to be enquoted when followed by the link text */
          SELECT
             WHEN (LEFT( Rest, 1) = '"') THEN
                PARSE VAR Rest '"'LinkSearch'"' Rest;
             WHEN (LEFT( Rest, 1) = "'") THEN
                PARSE VAR Rest "'"LinkSearch"'" Rest;
             /* if 2nd parm not enquoted, take the rest as search string */
             OTHERWISE
                LinkSearch = Rest;
                Rest       = '';
          END;
          LinkSearch = STRIP( LinkSearch);
          Rest       = STRIP( Rest);

          /* if no link text is specified, use   */
          /* either search string or link itself */
          IF (LinkText = '') THEN
          DO
             SELECT
                WHEN (Rest \= '') THEN
                   LinkText = Rest;
                WHEN (LinkSearch \= '') THEN
                DO
                   LinkText = LinkSearch
                   fLinkTextCopied = TRUE;
                END;
             OTHERWISE
                LinkText = Link
                fLinkTextCopied = TRUE;
             END
          END;

          IF (LinkSearch \= '') THEN
          DO
             /* enquote search strings with special chars */
             IF (VERIFY( LinkSearch, ' &|<>', 'M') > 0) THEN
                LinkSearch = '"'LinkSearch'"';

             /* append search string to filename */
             LinkSource = LinkSource LinkSearch;
          END;

       END;

       OTHERWISE LEAVE;
    END;

    /* take care for IPF symbols in LinkText */
    IF (fLinkTextCopied) THEN
       /* LinkText is unmasked */
       LinkText = MakeIPFLine( LinkText, FALSE);
    ELSE
       LinkText = MakeIPFLine( LinkText, TRUE);

    /* create the program link */
    Bitmap = '';
    Text   = '';
    LinkAttr = "reftype=launch object='"Program"' data='"LinkSource"'"
    IF (TRANSLATE( LinkBitmap) \= 'NONE' & LinkBitmap \= '') THEN
       Bitmap = LinkBitmap""CrLf":artlink."CrLf":link "LinkAttr"."CrLf":eartlink.";
    IF (TRANSLATE( LinkText) \= 'NONE') THEN
       Text = ":link "LinkAttr"."LinkText":elink.";
    SELECT
       WHEN (Bitmap \= '' & Text \= '') THEN
          Link = Bitmap""CrLf""Text;
       WHEN (Bitmap \= '') THEN
          Link = Bitmap;
       WHEN (Text \= '') THEN
          Link = Text;
       OTHERWISE
          SAY SourceFile'(***): Warning: link text or bitmap missing for "'Link'"';
    END;

 END;

 RETURN( Link);

/* ========================================================================= */
/* Called by ReplaceLinks to process URLs links */
AddToListOfLinks: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Url;
 rcx = ERROR.NO_ERROR

 CheckUrl = TRANSLATE( Url);

 /* parse list word-wise to keep it sorted */
 Rest     = Doc._ListOfLinks
 LeftPart = '';
 PrevLink = '';

 DO UNTIL (Rest = '')
    PARSE VAR Rest ThisLink Rest;

    SELECT
       WHEN (CheckUrl = TRANSLATE( ThisLink)) THEN
          /* already in list */
          LEAVE;

       WHEN (CheckUrl > TRANSLATE( PrevLink) & CheckUrl < TRANSLATE( ThisLink)) THEN
       DO
          /* add link to list here */
          Doc._ListOfLinks = STRIP( LeftPart Url ThisLink Rest);
          LEAVE;
       END;

       WHEN (Rest = '') THEN
       DO
          /* append link to list */
          Doc._ListOfLinks = STRIP( LeftPart ThisLink Url);
          LEAVE;
       END;

       OTHERWISE LeftPart = LeftPart ThisLink;

       PrevLink = ThisLink;
    END;

 END;

 RETURN( rcx);

/* ========================================================================= */
ReplaceNewLineEscapes: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG OldStr;

 /* |   -->  line break */
 /* ||  -->  par break  */
 /* ^|  -->  |          */
 NewStr = '';
 DO WHILE (LENGTH( OldStr) > 0)

    IF POS( '|', OldStr) = 0 THEN
    DO
       NewStr = NewStr''OldStr;
       LEAVE;
    END;

    PARSE VAR OldStr Next1'^|'Rest1;
    PARSE VAR OldStr Next2'|'Rest2;

    IF LENGTH( Next2) > LENGTH( Next1) THEN
    DO
       NewStr = NewStr''Next1'|';
       OldStr = Rest1;
    END;
    ELSE
    DO
       NewStr = NewStr''Next2''CrLf;
       OldStr = Rest2;
    END;

 END;

 RETURN( NewStr);

/* ========================================================================= */
GetPageFromAnchor: PROCEDURE EXPOSE (GlobalVars)
 ARG Anchor;

 Page = 0;

 DO UNTIL (TRUE)
    /* anchor name must start with period! */
    IF (LEFT( Anchor, 1) \= '.') THEN
       LEAVE;

    /* prepare anchor name */
    PARSE VAR Anchor . +1 AnchorName;

    /* search link in pages */
    DO p = 1 TO Pages.0
       IF (WORDPOS( AnchorName, Pages.p._AnchorList) > 0) THEN
       DO
          Page = p;
          LEAVE;
       END;
    END;
 END;

 RETURN( Page);

/* ========================================================================= */
Lower: PROCEDURE

 Lower = 'abcdefghijklmnopqrstuvwxyz';
 Upper = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';

 PARSE ARG String
 RETURN( TRANSLATE( String, Lower, Upper));

/* ========================================================================= */
/* This is called                                                      */
/*   -  for a text replacement link (with fIgnoreAttributes = TRUE)    */
/*   -  for the doc title (attribute chars were already stripped off)  */
/*   -  for a page's title (attribute chars were already stripped off) */
/*   -  for the entire content of a page (but not for its title)       */
ReplaceLinks: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Str, fIgnoreAttributes;

 /* defaults */
 IF (fIgnoreAttributes = '') THEN
    fIgnoreAttributes = FALSE;

 fKeepEscapedLinks = fIgnoreAttributes;

 /* check for links and resolve link ecapes */
 startp = 1
 LinkStart = 0;
 LinkEnd   = 0;
 DO FOREVER

    /* search next '[' or ']' char */
    p = VERIFY( Str, '[]', 'M', startp);
    IF p = 0 THEN
       LEAVE;

    Char = SUBSTR( Str, p, 1);
    IF p < LENGTH( Str) THEN
       NextChar = SUBSTR( Str, p + 1, 1);
    ELSE
       NextChar = '';

    SELECT
       /* replace doubled escape char at final call of ReplaceLinks only */
       WHEN (Char = NextChar & fKeepEscapedLinks & LinkStart = 0) THEN
       DO
          /* keep doubled escape char */
          startp = p + 2;
       END;

       WHEN (Char = NextChar) THEN
       DO
          /* remove doubled escape char */
          Str = DELSTR( Str, p, 1);
          startp = p + 1;
       END;

       WHEN (Char = '[' & LinkStart = 0) THEN
       DO
          /* link start found */
          LinkStart = p;
          startp = p + 1;
       END;

       WHEN (Char = ']' & LinkStart > 0) THEN
       DO
          /* link end found */
          LinkEnd = p;
          startp = p + 1;
       END;

       OTHERWISE
          /* keep single bracket */
          startp = p + 1;
    END;

    /* search next bracket if link not complete */
    IF (LinkStart = 0 | LinkEnd = 0) THEN
       ITERATE;

    /* remove link from string */
    Link = SUBSTR( Str, LinkStart + 1, LinkEnd - LinkStart - 1);
    Str  = DELSTR( Str, LinkStart, LinkEnd - LinkStart + 1);

    /* parse link and remove key=value strings from it */
    LinkBitmap = '';
    LinkText   = '';
    Keys = 'TEXT BITMAP';
    DO k = 1 TO WORDS( Keys)
       Key = WORD( Keys, k);

       KeyPos = POS( ' 'Key'=', ' 'TRANSLATE( Link));
       KeyLen = LENGTH( ' 'Key'=');
       IF (KeyPos > 0) THEN
       DO
          Next = SUBSTR( ' 'Link, 2, KeyPos - 1);
          Rest = SUBSTR( ' 'Link, KeyPos + KeyLen);
          /* take care for spaces in Val, either in single or double quotes */
          SELECT
             WHEN (LEFT( Rest, 1) = "'") THEN
                PARSE VAR Rest "'"Val"'" Rest;
             WHEN (LEFT( Rest, 1) = '"') THEN
                PARSE VAR Rest '"'Val'"' Rest;
             OTHERWISE
                PARSE VAR Rest Val Rest;
          END;
          Rest = STRIP( Rest);

          /* set link vars */
          IF Key = 'TEXT' THEN
             LinkText = Val;
          ELSE
             LinkBitmap = GetBitmap( Val);
          Link = STRIP( Next Rest);
       END;
    END;

    CheckLink = TRANSLATE( Link);
    FirstWord = WORD( CheckLink, 1)
    LinkTag = '';
    LinkData = Link;
    IF (POS( ':', FirstWord) > 0) THEN
    DO
       PARSE VAR Link LinkTag':'LinkData;
       LinkTag = TRANSLATE( LinkTag);
    END;
    LinkData = STRIP( LinkData);

    SELECT

       /* is it an anchor name? */
       WHEN (LEFT( Link, 2) = '=.') THEN
       DO
          PARSE VAR Link . +1 AnchorName;
          LinkedPage = GetPageFromAnchor( AnchorName);
          IF (LinkedPage > 0) THEN
             Link = Pages.LinkedPage._RefTitle;

          /* make it IPF conform */
          IF (\fIgnoreAttributes) THEN
             Link = MakeIPFLine( Link, TRUE);
       END;

       /* is it an anchor link? */
       WHEN (LEFT( Link, 1) = '.') THEN
       DO
          IF (LinkText = '') THEN
             PARSE VAR Link LinkAnchor LinkText;
          ELSE
             PARSE VAR Link LinkAnchor .;

          LinkedPage = GetPageFromAnchor( LinkAnchor);
          IF ((LinkedPage > 0) & (STRIP( LinkText) = '')) THEN
             LinkText = Pages.LinkedPage._RefTitle;

          /* make it IPF conform */
          IF (\fIgnoreAttributes) THEN
             LinkText = MakeIPFLine( LinkText, TRUE, , FALSE)

          IF (LinkedPage > 0) THEN
             SELECT
                WHEN (Pages.LinkedPage._Level > 6) THEN
                   LinkAttr = 'reftype=fn refid='LinkedPage;
                OTHERWISE
                   LinkAttr = 'reftype=hd viewport dependent refid='LinkedPage;
             END;
          ELSE
          DO
             PARSE VAR LinkAnchor '.'LinkAnchorName;
             /* allow anchor links to refids outside the resulting ipf source */
             /* this makes sense, when the resulting IPF source is being embedded */
             LinkAttr = 'reftype=hd viewport dependent refid='LinkAnchorName;
          END;

          Bitmap = '';
          Text   = '';
          IF (TRANSLATE( LinkBitmap) \= 'NONE' & LinkBitmap \= '') THEN
             Bitmap = LinkBitmap""CrLf":artlink."CrLf":link "LinkAttr"."CrLf":eartlink.";
          IF (TRANSLATE( LinkText) \= 'NONE') THEN
             Text = ":link "LinkAttr"."LinkText":elink.";

          SELECT
             WHEN (Bitmap \= '' & Text \= '') THEN
                Link = Bitmap""CrLf""Text;
             WHEN (Bitmap \= '') THEN
                Link = Bitmap;
             WHEN (Text \= '') THEN
                Link = Text;
             OTHERWISE
                SAY SourceFile'(***): Warning: link text or bitmap missing for "'Link'"';
          END;
       END;

       /* is it an external link? */
       WHEN (WORDPOS( LinkTag, Char._Urls' INF') > 0) THEN
       DO
          /* add url links to list of links */
          IF (WORDPOS( LinkTag, Char._Urls) > 0) THEN
          DO
             PARSE VAR Link Url .;
             rcx = AddToListOfLinks( Url);
          END;

          Link = MakeExternalLink( Link, LinkTag, LinkData, LinkBitmap, LinkText);
       END;

       /* Handle pseudo link for list of URL links.     */
       /* This must be processed after all other links, */
       /* therefore it's implemented as a pseudo link.  */
       WHEN (Link = 'lOl:') THEN
       DO
          /* add links as unordered compact list */
          Rest        = Doc._ListOfLinks;
          ListOfLinks = ':ul compact.';
          Program     = Launch._Exec;

          DO WHILE (Rest \= '')
             PARSE VAR Rest Next Rest;

             /* LinkText is unmasked */
             LinkText = Next;
             IpfLink = MakeIPFLine( LinkText, FALSE);

             ThisLink = ":link reftype=launch object='"Program"' data='"Next"'."IpfLink":elink.";
             ListOfLinks = ListOfLinks''CrLf':li.'ThisLink;
          END;

          ListOfLinks = ListOfLinks''CrLf':eul.';

          Link = ListOfLinks;
       END;

       OTHERWISE
       DO
          /* search link in page titles */
          fLinkFound  = FALSE;
          DO p = 1 TO Pages.0
             fLinkFound = (TRANSLATE( Pages.p._RefTitle) = CheckLink);

             IF (fLinkFound) THEN
             DO
                SELECT
                   /* footnotes */
                   WHEN (Pages.p._Level > 6) THEN
                      Link = ':link reftype=fn id='p'.'Link':elink.';
                   /* page headers */
                   OTHERWISE
                      Link = ':link reftype=hd viewport dependent refid='p'.'Link':elink.';
                END;
                LEAVE;
             END;
          END;
       END;

    END; /* SELECT */

    /* To make artlinks work, both :artlink. and :artwork. */
    /* must start on new lines.                            */
    LStr = LEFT( Str, LinkStart - 1);
    DO UNTIL (TRUE)
       IF (LStr = '') THEN LEAVE;
       IF (RIGHT( LStr, 2) = CrLf) THEN LEAVE;
       IF (WORD( Link, 1) \= ':artwork') THEN LEAVE;
       Link = CrLf''Link;
    END;

    /* add link to string */
    Str = INSERT( Link, Str, LinkStart - 1);

    /* move for searching next link */
    startp = LinkStart + LENGTH( Link);
    LinkStart = 0;
    LinkEnd   = 0;
 END;

 RETURN( Str);

/* ========================================================================= */
GetTextAttr: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Parms;

 fUseBold       = FALSE;
 fUseItalic     = FALSE;
 fUseUnderlined = FALSE;
 fUseHilite     = FALSE;
 fUseTt         = FALSE;
 TextAttr       = '';

 NewParms = '';
 DO WHILE (Parms \= '')
    PARSE VAR Parms ThisParm Parms;
    ThisParm = TRANSLATE( ThisParm);
    SELECT
       WHEN (POS( ThisParm, 'BOLD')       = 1) THEN fUseBold       = TRUE;
       WHEN (POS( ThisParm, 'ITALIC')     = 1) THEN fUseItalic     = TRUE;
       WHEN (POS( ThisParm, 'UNDERLINED') = 1) THEN fUseUnderlined = TRUE;
       WHEN (POS( ThisParm, 'HILITE')     = 1) THEN fUseHilite     = TRUE;
       WHEN (POS( ThisParm, 'TT')         = 1) THEN fUseTt         = TRUE;
       OTHERWISE NOP;
    END;
 END;
 SELECT
    WHEN (fUseHilite) THEN
       TextAttr = ':hp8.|:ehp8.';
    WHEN ((fUseItalic) & (fUseUnderlined)) THEN
       TextAttr = ':hp6.|:ehp6.';
    WHEN ((fUseBold) & (fUseUnderlined)) THEN
       TextAttr = ':hp7.|:ehp7.';
    WHEN ((fUseBold) & (fUseItalic)) THEN
       TextAttr = ':hp3.|:ehp3.';
    WHEN (fUseBold) THEN
       TextAttr = ':hp2.|:ehp2.';
    WHEN (fUseItalic) THEN
       TextAttr = ':hp1.|:ehp1.';
    WHEN (fUseUnderlined)  THEN
       TextAttr = ':hp5.|:ehp5.';
    WHEN (fUseTt)  THEN
       TextAttr = ':font 'Char._TtAttr'.|:font facename=default.';
    OTHERWISE NOP;
 END;

 RETURN( TextAttr);

/* ========================================================================= */
GetFontAttr: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Parms;

 FontAttrs       = '';
 ForegroundColor = '';
 BackgroundColor = '';
 FontName        = '';
 FontSize        = '';
 FontWidth       = '';

 DO UNTIL (TRUE)
    DO WHILE (Parms \= '')

       PARSE VAR Parms NextWord Rest;
       IF (POS( '=', NextWord) > 0) THEN
       DO
          PARSE VAR Parms ThisTag'='Rest;
          SELECT
             WHEN LEFT( Rest, 1) = "'" THEN
                PARSE VAR Rest "'"ThisVal"'" Parms;
             WHEN LEFT( Rest, 1) = '"' THEN
                PARSE VAR Rest '"'ThisVal'"' Parms;
             OTHERWISE
                PARSE VAR Rest ThisVal Parms;
          END;
       END;
       ELSE
       DO
          PARSE VAR Parms ThisTag Parms;
          ThisVal = '';
       END
       ThisTag = TRANSLATE( ThisTag);

       SELECT
          WHEN (ThisTag = 'FC') THEN
             ForegroundColor = ThisVal;

          WHEN (ThisTag = 'BC') THEN
             BackgroundColor = ThisVal;

          WHEN (POS( ThisTag, 'FACENAME') = 1 | POS( ThisTag, 'FONT') = 1) THEN
          DO
             PARSE VAR ThisVal Next'.'FontName;
             IF (DATATYPE( Next) = 'NUM') THEN
             DO
                FontSize = Next;
                FontWidth = FontSize;  /* View needs both and uses the best match */
             END;
             ELSE
                FontName = ThisVal;
             CheckFontName = TRANSLATE( TRANSLATE( FontName), '_', ' ');
          END;

          WHEN (POS( ThisTag, 'SIZE') = 1) THEN
          DO
             PARSE VALUE TRANSLATE( ThisVal) WITH FontSize'X'FontWidth;
             IF FontWidth = '' THEN
                FontWidth = FontSize;
          END;

          OTHERWISE NOP;
       END;

    END;

    /* validate values */
    /* return error message instead of font attributes ... */
    SELECT
       WHEN ((ForegroundColor \= '') & (WORDPOS( TRANSLATE( ForegroundColor), Valid._Colors) = 0)) THEN
          FontAttrs = 'invalid foreground color' ForegroundColor 'specified.';

       WHEN ((BackgroundColor \= '') & (WORDPOS( TRANSLATE( BackgroundColor), Valid._Colors) = 0)) THEN
          FontAttrs = 'invalid background color' BackgroundColor 'specified.';

       WHEN (TRANSLATE( FontSize) = 'DEFAULT') THEN NOP;

       WHEN ((FontName \= '') & (WORDPOS( CheckFontName, Valid._Fonts) = 0)) THEN
          FontAttrs = 'invalid bitmap font' Fontname 'specified.';

       WHEN ((FontSize \= '') & (DATATYPE( FontSize) \= 'NUM')) THEN
          FontAttrs = 'specified font size 'FontSize' is not numeric.';

       OTHERWISE NOP;
    END;
    IF( FontAttrs \= '') THEN
    DO
       FontAttrs = 'Error:' FontAttrs;
       LEAVE;
    END;

    /* concatenate values */
    SELECT
       WHEN ((ForegroundColor \= '')  & (BackgroundColor \= '')) THEN
          FontAttrs = FontAttrs':color fc='ForegroundColor' bc='BackgroundColor'.';
       WHEN (ForegroundColor \= '') THEN
          FontAttrs = FontAttrs':color fc='ForegroundColor'.';
       WHEN (BackgroundColor \= '') THEN
          FontAttrs = FontAttrs':color bc='BackgroundColor'.';
       OTHERWISE NOP;
    END;
    SELECT
       WHEN (TRANSLATE( FontSize) = 'DEFAULT') THEN
          FontAttrs = FontAttrs':font.'
       WHEN (FontName \= '') THEN
          FontAttrs = FontAttrs":font facename='"FontName"' size="FontSize"x"FontWidth".";
       OTHERWISE NOP;
    END;

 END;

 RETURN( FontAttrs);

/* ========================================================================= */
/* Used for header titles to remove text attributes that would make IPF end */
/* the header tag otherwise */
DiscardEscapeChars: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Str;

 startp = 1;
 DO FOREVER
    p = VERIFY( Str, Char._EscapeChars, 'M', startp);
    IF (p = 0) THEN
       LEAVE;
    ThisChar = SUBSTR( Str, p, 1);
    NextChar = SUBSTR( Str, p + 1, 1);
    /* keep doubled escape char */
    IF ThisChar = NextChar THEN
       startp = p + 2;
    ELSE
    /* remove single escape char */
    DO
       Str = DELSTR( Str, p, 1);
       startp = p;
    END;
 END;

 RETURN( Str);

/* ========================================================================= */
/* Used for C include files. */
DiscardQuotes: PROCEDURE
 PARSE ARG Line;

 fQuotesOpen = 0;

 cpos = POS( '"', Line);
 DO WHILE (cpos > 0)

    Line = DELSTR( Line, cpos, 1);
    fQuotesOpen = \fQuotesOpen;

    /* check for either trailing or leading blanks */
    IF (fQuotesOpen) THEN
       CheckPos = cpos - 1;
    ELSE
       CheckPos = cpos;

    IF ((CheckPos > 0) & (CheckPos < LENGTH( Line))) THEN
    DO
       DO WHILE (SUBSTR( Line, CheckPos, 1) = ' ')
          Line = DELSTR( Line, CheckPos, 1);

          /* for opening quotes, adjust check position */
          IF (fQuotesOpen) THEN
             CheckPos = CheckPos - 1;

       END;
    END;

    cpos = POS( '"', Line);

 END;

 /* remove tilde chars */
 cpos = POS( '~', Line);
 DO WHILE (cpos > 0)
    Line = DELSTR( Line, cpos, 1);
    cpos = POS( '~', Line);
 END

 /* remove all after tab */
 cpos = POS( '\t', Line);
 IF (cpos > 0) THEN
    Line = LEFT( Line, cpos - 1);

 RETURN( Line);

/* ========================================================================= */
DiscardTextAttrs: PROCEDURE
 PARSE ARG Parms

 NewParms = '';
 DO WHILE (Parms \= '')
    PARSE VAR Parms ThisParm Parms;
    CheckParm = TRANSLATE( ThisParm);
    SELECT
       WHEN (POS( CheckParm, 'BOLD')       = 1) THEN NOP;
       WHEN (POS( CheckParm, 'ITALIC')     = 1) THEN NOP;
       WHEN (POS( CheckParm, 'UNDERLINED') = 1) THEN NOP;
       WHEN (POS( CheckParm, 'HILITE')     = 1) THEN NOP;
       WHEN (POS( CheckParm, 'TT')         = 1) THEN NOP;
       OTHERWISE
          IF NewParms = '' THEN
             NewParms = ThisParm;
          ELSE
             NewParms = NewParms ThisParm;
    END;
 END;

 RETURN( NewParms);


/* ========================================================================= */
/* clone of SysSearchPath, which does not return fully qualified name */
SearchPath: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG EnvVar, Filename;

 FoundFilename = '';

 DO UNTIL (TRUE)
    EnvPath = VALUE( EnvVar,,env);

    DO WHILE (EnvPath \= '')
       PARSE VAR EnvPath ThisDir';'EnvPath;
       ThisFile = ThisDir'\'Filename;
       IF (FileExist( ThisFile)) THEN
       DO
          FoundFilename = ThisFile;
          LEAVE;
       END;
    END;
 END;

 RETURN( FoundFilename);

/* ========================================================================= */
SearchIncludeFile: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename;

 FoundFilename = '';
 Filename = STRIP( Filename);

 DO UNTIL (TRUE)

    /* check filename as is */
    IF (FileExist( Filename)) THEN
    DO
       FoundFilename = Filename;
       LEAVE;
    END;

    /* if drive letter given, we cannot search further */
    IF (POS( ':',  Filename) > 1) THEN
       LEAVE;

    /* search in environment vars */
    /* do not use SysSearchPath, it returns fully qualified pathnames */
    FoundFilename = SearchPath( 'INCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
    FoundFilename = SearchPath( 'HTINCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;

 END;

 RETURN( FoundFilename);

/* ========================================================================= */
EvaluateCondition: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG CompileCondition

 Condition.0 = 0;

 DO WHILE (CompileCondition \= '')
    PARSE VAR CompileCondition ThisAndCondition'&'CompileCondition;
    DO WHILE (ThisAndCondition \= '')
       PARSE VAR ThisAndCondition ThisOrCondition'|'ThisAndCondition;
       c           = Condition.0 + 1;
       Condition.c = STRIP( ThisOrCondition);
       Condition.0 = c;
       IF (ThisAndCondition = '') THEN
          Condition.c._Operator = '&';
       ELSE
          Condition.c._Operator = '|';
    END;
 END;

 /* check all conditions and reconcatenate */
 NewCondition = '';
 LastOperator = '';
 DO c = 1 TO Condition.0
    PARSE VAR Condition.c SymbolName'='SymbolValue;
    SymbolName  = TRANSLATE( STRIP( SymbolName));
    SymbolValue = STRIP( SymbolValue);
    IF (SymbolValue = '') THEN
       /* just return symbol value */
       Condition.c._Value = VALUE( SymbolName,,env);
    ELSE
       Condition.c._Value = (VALUE( SymbolName,,env) = SymbolValue);

    NewCondition = NewCondition LastOperator Condition.c._Value;
    LastOperator = Condition.c._Operator;
 END;

 RETURN( NewCondition);

/* ========================================================================= */
AddHeaderNum: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG HLevel, HTitle;

 AlphaNumLevels = Pages._HNumbering;
 /* find specified alpha levels */
 AlphaLevels = '';
 startp = 1;
 DO FOREVER
    /* A23456 or B3456 instead of 123456 specified? */
    p = VERIFY( AlphaNumLevels, 'ABCDEF', 'M', startp);

    IF p = 0 THEN
       LEAVE;
    ELSE
    DO
       chr = SUBSTR( AlphaNumLevels, p, 1);
       num = C2D( chr) - 64;
       rest = SUBSTR( AlphaNumLevels, p + 1);
       AlphaNumLevels = LEFT( AlphaNumLevels, p - 1)''num''rest;
       AlphaLevels = AlphaLevels''num;
       startp = p + 1;
    END;
 END;
 /* AlphaNumLevels contains here only numbers, AlphaLevels contains */
 /* those numbers where alphabetic levels were specified            */

 /* The current number is saved in a global var with the following syntax: */
 /* Pages._CurNumber = '3.1.0.0.0.0.'                                      */
 Rest = Pages._CurNumber;
 NewNumber = '';
 HPrefix   = '';
 fLevelFound = 0;
 i = 0;  /* header level */

 /* advance _CurNumber */
 DO WHILE (LENGTH( Rest) > 0)
    i = i + 1;
    PARSE VAR Rest num'.'Rest;

    SELECT
       WHEN fLevelFound = 1 THEN
          /* set all following sublevels to 0 */
          num = 0;
       WHEN i = HLevel THEN
       DO
          fLevelFound = 1;
          /* advance found level by 1 */
          num = num + 1;
       END;
       OTHERWISE
          NOP;
    END;

    /* append current level for global var */
    NewNumber = NewNumber''num'.';

    /* convert to alphabetic number if specified */
    SELECT
       WHEN ((POS( i, AlphaLevels) > 0) & (num <> 0)) THEN
          alphanum = D2C( num + 64);
       WHEN (POS( i, AlphaNumLevels) > 0) THEN
          alphanum = num;
       OTHERWISE
          alphanum = 0;
    END;

    /* append current level for header prefix */
    HPrefix   = HPrefix''alphanum'.';
 END;

 /* save new value to global var */
 Pages._CurNumber = NewNumber;

 IF (POS( HLevel, AlphaNumLevels) > 0) THEN
 DO
    /* strip all "0." numbers */
    Rest = HPrefix;
    HPrefix = '';
    DO WHILE (LENGTH( Rest) > 0)
       PARSE VAR Rest next'.'Rest;
       IF NEXT <> 0 THEN
          HPrefix = HPrefix''next'.';
    END;
    /* strip trailing "." */
    HPrefix = STRIP( HPrefix, 'T', '.');

    IF (LENGTH( HPrefix) > 0) THEN
    DO
       /* prepend HPrefix to HTitle */
       HTitle = HPrefix'  'HTitle;
    END;
 END;

 RETURN( HTitle);

/* ========================================================================= */
FindCommandInList: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Wrd, List

 FoundCommand = '';
 DO i = 1 TO WORDS( List)
    Command = WORD( List, i);
    IF ((POS( Wrd, Command) = 1) & (Wrd = Command | LENGTH( Wrd) > 2)) THEN
    DO
       FoundCommand = Command;
       LEAVE;
    END;
 END;

 RETURN( FoundCommand);

/* ========================================================================= */
GetThisNewPar: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG CurLine;

 CurLine = STRIP( CurLine);
 /* get first word, but including a possible space on column 1 */
 p2 = WORDINDEX( CurLine, 2)
 IF (p2 > 0) THEN
    PARSE VAR CurLine Wrd1 =(p2) .;
 ELSE
    Wrd1 = CurLine;
 Wrd1 = STRIP( TRANSLATE( Wrd1), 'T');
 Wrd2 = STRIP( WORD( CurLine, 2));

 /* these commands set ThisPendingNewPar if PendingNewPar \= '' */
 /* .LINES was added for compatibility with NewView 2.19.3, because */
 /* IBMview always adds an empty line before it and NewView doesn't */
 HModeCommands = '.BITMAP .CENTER .LINES';

 /* these commands set PendingNewPar = '' */
 /* included abbreviation of .PAR, because length is < 2 */
 VModeCommands = '. .P .PAR .OLIST .ULIST .SLIST .PLIST .TABLE .FORMAT';
 ItemBullets   = '-';

 /* these commands set ThisPendingNewPar = ':p.' */
 ReqLeadingNewParCommands = '.NOTE .HRULE';

 /* these commands set PendingNewPar = ':p.' */
 ReqTrailingNewParCommands = '.NOTE .HRULE .1 .2 .3 .4 .5 .6 .FN .ELIST';

 /* these commands ignore following empty lines */
 /* included abbreviation of .PAR, because length is < 2 */
 NoTrailingNewParCommands = '.P .PAR .FORMAT';

 /* other commands don't change PendingNewPar */

 /* save previous value */
 PrevPendingNewPar = PendingNewPar;
 /* default is to not prepend a new par for the current line */
 ThisNewPar = '';
 /* default is to keep PendingNewPar */
 SELECT

    /* empty line */
    WHEN (Wrd1 = '') THEN
    DO
       /* for current line: don't prepend here a newpar,   */
       /* for following lines: add a new par for one empty */
       /* line only and ignore multiple empty lines        */
       IF (PendingNewPar = '') THEN
       DO
          PendingNewPar = ':p.';

          /* check PrevCommand */
          IF PrevCommand \= '' THEN
          DO
             /* empty line gobbled after these commands */
             FoundCommand = FindCommandInList( PrevCommand, NoTrailingNewParCommands);
             IF (FoundCommand \= '') THEN
                PendingNewPar = '';
          END;
       END;
    END;

    /* handle list items only when a list is open */
    WHEN (FindCommandInList( Wrd1, ItemBullets) \= '' & List.0 > 0) THEN
    DO
       /* new par not required, so reset PendingNewPar */
       PrevCommand = Wrd1;
       PendingNewPar = '';
    END;

    /* check command either for HMode or VMode */
    WHEN (LEFT( CurLine, 1) = '.') THEN
    DO UNTIL (TRUE)
       fFound = FALSE;

       FoundCommand = FindCommandInList( Wrd1, VModeCommands);
       IF (FoundCommand \= '') THEN
       DO
          /* new par not required, so reset PendingNewPar */
          PrevCommand = FoundCommand;
          PendingNewPar = '';
          LEAVE;
       END;

       FoundCommand = FindCommandInList( Wrd1, HModeCommands);
       IF (FoundCommand \= '') THEN
       DO
          /* start a new par now if PendingNewPar was set before */
          PrevCommand = FoundCommand;
          /* maybe prepend a previously set newpar */
          ThisNewPar = PrevPendingNewPar;
          /* after that, reset PendingNewPar */
          PendingNewPar = '';
          LEAVE;
       END;

       /* for other commands: don't change PendingNewPar and don't prepend it now, */
       /* when not found in on of the Req*NewParCommands lists at the bottom       */
    END;

    /* not a command */
    OTHERWISE
    DO
       PrevCommand = '';
       /* standard line: process like HModeCommands */
       /* maybe prepend a previously set newpar */
       ThisNewPar = PrevPendingNewPar;
       /* after that, reset PendingNewPar */
       PendingNewPar = '';
    END;
 END;

 /* check command for required leading and/or trailing newpar */
 IF (LEFT( CurLine, 1) = '.') THEN
 DO

    FoundCommand = FindCommandInList( Wrd1, ReqLeadingNewParCommands);
    IF (FoundCommand \= '') THEN
    DO

       /* force to prepend a newpar */
       PrevCommand = FoundCommand;
       /* prepend newpar */
       ThisNewPar = ':p.';
       /* after that, reset PendingNewPar */
       PendingNewPar = '';
    END;


    FoundCommand = FindCommandInList( Wrd1, ReqTrailingNewParCommands);
    IF (FoundCommand \= '') THEN
    DO
       /* force to prepend a newpar for following lines */
       PrevCommand = FoundCommand;
       /* set newpar for following lines */
       PendingNewPar = ':p.';
       /* set a newpar after lists and a newline after compact lists */
       IF (FoundCommand = '.ELIST') THEN
          PendingNewPar = ListPendingNewPar;
    END;

 END;

 RETURN( ThisNewPar);

/* ========================================================================= */
/* resolve env vars only; links and escape chars are processed later */
ReplaceLinkEnvVars: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Str;

 startp = 1;
 p1 = '';
 DO UNTIL p1 = 0;
    /* find '[=' */
    p1 = POS( '[=', Str, startp);

    /* nothing to resolve? */
    IF (p1 = 0) THEN
         LEAVE;

    /* ignore doubled escape char */
    IF p1 > 1 THEN
    DO
       IF SUBSTR( Str, p1 - 1, 2) = '[[' THEN
       DO
          startp = p1 + 2;
          ITERATE;
       END;
    END;

    /* ignore anchor link */
    IF SUBSTR( Str, p1, 3) = '[=.' THEN
    DO
       startp = p1 + 3;
       ITERATE;
    END;

    /* found, now find ']' */
    startp = p1 + 2;

    p2 = ''
    DO UNTIL p2 = 0;
       p2 = POS( ']', Str, startp);

       /* nothing to resolve? */
       IF (p2 = 0) THEN
           LEAVE;

       /* ignore doubled escape char */
       IF SUBSTR( Str, p2, 2) = ']]' THEN
       DO
          startp = p2 + 2;
          ITERATE;
       END;

       /* found, now resolve env var */
       LPart   = SUBSTR( Str, 1, p1 - 1);
       RPart   = SUBSTR( Str, p2 + 1);
       VarName = SUBSTR( Str, p1 + 2, p2 - p1 - 2);
       VarVal  = VALUE( VarName,, env);
       Str = LPart''ReplaceNewLineEscapes( VarVal);
       startp = LENGTH( Str) + 1;
       Str = Str''RPart;
       LEAVE;
    END;

 END;

 RETURN( Str);

/* ========================================================================= */
ProcessSublink: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG CommandParms;

 rcx = ERROR.NO_ERROR;
 p = Pages.0;

 IF (Pages.p._SublinkPos > -1) THEN
 DO
    SAY SourceFile'('LineCount'): Error: .SUBLINK command used more than once within same page.';
    rcx = ERROR.INVALID_DATA;
    LEAVE;
 END;

 Pages.p._SublinkPos     = LENGTH( Pages.p._Contents);
 Pages.p._SublinkType    = Char._ListAttrUnordered;
 Pages.p._SublinkPercent = '';
 Pages.p._SublinkAnchor  = '';

 /* are attributes valid? */

 DO WHILE (CommandParms \= '')
    PARSE VAR CommandParms ThisParm CommandParms;
    ThisParm = TRANSLATE( ThisParm);
    SELECT
       WHEN (POS( LEFT( ThisParm, 1), 'HV') > 0) THEN
       DO
         IF (Pages.p._SublinkAnchor \= '') THEN
         DO
            SAY SourceFile'('LineCount'): Error: .SUBLINK parameters for both anchor name and split windows specified.';
            rcx = ERROR.INVALID_DATA;
            LEAVE;
         END;
          PARSE VAR ThisParm . +1 Percentage;
          IF (DATATYPE( Percentage) \= 'NUM') THEN
          DO
             SAY SourceFile'('LineCount'): Error: invalid attribute for .SUBLINK command: no numeric percentage specified';
             rcx = ERROR.INVALID_DATA;
             ITERATE;
          END;
          IF ((Percentage < 20) | (Percentage > 80)) THEN
          DO
             SAY SourceFile'('LineCount'): Error: invalid attribute for .SUBLINK command: percentage must be between 20 and 80.';
             rcx = ERROR.INVALID_DATA;
             ITERATE;
          END;

          Pages.p._SublinkPercent = ThisParm;
       END;

       WHEN (DATATYPE( ThisParm) = 'NUM') THEN
       DO
          Pages.p._SublinkLevels = ThisParm;
       END;

       WHEN (LEFT( ThisParm, 1) = '.') THEN
       DO
          IF (Pages.p._SublinkPercent \= '') THEN
          DO
             SAY SourceFile'('LineCount'): Error: .SUBLINK parameters for both anchor name and split windows specified.';
             rcx = ERROR.INVALID_DATA;
             LEAVE;
          END;
          Pages.p._SublinkAnchor = ThisParm;
       END;

       WHEN (POS( ThisParm, 'CLEAR') = 1) THEN
          Pages.p._Clear = 1;

       WHEN (POS( ThisParm, 'ORDERED') = 1) THEN
          Pages.p._SublinkType = Char._ListAttrOrdered;

       WHEN (POS( ThisParm, 'UNORDERED') = 1) THEN
          Pages.p._SublinkType = Char._ListAttrUnordered;

       WHEN (POS( ThisParm, 'SIMPLE') = 1) THEN
          Pages.p._SublinkType = Char._ListAttrSimple;

       WHEN (POS( ThisParm, 'BREAKS') = 1) THEN
          Pages.p._SublinkType = Char._ListAttrBreaks;

       WHEN (POS( ThisParm, 'NOLIST') = 1) THEN
          Pages.p._SublinkType = Char._ListAttrNoList;

       OTHERWISE
       DO
          SAY SourceFile'('LineCount'): Error: invalid attribute for .SUBLINK command.';
          rcx = ERROR.INVALID_DATA;
       END;
    END; /* SELECT */
 END; /* WHILE (CommandParms \= '') */

 RETURN( rcx);

/* ========================================================================= */
ProcessStandardLine: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG ThisLine;

 /* check for list items */
 FirstWord = WORD( ThisLine, 1);
 NewList._Pos = WORDINDEX( ThisLine, 1);

 /* Keep indent for non-list lines */
 IF (List.0 > 0) THEN
    ThisLine = STRIP( ThisLine);

 x = List.0
 SELECT
    WHEN (List.0 = 0) THEN
       fIsItem = FALSE;

    WHEN (List.x._Type = 't') THEN
       fIsItem = (WORDPOS( FirstWord, '- =') > 0);

    WHEN (List.x._Type = 'p') THEN
       fIsItem = (WORDPOS( FirstWord, '- =') > 0);

    OTHERWISE
       fIsItem = (FirstWord = '-');
 END;

 IF (fIsItem) THEN
 DO

    /* is this the current list level? */
    /* list type may change! */
    CurrentList._Pos = LastWord( List.x._Pos);

    SELECT
       /* new list - do nothing */
       WHEN (CurrentList._Pos = '') THEN
          List.x._Pos = NewList._Pos;

       /* go one level deeper */
       WHEN (NewList._Pos > CurrentList._Pos) THEN
       DO
          List.x._Pos = List.x._Pos NewList._Pos;
          p = Pages.0;
          IF (LENGTH( Pages.p._Contents) > 0) THEN
             Pages.p._Contents = Pages.p._Contents''CrLf;
          Pages.p._Contents = Pages.p._Contents''List.x._Start;
       END;

       /* keep same level */
       WHEN (NewList._Pos = CurrentList._Pos) THEN
          NOP;

       /* go up again */
       WHEN (NewList._Pos < CurrentList._Pos) THEN
       DO
          TodoList = REVERSE( List.x._Pos);
          DO WHILE (TodoList \= '')
             PARSE VAR TodoList ThisReversePos TodoList;
             ThisPos = REVERSE( ThisReversePos);
             IF (NewList._Pos < ThisPos) THEN
             DO
                p = Pages.0;
                IF (LENGTH( Pages.p._Contents) > 0) THEN
                   Pages.p._Contents = Pages.p._Contents''CrLf;
                Pages.p._Contents = Pages.p._Contents''List.x._End;

                /* if lowest level of this list reached, close it */
                IF (TodoList = '') THEN
                DO
                   x      = List.0 - 1;
                   List.0 = x;
                END;
             END;
             ELSE
                /* level reached */
             DO
                List.x._Pos  =  REVERSE( ThisPos TodoList);
                TodoList = '';
             END;
          END;
       END;
    END;

    /* prepare line */
    SELECT

       WHEN (List.x._Type = 't') THEN
       DO
          SELECT
             WHEN (FirstWord = '-') THEN
             DO
                IF (ThisLine = '-') THEN
                   ThisLine = ':row.';
                ELSE
                   ThisLine = ':row.'CrLf':c.'MakeIPFLine( SUBSTR( STRIP( ThisLine), 3), TRUE);
             END;

             WHEN (FirstWord = '=') THEN
             DO
                ThisLine = ':c.'MakeIPFLine( SUBSTR( STRIP( ThisLine), 3), TRUE);
             END;

             OTHERWISE
                ThisLine = MakeIPFLine( STRIP( ThisLine), TRUE, TRUE);
          END;
       END;

       WHEN (List.x._Type = 'p') THEN
       DO
          SELECT
             WHEN (FirstWord = '-') THEN
             DO
                PARSE VAR List.x._TAttr AttrStart'|'AttrEnd;
                ThisLine = ':pt.'AttrStart''MakeIPFLine( SUBSTR( STRIP( ThisLine), 3), TRUE)''AttrEnd;
                /* append a space to ensure that the description text is separated  */
                /* by 2 spaces from the term text (required for non-breaking lists) */
                ThisLine = ThisLine''IpfSymbol._Space;
             END;

             WHEN (FirstWord = '=') THEN
             DO
                ThisLine = ':pd.'MakeIPFLine( SUBSTR( STRIP( ThisLine), 3), TRUE);
             END;

             OTHERWISE
                ThisLine = MakeIPFLine( STRIP( ThisLine), TRUE, TRUE);
          END;
       END;

       OTHERWISE
          ThisLine = ':li.'MakeIPFLine( SUBSTR( ThisLine, 3), TRUE);
    END;
 END;

 ELSE /* not a list item */
    ThisLine = MakeIPFLine( ThisLine, TRUE, TRUE);

 RETURN( ThisLine);

/* ========================================================================= */
ProcessSourceFile: PROCEDURE EXPOSE (GlobalVars)

 PrevSourceFile = SourceFile;
 PrevLineCount  = LineCount;
 PrevStartLine  = StartLine;
 PrevEndLine    = EndLine;

 PARSE ARG SourceFile, TargetFile, IncludeLevel, IncludeType, StartLine, EndLine;
 /* SourceFile, LineCount, StartLine and EndLine are global */

 LineCount     = 0;
 rc            = ERROR.NO_ERROR;
 fTrace        = FALSE;

 Bullet = '-';
 IF (IncludeType = 'T') THEN
    Bullet = '=';

 IF (Flag.fVerbose) THEN
    SAY COPIES( ' ', IncludeLevel * 2)Bullet SourceFile;

 IncludeLevel = IncludeLevel + 1;
 rcx = STREAM( SourceFile, 'C', 'OPEN READ');
 DO WHILE (LINES( SourceFile));

    /* read line */
    LineCount = LineCount + 1;
    ThisLine = LINEIN( SourceFile);
    IF (LineCount < StartLine & StartLine > '') THEN ITERATE;
    IF (LineCount > EndLine & EndLine > '') THEN LEAVE;

    /* don't do anything with pure text include */
    IF (IncludeType = 'T') THEN
    DO
       p = Pages.0;
       IF (LENGTH( Pages.p._Contents) > 0) THEN
          Pages.p._Contents = Pages.p._Contents''CrLf;
       Pages.p._Contents = Pages.p._Contents''MakeIPFLine( ThisLine, FALSE, TRUE);
       ITERATE;
    END;

    IF (IncludeType = 'C') THEN
    DO
       /* check for #defines */
       PARSE VAR ThisLine '#' ThisLine;
       IF (ThisLine = '') THEN ITERATE;
       ThisLine = STRIP( ThisLine);
       IF (TRANSLATE( WORD( ThisLine, 1)) \= 'DEFINE') THEN ITERATE;
       PARSE VAR ThisLine . ThisLine;
       ThisLine = STRIP( ThisLine);

       /* check for concatenated lines */
       DO WHILE ((RIGHT( ThisLine, 1) = '\') & (LINES( SourceFile)))
          NewLine = STRIP( LINEIN( SourceFile));
          ThisLine = ThisLine NewLine;
          LineCount = LineCount + 1;
       END;

       /* split up name and value */
       PARSE VAR ThisLine VarName VarValue;

       /* replace all known C symbols */
       TodoList = CSymbolList;
       DO WHILE (TodoList \= '')
          PARSE VAR TodoList ThisSymbol TodoList;
          spos = POS( ThisSymbol, VarValue);
          DO WHILE (spos > 0)
             VarValue = DELSTR( VarValue, spos, LENGTH( ThisSymbol));
             VarValue = INSERT( VALUE(ThisSymbol,,env), VarValue, spos);
             spos = POS( ThisSymbol, VarValue);
          END;
       END;

       /* remove all double quotes */
       VarValue = DiscardQuotes( STRIP( VarValue));

       /* add var to environment */
       IF (WORDPOS( VarName, CSymbolList) = 0) THEN
          CSymbolList = CSymbolList VarName;
       rcx = VALUE( VarName, VarValue, env);

       ITERATE;
    END;

    /* filter out EXTPROC command setting */
    IF (LineCount = 1) THEN
       IF (TRANSLATE( WORD( ThisLine, 1)) = 'EXTPROC') THEN
          ITERATE;

    /* skip comments */
    IF \Pages._fTextMode THEN
       IF (LEFT( ThisLine, 2) = '..') THEN ITERATE;

    /* is trace activated? */
    IF (fTrace) THEN
       TRACE ?i;

    /* expand env vars before the line processing loop to */
    /* handle multiple lines specified as env var value   */
    IF Pages._fTextMode THEN
       NextLines = ThisLine;
    ELSE
       NextLines = ReplaceLinkEnvVars( ThisLine);

    /* process NextLines first before reading the next line of SourcFile */
    DO UNTIL (LENGTH( NextLines) = 0)

       /* get next line */
       PARSE VAR NextLines ThisLine(CrLf)NextLines;

       /* get first word and rest, but including a possible space on column 1 */
       CommandParms = '';
       p2 = WORDINDEX( ThisLine, 2);
       IF (p2 > 0) THEN
          PARSE VAR ThisLine Command =(p2) CommandParms;
       ELSE
          Command = ThisLine;
       Command = STRIP( Command, 'T');

       /* resolve text replacement in command parms first */
       IF \(Pages._fTextMode) THEN
          CommandParms = ReplaceLinks( CommandParms, TRUE);

       /* - - - - - - - - - - - - - */

       x = List.0;
       IF (LEFT( ThisLine, 1) = '.') THEN
       DO
          Command = TRANSLATE( Command);
          CommandName = SUBSTR( Command, 2);
          fIsHeader = (DATATYPE( CommandName) = 'NUM');

          SELECT

             /* in text mode, recognize only .FO ON - ignore all others */
             WHEN (Pages._fTextMode) THEN
             DO
                IF (POS( Command, '.FORMAT') = 1 &,
                   TRANSLATE( STRIP( CommandParms)) = 'ON') THEN
                DO
                   ThisLine = ':exmp.';
                   Pages._fTextMode = FALSE;
                   Pages._fMonospaceMode = FALSE;
                END;
                ELSE
                   ThisLine = MakeIpfLine( ThisLine, FALSE, TRUE);
             END;

             /* handle ifdef */
             WHEN (POS( '.IF', Command) = 1) THEN
             DO
                IF (If._fIfOpen) THEN
                DO
                   SAY SourceFile'('LineCount'): Error: nested .IF/.IFDEF/.IFNDEF invalid.';
                   rc = ERROR.FILE_NOT_FOUND;
                   ITERATE;
                END;

                /* determine condition */
                ConditionResult = EvaluateCondition( CommandParms);
                /* SAY '>' CommandParms '->' ConditionResult */
                SELECT
                   WHEN (Command = '.IFDEF')  THEN If._fIncludeSource = (ConditionResult \= '');
                   WHEN (Command = '.IFNDEF') THEN If._fIncludeSource = (ConditionResult  = '');
                   WHEN (Command = '.IF')     THEN INTERPRET( 'If._fIncludeSource = (('ConditionResult') == TRUE);');
                   OTHERWISE
                   DO
                      SAY SourceFile'('LineCount'): Error: invalid command' Command'.';
                      rc = ERROR.FILE_NOT_FOUND;
                      ITERATE;
                   END;
                END;
                If._fIfOpen        = TRUE;
                If._fIfElse        = FALSE;
                ITERATE;
             END;

             /* handle else */
             WHEN (Command = '.ELSE') THEN
             DO
                IF ((\If._fIfOpen) | (If._fIfElse)) THEN
                DO
                   SAY SourceFile'('LineCount'): Error: .ELSE invalid.';
                   rc = ERROR.FILE_NOT_FOUND;
                   ITERATE;
                END;
                If._fIncludeSource = \If._fIncludeSource;
                If._fIfElse        = TRUE;
                ITERATE;
             END;

             /* handle endif */
             WHEN (Command = '.ENDIF') THEN
             DO
                IF (\If._fIfOpen) THEN
                DO
                   SAY SourceFile'('LineCount'): Error: .ENDIF invalid.';
                   rc = ERROR.FILE_NOT_FOUND;
                   ITERATE;
                END;
                If._fIfOpen        = FALSE;
                If._fIfElse        = FALSE;
                If._fIncludeSource = FALSE;
                ITERATE;
             END;

             /* skip section with wrong conditions */
             /* do that after checking for .IF .ELSE .ENDIF! */
             WHEN ((If._fIfOpen) & (\If._fIncludeSource)) THEN ITERATE;

             /* gibe line breaks */
             WHEN (Command = '.') THEN
             DO
                /* handle specified line break when a par break is pending */
                IF (PendingNewPar = ':p.') THEN
                   /* replace line break with a par break */
                   /* IBMView handles both breaks equal, but NewView doesn't */
                   /* handle a leading linebreak correct. Moreover, it looks */
                   /* like the IPF syntax claims that every par must be      */
                   /* started with a parbreak, except for some tags, that    */
                   /* add linebreaks internally.                             */
                   ThisLine = ':p.';
                ELSE
                   /* insert line break */
                   ThisLine = '.br';
             END;

             /* par, useful for pars in list items without ending lists */
             WHEN (POS( Command, '.PAR') = 1) THEN
             DO
                ThisLine = ':p.'MakeIPFLine( CommandParms, TRUE);
             END;

             /* handle headers */
             WHEN (((fIsHeader) & (CommandName < 7)) | (Command = '.FN')) THEN
             DO

                p                       = Pages.0 + 1;
                /* take care for footnotes */
                IF (DATATYPE( CommandName) \= 'NUM') THEN
                   Pages.p._Level = 7;
                ELSE
                DO
                   /* check for a panel title */
                   IF (STRIP( CommandParms) = '') THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: header directive has no title text.';
                      rc = ERROR.INVALID_FORMAT;
                   END;

                   Pages._MaxLevel         = MAX( Pages._MaxLevel, CommandName);
                   Pages.p._Level          = CommandName;
                   /* IPF doesn't allow :hp#.-empasizing in headers,   */
                   /* so discard non-doubled escape chars in title     */
                   /* This is the 2nd call to ReplaceLinks because the */
                   /* titles are processed separate from page contents */
                   Pages.p._RefTitle       = ReplaceLinks( DiscardEscapeChars( CommandParms));
                   Pages.p._Title          = AddHeaderNum( CommandName, Pages.p._RefTitle);
                END;
                Pages.p._Contents       = '';
                Pages.p._MirrorAnchor   = '';
                Pages.p._Index          = (POS( Pages.p._Level, Pages._IndexLevels) > 0);
                Pages.p._Clear          = (POS( Pages.p._Level, Pages._ClearLevels) > 0);
                Pages.p._Hide           = 0;
                Pages.p._Alias          = '';
                Pages.p._SublinkPos     = -1;
                Pages.p._Dimensions     = '';
                Pages.p._SpecifiedDimensions = '';
                Pages.p._SublinkPercent = '';
                Pages.p._SublinkLevels  = 0;
                Pages.p._AnchorList     = '';
                Pages.p._Resid          = '';
                Pages.p._HTrace         = FALSE;
                Pages.0                 = p;

                Pages._fMonospaceMode   = FALSE;
                /* reset previous text attributes */
                Char._LastAttrs = '';
                /* don't iterate here to pass the pending newpar check at the end */
                /* "ThisLine = ''" will not produce an output line */
                ThisLine = '';

             END;

             /* no more command with less than two letters from here */
             WHEN (LENGTH( Command) < 3) THEN
             DO
                SAY SourceFile'('LineCount'): Error: invalid command' Command'.';
                rc = ERROR.FILE_NOT_FOUND;
             END;

             WHEN (POS( Command, '.EXIT') = 1) THEN
             DO
                LEAVE;
             END;

             /* handle setting of variables */
             WHEN (POS( Command, '.SET') = 1) THEN
             DO
                /* check parameters */
                PARSE VAR CommandParms SymbolName'='SymbolValue;
                SymbolName  = STRIP( SymbolName);
                SymbolValue = STRIP( SymbolValue);
                /* better allow reset of env vars as well */
                /*IF ((SymbolName = '') | (SymbolValue = '')) THEN*/
                IF (SymbolName = '') THEN
                DO
                   SAY SourceFile'('LineCount'): Error: invalid SET command.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                   rcx = VALUE( SymbolName, SymbolValue, env);
                ITERATE;
             END;

             /* handle immediate expansion of variables */
             WHEN (POS( Command, '.GET') = 1) THEN
             DO
                /* check parameters */
                SymbolName  = STRIP( CommandParms);
                IF (SymbolName = '') THEN
                DO
                   SAY SourceFile'('LineCount'): Error: invalid GET command.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   NextLines = VALUE( SymbolName,, env);
                   NextLines = ReplaceNewLineEscapes( NextLines);
                   /* process these lines immediately before reading the next lines */
                   /* from the file */
                   ITERATE;
                END;
             END;

             /* handle titles */
             WHEN (POS( Command, '.TITLE') = 1) THEN
            DO
                /* This is the 2nd call to ReplaceLinks because the */
                /* titles are processed separate from page contents */
                Pages._DocTitle = ReplaceLinks( DiscardEscapeChars( CommandParms));
             END;

             /* handle bitmaps */
             WHEN (POS( Command, '.BITMAP') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                DO
                   SAY SourceFile'('LineCount'): Error: no bitmap specified.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   /* get list of bitmaps and align attribute */
                   Align      = '';  /* default is to add 'runin' by GetBitmap */
                   BitmapList = '';
                   ThisLine   = '';
                   BitmapLink = '';
                   DO WHILE (CommandParms \= '')
                      /* take care for spaces in ThisParm, either in single or double quotes */
                      SELECT
                         WHEN (LEFT( CommandParms, 1) = "'") THEN
                            PARSE VAR CommandParms "'"ThisParm"'" CommandParms;
                         WHEN (LEFT( CommandParms, 1) = '"') THEN
                            PARSE VAR CommandParms '"'ThisParm'"' CommandParms;
                         OTHERWISE
                            PARSE VAR CommandParms ThisParm CommandParms;
                      END;
                      UpThisParm = TRANSLATE( ThisParm);
                      SELECT
                         WHEN (POS( UpThisParm, 'LEFT')   = 1) THEN Align = 'left';
                         WHEN (POS( UpThisParm, 'RIGHT')  = 1) THEN Align = 'right';
                         WHEN (POS( UpThisParm, 'CENTER') = 1) THEN Align = 'center';
                         OTHERWISE BitmapList = BitmapList''ThisParm';';
                      END;
                   END;

                   /* get all bitmaps */
                   DO WHILE (BitmapList \= '')
                      PARSE VAR BitmapList ThisBitmap';'BitmapList;
                      /* GetBitmap returns the entire artwork line */
                      ThisLine = ThisLine''GetBitmap( ThisBitmap, Align);
                   END;

                END;
             END;

             /* handle index */
             WHEN (POS( Command, '.INDEX') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                DO
                   p = Pages.0;
                   Pages.p._Index = TRUE;
                   ITERATE;
                END;
                ELSE
                   ThisLine = ':i1.'MakeIPFLine( CommandParms, TRUE);
             END;

             /* handle automatic header index */
             WHEN (POS( Command, '.HINDEX') = 1) THEN
             DO
                Pages._IndexLevels = CommandParms;
                ITERATE;
             END;

             /* handle automatic header clearance */
             WHEN (POS( Command, '.HCLEAR') = 1) THEN
             DO
                Pages._ClearLevels = CommandParms;
                ITERATE;
             END;

             /* handle automatic header numbering */
             WHEN (POS( Command, '.HNUMBERING') = 1) THEN
             DO
                IF TRANSLATE( CommandParms) = 'RESET' THEN
                   Pages._CurNumber  = Pages._ResetNumber;
                ELSE
                   Pages._HNumbering = CommandParms;
                ITERATE;
             END;

             /* handle specified header levels in TOC (default is include all) */
             WHEN (Command = '.HTOC') THEN
             DO
                Pages._HToc = CommandParms;
                ITERATE;
             END;

             /* handle additional escape char definition for ttfont and its deletion */
             WHEN (Command = '.TTCHAR') THEN
             DO
                CommandParms = STRIP( CommandParms);
                IF CommandParms = '' THEN
                DO
                   /* delete from Char._EscapeChars and Char._Attrs */
                   i = WORDPOS( 'tt', Char._Attrs);
                   IF (i > 0) THEN
                   DO
                      Char._EscapeChars = DELSTR( Char._EscapeChars, i, 1);
                      Char._Attrs       = DELWORD( Char._EscapeChars, i, 1);
                   END;
                END
                ELSE
                DO
                   /* append to Char._EscapeChars and Char._Attrs */
                   IF (POS( CommandParms, Char._EscapeChars) = 0) THEN
                      Char._EscapeChars = Char._EscapeChars''STRIP( CommandParms);
                   IF (WORDPOS( 'tt', Char._Attrs) = 0) THEN
                      Char._Attrs       = Char._Attrs' tt';
                END;
                ITERATE;
             END;

             /* handle user-defined ttfont (used by ttchar only) */
             WHEN (Command = '.TTFONT') THEN
             DO
                IF CommandParms = '' THEN
                   Char._TtAttr = Char._DefaultTtAttr;
                ELSE
                   Char._TtAttr = CommandParms;
                ITERATE;
             END;

             /* handle left margin */
             WHEN (POS( Command, '.LMARGIN') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                   Margin = 1;
                ELSE
                DO
                   IF (DATATYPE( CommandParms) \= 'NUM') THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: value for left margin not numeric.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;
                   Margin = CommandParms
                END;
                ThisLine = ':lm margin='Margin'.';
             END;

             /* handle right margin */
             WHEN (POS( Command, '.RMARGIN') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                   Margin = 1;
                ELSE
                DO
                   IF (DATATYPE( CommandParms) \= 'NUM') THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: value for right margin not numeric.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;
                   Margin = CommandParms
                END;
                ThisLine = ':rm margin='Margin'.';
             END;

             /* handle center */
             WHEN (POS( Command, '.CENTER') = 1) THEN
                ThisLine = '.ce' DiscardEscapeChars( CommandParms);

             /* handle external resource ids*/
             WHEN (POS( Command, '.RESID') = 1) THEN
             DO
                IF (Pages.p._Resid \= '') THEN
                DO
                   SAY SourceFile'('LineCount'): Error: resource id already specified.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                IF (DATATYPE( CommandParms) \= 'NUM') THEN
                DO
                   SAY SourceFile'('LineCount'): Error: value for resourceid not numeric.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                Pages.p._Resid = CommandParms;

                ITERATE;
             END;

             /* handle anchor names */
             WHEN (POS( Command, '.ANCHOR') = 1) THEN
             DO
                AnchorName = TRANSLATE( CommandParms);
                SELECT
                   WHEN (STRIP( CommandParms) = '') THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: no anchor name specified.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   WHEN (WORDS( CommandParms) \= 1) THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: anchorname contains blanks.'
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   WHEN (WORDPOS( AnchorName, Pages._AnchorList) > 0) THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: anchorname' CommandParms 'already exists.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   OTHERWISE
                   DO
                      /* store anchor name */
                      p = Pages.0;
                      Pages.p._AnchorList = Pages.p._AnchorList AnchorName;
                      Pages._AnchorList   = Pages._AnchorList AnchorName;
                   END;
                END;
                ITERATE;
             END;

             /* copy content of other section */
             WHEN (POS( Command, '.MIRROR') = 1) THEN
             DO
                AnchorName = TRANSLATE( CommandParms);
                SELECT
                   WHEN (STRIP( CommandParms) = '') THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: no anchor name specified.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   WHEN (WORDS( CommandParms) \= 1) THEN
                   DO
                      SAY SourceFile'('LineCount'): Error: anchorname contains blanks.'
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   OTHERWISE
                   DO
                      /* store anchor name */
                      p = Pages.0;
                      Pages.p._MirrorAnchor = '.'AnchorName;
                   END;
                END;
                ITERATE;
             END;

             /* handle attributes */
             WHEN (POS( Command, '.ATTRIBUTE') = 1) THEN
             DO
                ThisLine = '';
                IpfAttrs = STRIP( GetTextAttr( CommandParms));

                /* cleanup old highlight attributes */
                fFcAttr  = (POS( ' FC=', ' 'TRANSLATE( CommandParms)) > 0);
                IF ((IpfAttrs \= '') | (CommandParms = '') | fFcAttr) THEN
                DO
                   /* any attribute active? - end it first */
                   IF (Char._LastAttrs \= '') THEN
                   DO
                      LastIpfAttrs = GetTextAttr( Char._LastAttrs);
                      PARSE VAR LastIpfAttrs StartAttr'|'EndAttr;
                      ThisLine = EndAttr;
                      Char._LastAttrs = '';
                   END;
                END;

                /* take care for color and font attributes */
                IF (CommandParms = '') THEN
                   ThisLine = ThisLine':font.:color fc=default bc=default.';
                ELSE
                DO
                   /* new font attributes? */
                   FontAttrs = GetFontAttr( CommandParms);
                   SELECT
                      WHEN (WORD( TRANSLATE( FontAttrs), 1) = 'ERROR:') THEN
                      DO
                         SAY SourceFile'('LineCount'):' FontAttrs;
                         rc = ERROR.INVALID_DATA;
                         LEAVE;
                      END;
                      WHEN (FontAttrs \= '') THEN
                         ThisLine = ThisLine''FontAttrs;

                      OTHERWISE NOP;
                   END;
                END;

                IF (IpfAttrs \= '') THEN
                DO
                   /* now set new hilight attributes */
                   PARSE VAR IpfAttrs StartAttr'|'EndAttr;

                   ThisLine = ThisLine''StartAttr;
                   Char._LastAttrs = CommandParms;
                END;

             END;

             /* handle xmp */
             WHEN (POS( Command, '.FORMAT') = 1) THEN
             DO
                CommandParms = TRANSLATE( STRIP( CommandParms));
                Pages._fTextMode = FALSE;
                Pages._fMonospaceMode = TRUE;

                SELECT
                   WHEN (CommandParms = 'TEXT') THEN
                   DO
                      ThisLine = ':xmp.';
                      Pages._fTextMode = TRUE;
                   END

                   WHEN (CommandParms = 'OFF')  THEN
                      ThisLine = ':xmp.';

                   /* for Pages._fTextMode = TRUE, .FO ON is already resolved here */
                   WHEN (CommandParms = 'ON')   THEN
                   DO
                      ThisLine = ':exmp.';
                      Pages._fMonospaceMode = FALSE;
                   END;

                   OTHERWISE
                   DO
                      SAY SourceFile'('LineCount'): Error: invalid parameter for command .FORMAT';
                      rc = ERROR.INVALID_DATA;
                      LEAVE;
                   END;
                END;
             END;

             /* handle trace option */
             WHEN (POS( Command, '.TRACE') = 1) THEN
             DO
                fTrace = TRUE;
                ITERATE;
             END;
             /* handle trace option */
             WHEN (POS( Command, '.HTRACE') = 1) THEN
             DO
                p = Pages.0;
                Pages.p._HTrace = TRUE;
                ITERATE;
             END;

             /* handle hide attribute */
             WHEN (POS( Command, '.HIDE') = 1) THEN
             DO
                p = Pages.0;
                Pages.p._Hide = 1;
                ITERATE;
             END;

             /* handle dimensions attribute */
             WHEN (POS( Command, '.DIMENSIONS') = 1) THEN
             DO
                PARSE VAR CommandParms _x _y _cx _cy _error;
                IF ((_cy = '')                |,
                    (DATATYPE( _x)  \= 'NUM') |,
                    (DATATYPE( _y)  \= 'NUM') |,
                    (DATATYPE( _cx) \= 'NUM') |,
                    (DATATYPE( _cy) \= 'NUM') |,
                    (_error \= '')) THEN
                DO
                   SAY SourceFile'('LineCount'): Error: invalid parameter for command .DIMENSION';
                   rc = ERROR.INVALID_DATA;
                   LEAVE;
                END;
                p = Pages.0;
                Pages.p._Dimensions = CommandParms;
                /* save initially specified value as well to differ from auto-applied dims */
                Pages.p._SpecifiedDimensions = CommandParms;
                ITERATE;
             END;

             /* handle title alias attribute */
             WHEN (POS( Command, '.ALIAS') = 1) THEN
             DO
                p = Pages.0;
                Pages.p._Alias = MakeIPFLine( CommandParms, TRUE);
                ITERATE;
             END;

             /* handle sublink list */
             WHEN (POS( Command, '.SUBLINK') = 1) THEN
             DO
                rcx = ProcessSublink( CommandParms)
                IF (rcx > ERROR.NO_ERROR) THEN
                   rc = rcx;
                ITERATE;
             END;

             /* handle page sections */
             WHEN (POS( Command, '.HRULE') = 1) THEN
             DO
                /* parse CommandParms */
                fRule = TRUE;
                fDefaultAttrib = TRUE;
                FontEscapeChars = '';
                ForegroundColor = '';
                HRuleText = ''
                /* first check if keyword TEXT= is used in CommandParms */
                fTextKeywordPresent = FALSE;
                DO w = 1 TO WORDS( CommandParms)
                   UpWrd = TRANSLATE( WORD( CommandParms, w));
                   IF (LEFT( UpThisParm, 5) = 'TEXT=') THEN
                   DO
                      fTextKeywordPresent = TRUE;
                      LEAVE;
                   END;
                END;
                /* parse CommandParms */
                DO WHILE (CommandParms \= '')
                   PARSE VAR CommandParms ThisParm CommandParms;
                   UpThisParm = TRANSLATE( ThisParm);
                   SELECT
                      WHEN (UpThisParm = 'NORULE') THEN
                         fRule = FALSE;
                      WHEN (UpThisParm = 'NORMAL') THEN
                         fDefaultAttrib = FALSE;
                      WHEN (WORDPOS( UpThisParm, Char._Attrs) > 0) THEN
                      DO
                         wp = WORDPOS( UpThisParm, Char._Attrs);
                         FontEscapeChars = FontEscapeChars''SUBSTR( Char._EscapeChars, wp, 1);
                      END;
                      WHEN (WORDPOS( UpThisParm, Valid._Colors) > 0) THEN
                         ForegroundColor = ThisParm;
                      WHEN (LEFT( UpThisParm, 5) = 'TEXT=') THEN
                      DO
                         PARSE VALUE ThisParm CommandParms WITH 6 CommandParms;
                         /* take care for spaces in HRuleText, either in single or double quotes */
                         SELECT
                            WHEN (LEFT( CommandParms, 1) = "'") THEN
                               PARSE VAR CommandParms "'"HRuleText"'" CommandParms;
                            WHEN (LEFT( CommandParms, 1) = '"') THEN
                               PARSE VAR CommandParms '"'HRuleText'"' CommandParms;
                            OTHERWISE
                               PARSE VAR CommandParms HRuleText CommandParms;
                         END;
                      END;
                      /* any text not known as keyword makes the rest get handled as text */
                      WHEN (\fTextKeywordPresent) THEN
                      DO
                         HRuleText = STRIP( ThisParm CommandParms);
                         CommandParms = '';
                      END;
                   OTHERWISE
                      /* if TEXT= keyword is used, then check for unknown keywords here */
                      SAY SourceFile'('LineCount'): Error: .HRULE parameter "'ThisParm'" not valid.';
                      rc = ERROR.INVALID_DATA;
                      LEAVE;
                   END
                END;
                /* any specified text attribute or color resets the default attrib */
                IF (FontEscapeChars \= '' | ForegroundColor \= '') THEN
                   fDefaultAttrib = FALSE;
                /* default text attribute is bold */
                IF (fDefaultAttrib) THEN
                DO
                   UpDefaultParm = 'BOLD';
                   wp = WORDPOS( UpDefaultParm, Char._Attrs);
                   FontEscapeChars = SUBSTR( Char._EscapeChars, wp, 1);
                END;

                /* build ThisLine */
                EndFontEscapeChars = REVERSE( FontEscapeChars);
                IF (HRuleText \= '') THEN
                DO
                   HRuleText = FontEscapeChars''HRuleText''EndFontEscapeChars;
                   HRuleText = MakeIpfLine( HRuleText, TRUE, TRUE);
                END;
                IF (fRule) THEN
                   ThisLine = COPIES( '_', 80)''CrLf'.br'CrLf''HRuleText;
                ELSE
                DO
                   IF (HRuleText \= '') THEN
                      ThisLine = ':p.'CrLf''HRuleText;
                END;
                IF (ForegroundColor \= '') THEN
                   ThisLine = ':color fc='ForegroundColor'.'CrLf''ThisLine''CrLf':color fc=default.';
             END;

             /* handle notes */
             WHEN (POS( Command, '.NOTE') = 1) THEN
             DO
                /* default note text */
                Note = VALUE( 'NOTE',, env);
                /* default list type */
                List = '.sl';
                /* process attributes */
                Parms    = CommandParms;
                ListAttr = '';
                DO WHILE (Parms \= '')
                   ThisVal = '';
                   PARSE VAR Parms ThisParm Rest;
                   UpThisParm = TRANSLATE( ThisParm);

                   IF (LEFT( UpThisParm, 5) = 'TEXT=') THEN
                   DO
                      Rest = SUBSTR( Parms, 6)
                      SELECT
                         WHEN (LEFT( Rest, 1) = "'") THEN
                            PARSE VAR Rest "'"ThisVal"'" Rest;
                         WHEN (LEFT( Rest, 1) = '"') THEN
                            PARSE VAR Rest '"'ThisVal'"' Rest;
                      OTHERWISE
                         PARSE VAR Rest ThisVal Rest;
                      END
                   END;

                   Parms = Rest;

                   SELECT
                      WHEN (POS( UpThisParm, 'ORDERED')   = 1) THEN List = '.ol';
                      WHEN (POS( UpThisParm, 'UNORDERED') = 1) THEN List = '.ul';
                      WHEN (POS( UpThisParm, 'SIMPLE')    = 1) THEN List = '.sl';
                      WHEN (LEFT( UpThisParm, 5) = 'TEXT=')    THEN Note = ThisVal;
                      OTHERWISE ListAttr = ListAttr ThisParm;
                   END;
                END;
                List = STRIP( List ListAttr);

                /* write note text */
                ThisLine = MakeIpfLine( Note, TRUE, TRUE);

                /* this is processed before the next line is read */
                NextLines = List;
             END;

             /* handle lists */
             WHEN ((POS( Command, '.ULIST') = 1) | ,
                   (POS( Command, '.OLIST') = 1) | ,
                   (POS( Command, '.SLIST') = 1) | ,
                   (POS( Command, '.PLIST') = 1) | ,
                   (POS( Command, '.TABLE') = 1) | ,
                   (POS( Command, '.LINES') = 1)) THEN
             DO
                x             = List.0 + 1;
                List.x._Type  = TRANSLATE( SUBSTR( Command, 2, 1), 'uosptl', 'UOSPTL');
                List.0        = x

                /* save current pos as first for this list */
                IF (x = 1) THEN
                   List.x._Pos = 1
                ELSE
                   List.x._Pos = '';

                SELECT
                   WHEN (List.x._Type = 't') THEN
                   DO
                      /* convert double quotes to single quotes (for cols='...' parm) */
                      CommandParms = TRANSLATE( CommandParms, "'", '"')
                      List.x._TAttr = GetTextAttr( CommandParms);
                      List.x._Attr  = DiscardTextAttrs(CommandParms);
                      List.x._Start = ':'STRIP( 'table' List.x._Attr)'.';
                      List.x._End   = ':etable.';
                   END;

                   WHEN (List.x._Type = 'l') THEN
                   DO
                      /* default alignment */
                      Align = 'left';
                      /* process attributes */
                      Parms = CommandParms;
                      DO WHILE (Parms \= '')
                         PARSE VAR Parms ThisParm Parms;
                         UpThisParm = TRANSLATE( ThisParm);
                         SELECT
                            WHEN (POS( UpThisParm, 'LEFT')   = 1) THEN Align = 'left';
                            WHEN (POS( UpThisParm, 'RIGHT')  = 1) THEN Align = 'right';
                            WHEN (POS( UpThisParm, 'CENTER') = 1) THEN Align = 'center';
                            OTHERWISE NOP;
                         END;
                      END;
                      List.x._Attr  = 'align='Align;
                      List.x._Start = ':'STRIP( 'lines' List.x._Attr)'.';
                      List.x._End   = ':elines.';
                   END;

                   WHEN (List.x._Type = 'p') THEN
                   DO
                      List.x._TAttr = GetTextAttr( CommandParms);
                      List.x._Attr  = DiscardTextAttrs( CommandParms);
                      List.x._Start = ':'STRIP( 'parml' List.x._Attr)'.';
                      List.x._End   = ':eparml.';
                   END;

                   OTHERWISE
                   DO
                      List.x._Attr  = CommandParms;
                      List.x._Start = ':'STRIP( List.x._Type'l' List.x._Attr)'.';
                      List.x._End   = ':e'List.x._Type'l.';
                   END;
                END;

                /* set pending newpar command, processed for an .ELIST command */
                IF (WORDPOS( 'COMPACT', TRANSLATE( CommandParms)) > 0) THEN
                   List.x._PendingNewPar = '.br'
                ELSE
                   List.x._PendingNewPar = ':p.'

                ThisLine      = List.x._Start;
             END;

             /* handle lists */
             WHEN (POS( Command, '.ELIST') = 1) THEN
             DO
                IF (List.0 = 0) THEN
                DO
                   SAY SourceFile'('LineCount'): Error: list not open.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   /* close ipf list, add an empty line when closing the last list level */
                   ThisLine = List.x._End;

                   /* use PendingNewPar from level above and set global var */
                   ListPendingNewPar = List.x._PendingNewPar

                   /* delete last level or delete list entry */
                   IF (WORDS( List.x._Pos) > 1) THEN
                      List.x._Pos = DELWORD( List.x._Pos, WORDS( List.x._Pos));
                   ELSE
                      List.0 = List.0 - 1;
                END;
             END;

             /* handle include files */
             WHEN ((POS( Command, '.INCLUDE') = 1)     |,
                   (POS( Command, '.CINCLUDE') = 1)    |,
                   (POS( Command, '.TEXTINCLUDE') = 1)) THEN
             DO
                SELECT
                   WHEN (POS( Command, '.TEXTINCLUDE') = 1)     THEN IncType = 'T';
                   WHEN (POS( Command, '.CINCLUDE') = 1)        THEN IncType = 'C';
                   OTHERWISE                                         IncType = '';
                END;

                /* handle spaces in filename either in single or double quotes, */
                /* parse optional StartLine and EndLine params                  */
                SELECT
                   WHEN LEFT( CommandParms, 1) = "'" THEN
                      PARSE VAR CommandParms "'"CommandParms"'" NextStartLine NextEndLine;
                   WHEN LEFT( CommandParms, 1) = '"' THEN
                      PARSE VAR CommandParms '"'CommandParms'"' NextStartLine NextEndLine;
                   OTHERWISE
                      PARSE VAR CommandParms CommandParms NextStartLine NextEndLine;
                END;

                IncludeFile = SearchIncludeFile( CommandParms);
                IF (IncludeFile = '') THEN
                DO
                   SAY SourceFile'('LineCount'): Error: include file' CommandParms 'not found.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   rcInclude = ProcessSourceFile( IncludeFile, TargetFile,,
                                                  IncludeLevel, IncType,,
                                                  NextStartLine, NextEndLine);
                   IF (rcInclude > 0) THEN
                      rc = rcInclude;
                   ITERATE;
                END;
             END;

             /* handle list of links */
             WHEN (Command = '.LOL') THEN
             DO
                /* use special pseudo link to include list of links here */
                ThisLine = '[lOl:]';
             END;

             OTHERWISE
                /* not a command, so process as standard text line */
                ThisLine = ProcessStandardLine( ThisLine);
          END;

       END;  /* IF (LEFT( ThisLine, 1) = '.') */
       ELSE

       /* process text mode lines without a leading dot here */
       IF (Pages._fTextMode) THEN
          /* MakeIpfLine is required here to mask the line */
          ThisLine = MakeIpfLine( ThisLine, FALSE, TRUE);
       ELSE

       /* skip section with wrong conditions */
       IF ((If._fIfOpen) & (\If._fIncludeSource)) THEN
          ITERATE;
       ELSE

       /* - - - - - - - - - - - - - */
       /* process empty line */
       IF (ThisLine = '') THEN
       DO
          /* use empty lines to end all open lists or to separate paragraphs */
          IF (List.0 > 0) THEN
          DO
             /* close open lists */
             p = Pages.0;
             DO x = List.0 TO 1 BY -1
                DO i = 1 TO WORDS( List.x._Pos)

                   IF (LENGTH( Pages.p._Contents) > 0) THEN
                      Pages.p._Contents = Pages.p._Contents''CrLf;
                   Pages.p._Contents = Pages.p._Contents''List.x._End;
                END;
             END;
             DROP( List.);
             List.0 = 0;
             /* don't iterate here to pass the pending newpar check at the end */
          END;
          ELSE
             NOP;
             /* multiple empty lines are ignored, one prepends a :p. to the next */
             /* horizontal material */
             /* "ThisLine = ''" will not produce an output line */

       END;
       ELSE

          /* - - - - - - - - - - - - - */
          /* process standard text line or list item */
          ThisLine = ProcessStandardLine( ThisLine);

       /* - - - - - - - - - - - - - */
       /* add line */
       p = Pages.0;

       IF \(Pages._fTextMode | Pages._fMonospaceMode) THEN
       DO
          /* maybe add a parbreak before the current line */
          ThisNewPar = GetThisNewPar( Command CommandParms);

          IF (ThisNewPar \= '') THEN
          DO
             IF (LENGTH( Pages.p._Contents) > 0) THEN
                Pages.p._Contents = Pages.p._Contents''CrLf;
             Pages.p._Contents = Pages.p._Contents''ThisNewPar;
          END;
       END;

       IF (LENGTH( ThisLine) > 0 | Pages._fTextMode | Pages._fMonospaceMode) THEN
       DO
          IF (LENGTH( Pages.p._Contents) > 0) THEN
             Pages.p._Contents = Pages.p._Contents''CrLf;
          Pages.p._Contents = Pages.p._Contents''ThisLine;
       END;

    END;  /* DO UNTIL (NextLines = '') */

 END;

 /* close source file, otherwise we might run out of handles */
 rcx = STREAM( SourceFile, 'C', 'CLOSE');

 /* continue with vars of previous source file */
 SourceFile = PrevSourceFile;
 LineCount  = PrevLineCount;
 StartLine  = PrevStartLine;
 EndLine    = PrevEndLine;

 RETURN( rc);

/* ========================================================================= */
ShowIPFCOutput: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG OutputFile;

 fShowOutput = FALSE;

 DO WHILE (LINES( OutputFile) > 0)
    ThisLine = LINEIN( OutputFile);

    SELECT
       WHEN (ThisLine = '')                          THEN NOP;
       WHEN (LEFT( ThisLine, 1) = '.')               THEN fShowOutput = TRUE;
       WHEN (POS( 'Statistics :', ThisLine) = 1)     THEN fShowOutput = FALSE;
       WHEN (POS( 'Document title :', ThisLine) = 1) THEN
       DO
          SAY;
          SAY ThisLine;
       END;
       WHEN (POS( '(INF)', ThisLine) > 0)            THEN SAY ThisLine;

       WHEN (LEFT( ThisLine, 1) = '<') THEN
       DO
          PARSE VAR ThisLine '<'ThisFile':'ThisLine'>'ThisInfo;
          SAY ThisFile'('ThisLine'):'ThisInfo;
       END;

       WHEN (fShowOutput)                            THEN SAY ThisLine;
       OTHERWISE NOP;
    END;

 END;
 rcx = STREAM( OutputFile, 'C', 'CLOSE');

 RETURN( ERROR.NO_ERROR);

