/*----------------------------------------------------------------------*
 *	Bounds Checking for GCC.					*
 *	Written by Richard W.M. Jones.					*
 *----------------------------------------------------------------------*
 * File:
 *	lib/check.h
 * Summary:
 *	Functions are provided here to guess whether an object is
 *	unchecked or not.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		13/7/95		Initial implementation.
 *----------------------------------------------------------------------*/

#ifndef _CHECK_H_
#define _CHECK_H_

/* Define a test for pointers to dynamic link libraries here.
 */
#if defined(linux) && !defined(__ELF__)
#define POINTER_IN_DLL(p) ((p) >= (void *) 0x60000000 \
                           && (p) < (void *) 0x80000000)
#elif defined(linux) && defined(__ELF__)
#define POINTER_IN_DLL(p) ((p) >= (void *) 0x50000000 \
                           && (p) < (void *) 0x80000000)
#elif defined(_AIX)
#define POINTER_IN_DLL(p) ((p) >= (void *) 0x20000000 \
			   && (p) < (void *) 0x40000000)
#else
#define POINTER_IN_DLL(p) 0
#endif

/* Define a way to test if a pointer is on the stack.
 *	`DECLARE_SP' is a macro that declares the stack pointer (if necessary).
 *	`POINTER_ON_STACK' returns true if the pointer is between SP and
 *	the top of the stack (or vv. if the stack grows up not down).
 */
#if defined(linux)
/* #define DECLARE_SP register void *sp asm ("sp") */
#define DECLARE_SP int i; register void *sp = &i;
#define POINTER_ON_STACK(p) ((void *) sp <= (p) && (p) < (void *)0xC0000000)
#else
/* This is the generic method that will work on most 32-bit and 64-bit
 * architectures. (Note: 64-bit support is *untested*, since we have no
 * Alphas around here).
 */
#if defined(sun)
#define DECLARE_SP
#define POINTER_ON_STACK(p) ((void *)0xe0000000 <= (p) && (p) <= (void *)-1)
#else
#if defined(__hppa)
#define DECLARE_SP int i; register void *sp = &i;
#define POINTER_ON_STACK(p) ((void *)sp >= (p) && (p) >= (void *)0x7a000000)
#else
#define DECLARE_SP int i; register void *sp = &i;
#define POINTER_ON_STACK(p) ((void *)sp <= (p) && (p) <= (void *)-1)
#endif
#endif
#endif

/* We have looked up a pointer in the tree and not found it. Give it
 * a second chance here, where we may determine that it is an unchecked
 * static object. We give a warning if it is.
 */
static inline int
maybe_is_unchecked_static (void *pointer, char *filename, int line, char *function)
{
  if (pointer < (void *) &end || POINTER_IN_DLL (pointer))
    {
#if COLLECT_STATS
      ++__bounds_stats_unchecked_static;
#endif
      if (__bounds_warn_unchecked_statics)
	__bounds_warning (filename, line, function,
			  "unchecked static object used at address %p",
			  pointer);
      return 1;
    }
  return 0;
}

/* Similarly for unchecked stack objects, we can test if the pointer is
 * between SP and the top of the stack.
 */
static inline int
maybe_is_unchecked_stack (void *pointer, char *filename, int line, char *function)
{
  DECLARE_SP;

  if (POINTER_ON_STACK (pointer))
    {
#if COLLECT_STATS
      ++__bounds_stats_unchecked_stack;
#endif
      if (__bounds_warn_unchecked_stack)
	__bounds_warning (filename, line, function,
			  "unchecked stack object used at address %p",
			  pointer);
      return 1;
    }
  return 0;
}

#endif /* _CHECK_H_ */
