////////////////////////////////////////////////////////////////
// File - P9054_DIAG.C
//
// o A simple diagnostics program that lets you access the
//   PLX 9054 registers and local memory. 
// o This program is meant to be used as an example for using the P9054_LIB.H API,
//   you may use it as a skeleton for your driver, or 'cut & paste' parts
//   of it into your device driver code.
// o For a more advanced monitor program, use the standard PLXMON.EXE
//   from PLX.
// 
////////////////////////////////////////////////////////////////

#include "../lib/p9054_lib.h"
#include "../../../samples/shared/pci_diag_lib.h"
#include <stdio.h>

// input of command from user
static char line[256];

void PLX_EditReg(P9054_HANDLE hPlx)
{
    struct 
    {
        CHAR *name;
        DWORD dwOffset;
        DWORD dwVal;
    } fields[30];

    int cmd;
    int i;
    int field_count;

    i = 0;
    fields[i].name = "LAS0RR"; fields[i++].dwOffset = 0x00;
    fields[i].name = "LAS0BA"; fields[i++].dwOffset = 0x04;
    fields[i].name = "EROMRR"; fields[i++].dwOffset = 0x10;
    fields[i].name = "EROMBA"; fields[i++].dwOffset = 0x14;
    fields[i].name = "LBRD0"; fields[i++].dwOffset = 0x18;
    fields[i].name = "MBOX0"; fields[i++].dwOffset = 0x40;
    fields[i].name = "MBOX1"; fields[i++].dwOffset = 0x44;
    fields[i].name = "MBOX2"; fields[i++].dwOffset = 0x48;
    fields[i].name = "MBOX3"; fields[i++].dwOffset = 0x4c;
    fields[i].name = "MBOX4"; fields[i++].dwOffset = 0x50;
    fields[i].name = "MBOX5"; fields[i++].dwOffset = 0x54;
    fields[i].name = "MBOX6"; fields[i++].dwOffset = 0x58;
    fields[i].name = "MBOX7"; fields[i++].dwOffset = 0x5c;
    fields[i].name = "P2LDBELL"; fields[i++].dwOffset = 0x60;
    fields[i].name = "L2PDBELL"; fields[i++].dwOffset = 0x64;
    fields[i].name = "INTCSR"; fields[i++].dwOffset = 0x68;
    fields[i].name = "CNTRL"; fields[i++].dwOffset = 0x6c;
    fields[i].name = "SECID"; fields[i++].dwOffset = 0x70;
    fields[i].name = "REVISON"; fields[i++].dwOffset = 0x74;
    field_count = i;
    do
    {
        int row;
        int col;
        int row_count = field_count/2 + field_count%2;

        printf ("\n");
        printf ("Edit PLX 9054 registers\n");
        printf ("--------------------------------\n");
        for (row = 0; row<row_count; row++)
        {
            for (col = 0; col<=1; col++)
            {
                if (col==0) i = row;
                else i = row + row_count;

                if (i<field_count)
                {
                    char buf[10];
                    fields[i].dwVal = P9054_ReadReg(hPlx, fields[i].dwOffset);
                    sprintf(buf, "%08x",fields[i].dwVal);
                    printf ("%2d. %8s : %s     ",i+1, fields[i].name, buf);
                }
                if (col==1) printf ("\n");
            }
        }

        printf ("99. Back to main menu\n");
        printf ("Choose register to write to, or 99 to exit: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        if (cmd>=1 && cmd <=21)
        {
            i = cmd-1;
            printf ("Enter value to write to %s register (or 'X' to cancel): ",fields[i].name);
            gets (line);
            if (toupper (line[0])!='X')
            {
                DWORD dwVal;
                dwVal = 0;
                sscanf (line,"%x",&dwVal);
                P9054_WriteReg(hPlx, fields[i].dwOffset, dwVal);
            }
        }
    } while (cmd!=99);
}

char *PLX_GetAddrRangeName(P9054_ADDR addrSpace)
{
    return 
        addrSpace==P9054_ADDR_SPACE0 ? "Addr Space 0 - (BAR2)" :
        addrSpace==P9054_ADDR_SPACE1 ? "Addr Space 1 - (BAR3)" :
        addrSpace==P9054_ADDR_SPACE2 ? "Addr Space 2 - (BAR4)" :
        addrSpace==P9054_ADDR_SPACE3 ? "Addr Space 3 - (BAR5)" :
        addrSpace==P9054_ADDR_EPROM ? "EEPROM Addr Space" : "Invalid";
}

void PLX_BoardAccess(P9054_HANDLE hPlx, BOOL fLocalAddr)
{
    int cmd, cmd2;
    DWORD addr, data, i;
    P9054_ADDR ad_sp = P9054_ADDR_SPACE0;
    P9054_MODE ad_mode = P9054_MODE_DWORD;

    char *pcMemoryType = fLocalAddr ? "local address" : "offset";

    if (!fLocalAddr)
    {
        for (; ad_sp<=P9054_ADDR_EPROM && !P9054_IsAddrSpaceActive(hPlx, ad_sp); ad_sp++)
        if (ad_sp>P9054_ADDR_EPROM)
        {
            printf ("No active memory spaces on board!\n");
            return;
        }
    }

    do
    {
        printf ("Access the board's %s ranges\n",pcMemoryType);
        printf ("-------------------------------------------\n");
        printf ("(Access to invalid %s may hang the computer!)\n", pcMemoryType);
        if (!fLocalAddr)
            printf ("1. Change active memory space: %s\n",PLX_GetAddrRangeName(ad_sp));
        printf ("2. Toggle active mode: %s\n", 
            ad_mode==P9054_MODE_BYTE ? "BYTE (8 bit)" :
            ad_mode==P9054_MODE_WORD ? "WORD (16 bit)" : "DWORD (32 bit)");
        printf ("3. Read from board\n");
        printf ("4. Write to board\n");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            if (!fLocalAddr)
            {
                printf ("Choose memory space:\n");
                printf ("--------------------\n");
                for (i=P9054_ADDR_SPACE0; i<=P9054_ADDR_EPROM; i++)
                {
                    printf ("%d. %s", i, PLX_GetAddrRangeName(i));
                    if (P9054_IsAddrSpaceActive(hPlx, i)) printf ("\n");
                    else printf (" - space not active\n");
                }
                printf ("Enter option: ");
                cmd2 = 99;
                gets (line);
                sscanf (line, "%d",&cmd2);
                if (cmd2>=P9054_ADDR_SPACE0 && cmd2<=P9054_ADDR_EPROM)
                {
                    int new_ad_sp = cmd2;
                    if (P9054_IsAddrSpaceActive(hPlx, new_ad_sp)) ad_sp = new_ad_sp;
                    else printf ("Chosen space not active!\n");
                }
            }
            break;
        case 2:
            ad_mode = (ad_mode + 1) % 3;
            break;
        case 3:
            printf ("Enter %s to read from: ", pcMemoryType);
            gets (line);
            sscanf (line, "%x", &addr);
            switch (ad_mode)
            {
            case P9054_MODE_BYTE:
                if (fLocalAddr) data = P9054_ReadByteLocal(hPlx, addr);
                else data = P9054_ReadByte(hPlx, ad_sp, addr);
                break;
            case P9054_MODE_WORD:
                if (fLocalAddr) data = P9054_ReadWordLocal(hPlx, addr);
                else data = P9054_ReadWord(hPlx, ad_sp, addr);
                break;
            case P9054_MODE_DWORD:
                if (fLocalAddr) data = P9054_ReadDWordLocal(hPlx, addr);
                else data = P9054_ReadDWord(hPlx, ad_sp, addr);
                break;
            }
            printf ("Value read: %x\n", data);
            break;
        case 4:
            printf ("Enter %s to write to: ", pcMemoryType);
            gets (line);
            sscanf (line, "%x", &addr);
            printf ("Enter data to write %s: ",
                ad_mode==P9054_MODE_BYTE ? "BYTE (8 bit)" :
                ad_mode==P9054_MODE_WORD ? "WORD (16 bit)" : "DWORD (32 bit)");
            gets (line);
            sscanf (line, "%x",&data);
            switch (ad_mode)
            {
            case P9054_MODE_BYTE:
                if (fLocalAddr) P9054_WriteByteLocal(hPlx, addr, (BYTE) data);
                else P9054_WriteByte(hPlx, ad_sp, addr, (BYTE) data);
                break;
            case P9054_MODE_WORD:
                if (fLocalAddr) P9054_WriteWordLocal(hPlx, addr, (WORD) data);
                else P9054_WriteWord(hPlx, ad_sp, addr, (WORD) data);
                break;
            case P9054_MODE_DWORD:
                if (fLocalAddr) P9054_WriteDWordLocal(hPlx, addr, data);
                else P9054_WriteDWord(hPlx, ad_sp, addr, data);
                break;
            }
            break;
        }
    } while (cmd!=99);
}

void WINAPI PLX_IntHandlerRoutine(P9054_HANDLE hPlx, P9054_INT_RESULT *intResult)
{
    printf ("Got interrupt number %d\n", intResult->dwCounter);
}

void PLX_EnableDisableInterrupts(P9054_HANDLE hPlx)
{
    int cmd;

    printf ("WARNING!!!\n");
    printf ("----------\n");
    printf ("Your hardware has level sensitive interrupts.\n");
    printf ("You must modify the source code of P9054_IntEnable(), in the file P9054_lib.c,\n");
    printf ("to acknowledge the interrupt before enabling interrupts.\n");
    printf ("Without this modification, your PC will HANG upon interrupt!\n");

    do
    {
        printf ("Enable / Disable interrupts\n");
        printf ("---------------------------\n");
        printf ("1. %s interrupts\n", P9054_IntIsEnabled(hPlx) ? "Disable" : "Enable");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            if (P9054_IntIsEnabled(hPlx))
            {
                printf ("Disabling interrupt Int\n");
                P9054_IntDisable(hPlx);
            }
            else
            {
                printf ("Enabling interrupts\n");
                if (!P9054_IntEnable(hPlx, PLX_IntHandlerRoutine))
                    printf ("failed enabling interrupts\n");
            }
            break;
        }
    } while (cmd!=99);
}

void PLX_EEPROMAccess(P9054_HANDLE hPlx)
{
    int cmd;
    DWORD addr;
    DWORD dwData;

    do
    {
        printf ("Access the board's serial EERPOM\n");
        printf ("--------------------------------\n");
        if (!P9054_EEPROMValid(hPlx))
            printf ("Note: PLX EEPROM valid BIT is 0\n");
        printf (" Note: reading/writing to odd addresses is meaningless,\n");
        printf ("       as data is stored in words.\n");
        printf ("1. Display EEPROM content\n");
        printf ("2. Read dword from serial EEPROM on the board\n");
        printf ("3. Write dword to the serial EEPROM on the board\n");
        printf ("99. Back to main menu\n");
        printf ("\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1:
            for (addr=0; addr<0xff; addr += 4)
            {
                if (!(addr % 0x10))
                printf("\n %02x: ", addr);
                if (!P9054_EEPROMReadDWord(hPlx, addr, &dwData))
                {
                    printf("\nError occured reading serial EEPROM - %s\n", P9054_ErrorString);
                    break;
                }
                printf("%08x  ", dwData);
            }
            printf ("\n");
            break;
        case 2:
            printf ("Enter addr to read from (0-ff): ");
            gets (line);
            sscanf (line, "%x", &addr);
            if (P9054_EEPROMReadDWord(hPlx, addr, &dwData))
                printf ("Value read: %x\n", dwData);
             else
                printf("Error occured reading serial EEPROM - %s\n", P9054_ErrorString);
            break;

        case 3:
            printf ("Enter addr to write to (0-ff): ");
            gets (line);
            sscanf (line, "%x", &addr);
            printf ("Enter data to write: ");
            gets (line);
            sscanf (line, "%x",&dwData);
            if (!P9054_EEPROMWriteDWord(hPlx, addr, dwData))
                printf("Error occured reading serial EEPROM - %s\n", P9054_ErrorString);
            break;

        default:
            break;
        }
    } while (cmd!=99);
}

P9054_HANDLE PLX_LocateAndOpenBoard(DWORD dwVendorID, DWORD dwDeviceID, BOOL fUseInt)
{
    DWORD cards, my_card;
    P9054_HANDLE hPlx = NULL;

    if (dwVendorID==0)
    {
        printf ("Enter VendorID: ");
        gets (line);
        sscanf (line, "%x",&dwVendorID);
        if (dwVendorID==0) return NULL;

        printf ("Enter DeviceID: ");
        gets (line);
        sscanf (line, "%x",&dwDeviceID);
    }
    cards = P9054_CountCards (dwVendorID, dwDeviceID);
    if (cards==0) 
    {
        printf("%s", P9054_ErrorString);
        return NULL;
    }
    else if (cards==1) my_card = 1;
    else
    {
        DWORD i;

        printf("Found %d matching PCI cards\n", cards);
        printf("Select card (1-%d): ", cards);
        i = 0;
        gets (line);
        sscanf (line, "%d",&i);
        if (i>=1 && i <=cards) my_card = i;
        else 
        {
            printf ("Choice out of range\n");
            return NULL;
        }
    }
    if (P9054_Open (&hPlx, dwVendorID, dwDeviceID, my_card - 1, fUseInt ? P9054_OPEN_USE_INT : 0))
        printf ("PLX 9054 PCI card found!\n");
    else printf ("%s", P9054_ErrorString);
    return hPlx;
}

int main(int argc, char *argv[])
{
    int cmd;
    P9054_HANDLE hPlx = NULL;
    HANDLE hWD;
    BOOL fUseInt = FALSE; // by default - do not install interrupts
    BOOL fOpenedWithInt = fUseInt;

    printf ("PLX 9054 diagnostic utility.\n");
    printf ("Application accesses hardware using " WD_PROD_NAME ".\n");

    // make sure WinDriver is loaded
    if (!PCI_Get_WD_handle(&hWD)) return 0;
    WD_Close (hWD);

    hPlx = PLX_LocateAndOpenBoard(0x10b5, 0x1860, fUseInt);

    do
    {
        printf ("\n");
        printf ("PLX 9054 main menu\n");
        printf ("-------------------\n");
        printf ("1. Scan PCI bus\n");
        printf ("2. Set opening board %s interrupts\n", fUseInt ? "without" : "with");
        printf ("3. Locate/Choose PLX 9054 board (%s interrupts)\n", fUseInt ? "with" : "without");
        if (hPlx)
        {
            printf ("4. PCI configuration registers\n");
            printf ("5. PLX 9054 local registers\n");
            printf ("6. Access address spaces on the board\n");
            printf ("7. Access local address ranges on the board\n");
            if (fOpenedWithInt)
                printf ("8. Enable / Disable interrupts\n");
            printf ("9. Access serial EEPROM on the board\n");
        }
        printf ("99. Exit\n");
        printf ("Enter option: ");
        cmd = 0;
        gets (line);
        sscanf (line, "%d",&cmd);
        switch (cmd)
        {
        case 1: // Scan PCI bus
            PCI_Print_all_cards_info();
            break;
        case 2: // Set open board with / without interrupts
            fUseInt = !fUseInt;
            break;
        case 3: // Locate PLX 9054 board
            if (hPlx) P9054_Close(hPlx);
            hPlx = PLX_LocateAndOpenBoard(0, 0, fUseInt);
            if (!hPlx) printf ("PLX card open failed!\n");
            fOpenedWithInt = fUseInt;
            break;
        case 4: // PCI configuration registers
            if (hPlx) 
            {
                WD_PCI_SLOT pciSlot;
                P9054_GetPciSlot(hPlx, &pciSlot);
                PCI_EditConfigReg(pciSlot);
            }
            break;
        case 5: // PLX 9054 local registers
            if (hPlx) PLX_EditReg(hPlx);
            break;
        case 6: // Access address spaces on the board
            if (hPlx) PLX_BoardAccess(hPlx, FALSE);
            break;
        case 7: // Access local address ranges on the board
            if (hPlx) PLX_BoardAccess(hPlx, TRUE);
            break;
        case 8: // Enable / Disable interrupts
            if (hPlx && fOpenedWithInt) PLX_EnableDisableInterrupts(hPlx);
            break;
        case 9: // Access serial EEPROM on the board
            if (hPlx) PLX_EEPROMAccess(hPlx);
            break;
        }
    } while (cmd!=99);

    if (hPlx) P9054_Close(hPlx);

    return 0;
}

                                      
