////////////////////////////////////////////////////////////////
// File - P9080_LIB.C
//
// Library for 'WinDriver for PLX 9080' API. 
// The basic idea is to get a handle for the board
// with P9080_Open() and use it in the rest of the program
// when calling WD functions.  Call P9080_Close() when done.
// 
////////////////////////////////////////////////////////////////

#include "p9080_lib.h"
#include "../../../include/windrvr_int_thread.h"
#include <stdio.h>

// this string is set to an error message, if one occurs
CHAR P9080_ErrorString[1024];

// internal data structures and enums
enum { P9080_DMA_CHANNEL_SHIFT = 0x14 }; // shift in address between channels 0 and 1 of DMA

typedef struct P9080_DMA_STRUCT {
    WD_DMA dma;
    WD_DMA dmaList;
    P9080_DMA_CHANNEL dmaChannel;
} P9080_DMA_STRUCT;

enum { P9080_MODE_DESC       = 0xF9000140 };
enum { P9080_MODE_DESC_BYTE  = 0x00000000 };
enum { P9080_MODE_DESC_WORD  = 0x00010001 };
enum { P9080_MODE_DESC_DWORD = 0x00030003 };

typedef struct
{
    WD_INTERRUPT Int;
    HANDLE hThread;
    WD_TRANSFER Trans[2];
    P9080_INT_HANDLER funcIntHandler;
} P9080_INTERRUPT;

typedef struct 
{
    DWORD dwLocalBase;
    DWORD dwMask;
    DWORD dwBytes;
    DWORD dwAddr;
    DWORD dwAddrDirect;
    BOOL  fIsMemory;
} P9080_ADDR_DESC;

typedef struct P9080_STRUCT
{
    HANDLE hWD;
    WD_CARD cardLock;
    WD_PCI_SLOT pciSlot;
    WD_CARD_REGISTER cardReg;
    P9080_ADDR_DESC addrDesc[AD_PCI_BARS];
    DWORD  addrSpace;
    BOOL   fUseInt;
    P9080_INTERRUPT Int;
    BOOL   fUseCS46EEPROM;
} P9080_STRUCT;

// internal function used by P9080_Open()
BOOL P9080_DetectCardElements(P9080_HANDLE hPlx);
// internal function used by P9080_Read... and P9080_Write... functions
void P9080_SetMode (P9080_HANDLE hPlx, P9080_MODE mode, DWORD dwLocalAddr);

DWORD P9080_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    P9080_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( P9080_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( P9080_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( P9080_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}


BOOL P9080_Open (P9080_HANDLE *phPlx, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD dwOptions)
{
    P9080_HANDLE hPlx = (P9080_HANDLE) malloc (sizeof (P9080_STRUCT));
    DWORD dwIntStatus;

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;

    *phPlx = NULL;
    P9080_ErrorString[0] = '\0';
    BZERO(*hPlx);

    hPlx->cardReg.hCard = 0;
    hPlx->hWD = WD_Open();

    // check if handle valid & version OK
    if (hPlx->hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( P9080_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hPlx->hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( P9080_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hPlx->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( P9080_ErrorString, "Could not find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( P9080_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    WD_PciGetCardInfo (hPlx->hWD, &pciCardInfo);
    hPlx->pciSlot = pciCardInfo.pciSlot;
    hPlx->cardReg.Card = pciCardInfo.Card;

    hPlx->fUseInt = (dwOptions & P9080_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hPlx->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hPlx->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hPlx->hWD, &hPlx->cardReg);
    if (hPlx->cardReg.hCard==0)
    {
        sprintf ( P9080_ErrorString, "Failed locking device\n");
        goto Exit;
    }

    if (!P9080_DetectCardElements(hPlx))
    {
        sprintf ( P9080_ErrorString, "Card does not have all items expected for PLX 9080\n");
        goto Exit;
    }

    // this enables target abort so it wont get stuck 
    dwIntStatus = P9080_ReadReg (hPlx, P9080_INTCSR);
    P9080_WriteReg (hPlx, P9080_INTCSR, dwIntStatus | BIT12);

    // check for EEPROM type
    if( dwOptions & P9080_CS46_EEPROM )
    {
        hPlx->fUseCS46EEPROM = TRUE;
    }

    // Open finished OK
    *phPlx = hPlx;
    return TRUE;

Exit:
    // Error durin Open
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);
    if (hPlx->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hPlx->hWD);
    free (hPlx);
    return FALSE;
}

void P9080_GetPciSlot(P9080_HANDLE hPlx, WD_PCI_SLOT *pPciSlot)
{
    *pPciSlot = hPlx->pciSlot;
}


DWORD P9080_ReadPCIReg(P9080_HANDLE hPlx, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
    return dwVal;
}

void P9080_WritePCIReg(P9080_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO (pciCnf);
    pciCnf.pciSlot = hPlx->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hPlx->hWD,&pciCnf);
}

BOOL P9080_DetectCardElements(P9080_HANDLE hPlx)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hPlx->Int);
    BZERO(hPlx->addrDesc);

    for (i=0; i<hPlx->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hPlx->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwAddr;
                DWORD dwPhysAddr;
                DWORD dwAddrDirect = 0;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwAddr = pItem->I.Mem.dwTransAddr;
                    dwAddrDirect = pItem->I.Mem.dwUserDirectAddr;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwAddr = pItem->I.IO.dwAddr;
                    dwPhysAddr = dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=P9080_ADDR_REG; ad_sp<=P9080_ADDR_EPROM; ad_sp++)
                {
                    DWORD dwPCIAddr;
                    DWORD dwPCIReg;

                    if (hPlx->addrDesc[ad_sp].dwAddr) continue;
                    if (ad_sp==P9080_ADDR_REG) dwPCIReg = PCI_BAR0;
                    else if (ad_sp<P9080_ADDR_EPROM) 
                        dwPCIReg = PCI_BAR2 + 4*(ad_sp-P9080_ADDR_SPACE0);
                    else dwPCIReg = PCI_ERBAR;
                    dwPCIAddr = P9080_ReadPCIReg(hPlx, dwPCIReg);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<=P9080_ADDR_EPROM)
                {
                    DWORD j;
                    hPlx->addrDesc[ad_sp].dwBytes = dwBytes;
                    hPlx->addrDesc[ad_sp].dwAddr = dwAddr;
                    hPlx->addrDesc[ad_sp].dwAddrDirect = dwAddrDirect;
                    hPlx->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hPlx->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<hPlx->addrDesc[ad_sp].dwBytes && j!=0x80000000; j *= 2)
                    {
                        hPlx->addrDesc[ad_sp].dwMask = 
                            (hPlx->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hPlx->Int.Int.hInterrupt) return FALSE;
            hPlx->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hPlx->fUseInt && !hPlx->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that the registers space was found
    if (!P9080_IsAddrSpaceActive(hPlx, P9080_ADDR_REG)) 
            //|| hPlx->addrDesc[P9080_ADDR_REG].dwBytes!=P9080_RANGE_REG)
        return FALSE;

    // use address space 0 for accessing local addresses
    hPlx->addrSpace = P9080_ADDR_SPACE0;
    
    // check that address space 0 was found
    if (!P9080_IsAddrSpaceActive(hPlx, hPlx->addrSpace)) return FALSE;

    return TRUE;
}

void P9080_Close(P9080_HANDLE hPlx)
{
    // disable interrupts
    if (P9080_IntIsEnabled(hPlx))
        P9080_IntDisable(hPlx);

    // unregister card
    if (hPlx->cardReg.hCard) 
        WD_CardUnregister(hPlx->hWD, &hPlx->cardReg);

    // close WinDriver
    WD_Close(hPlx->hWD);

    free (hPlx);
}

BOOL P9080_IsAddrSpaceActive(P9080_HANDLE hPlx, P9080_ADDR addrSpace)
{
    return hPlx->addrDesc[addrSpace].dwAddr!=0;
}

DWORD P9080_ReadReg (P9080_HANDLE hPlx, DWORD dwReg)
{
    return P9080_ReadDWord(hPlx, P9080_ADDR_REG, dwReg);
}

void P9080_WriteReg (P9080_HANDLE hPlx, DWORD dwReg, DWORD dwData)
{
    P9080_WriteDWord(hPlx, P9080_ADDR_REG, dwReg, dwData);
}

BYTE P9080_ReadByte (P9080_HANDLE hPlx, P9080_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        return *pByte;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_BYTE;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Byte;
    }
}

void P9080_WriteByte (P9080_HANDLE hPlx, P9080_ADDR addrSpace, DWORD dwOffset, BYTE data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        BYTE *pByte = (BYTE *) dwAddr;
        *pByte = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_BYTE;
        trans.dwPort = dwAddr;
        trans.Data.Byte = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

WORD P9080_ReadWord (P9080_HANDLE hPlx, P9080_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        return *pWord;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_WORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Word;
    }
}

void P9080_WriteWord (P9080_HANDLE hPlx, P9080_ADDR addrSpace, DWORD dwOffset, WORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        WORD *pWord = (WORD *) dwAddr;
        *pWord = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_WORD;
        trans.dwPort = dwAddr;
        trans.Data.Word = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

DWORD P9080_ReadDWord (P9080_HANDLE hPlx, P9080_ADDR addrSpace, DWORD dwOffset)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        return *pDword;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = RP_DWORD;
        trans.dwPort = dwAddr;
        WD_Transfer (hPlx->hWD, &trans);
        return trans.Data.Dword;
    }
}

void P9080_WriteDWord (P9080_HANDLE hPlx, P9080_ADDR addrSpace, DWORD dwOffset, DWORD data)
{
    if (hPlx->addrDesc[addrSpace].fIsMemory)
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddrDirect + dwOffset;
        DWORD *pDword = (DWORD *) dwAddr;
        *pDword = data;
    }
    else
    {
        DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
        WD_TRANSFER trans;
        BZERO(trans);
        trans.cmdTrans = WP_DWORD;
        trans.dwPort = dwAddr;
        trans.Data.Dword = data;
        WD_Transfer (hPlx->hWD, &trans);
    }
}

void P9080_ReadWriteBlock (P9080_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, BOOL fIsRead, P9080_ADDR addrSpace, P9080_MODE mode)
{
    DWORD dwAddr = hPlx->addrDesc[addrSpace].dwAddr + dwOffset;
    WD_TRANSFER trans;

    BZERO(trans);

    if (hPlx->addrDesc[addrSpace].fIsMemory) 
    {
        if (fIsRead) 
        {
            if (mode==P9080_MODE_BYTE) trans.cmdTrans = RM_SBYTE;
            else if (mode==P9080_MODE_WORD) trans.cmdTrans = RM_SWORD;
            else trans.cmdTrans = RM_SDWORD;
        }
        else 
        {
            if (mode==P9080_MODE_BYTE) trans.cmdTrans = WM_SBYTE;
            else if (mode==P9080_MODE_WORD) trans.cmdTrans = WM_SWORD;
            else trans.cmdTrans = WM_SDWORD;
        }
    }
    else 
    {
        if (fIsRead) 
        {
            if (mode==P9080_MODE_BYTE) trans.cmdTrans = RP_SBYTE;
            else if (mode==P9080_MODE_WORD) trans.cmdTrans = RP_SWORD;
            else trans.cmdTrans = RP_SDWORD;
        }
        else 
        {
            if (mode==P9080_MODE_BYTE) trans.cmdTrans = WP_SBYTE;
            else if (mode==P9080_MODE_WORD) trans.cmdTrans = WP_SWORD;
            else trans.cmdTrans = WP_SDWORD;
        }
    }
    trans.dwPort = dwAddr;
    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hPlx->hWD, &trans);
}

void P9080_ReadBlock (P9080_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                    DWORD dwBytes, P9080_ADDR addrSpace, P9080_MODE mode)
{
    P9080_ReadWriteBlock (hPlx, dwOffset, buf, dwBytes, TRUE, addrSpace, mode);
}

void P9080_WriteBlock (P9080_HANDLE hPlx, DWORD dwOffset, PVOID buf, 
                     DWORD dwBytes, P9080_ADDR addrSpace, P9080_MODE mode)
{
    P9080_ReadWriteBlock (hPlx, dwOffset, buf, dwBytes, FALSE, addrSpace, mode);
}

void P9080_SetMode (P9080_HANDLE hPlx, P9080_MODE mode, DWORD dwLocalAddr)
{
    hPlx->addrDesc[hPlx->addrSpace].dwLocalBase = dwLocalAddr & ~hPlx->addrDesc[hPlx->addrSpace].dwMask;
    hPlx->addrDesc[hPlx->addrSpace].dwLocalBase |= BIT0;
    P9080_WriteReg (hPlx, P9080_LAS0BA, hPlx->addrDesc[hPlx->addrSpace].dwLocalBase);
}

BYTE P9080_ReadByteLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, P9080_MODE_BYTE, dwLocalAddr);
    return P9080_ReadByte(hPlx, hPlx->addrSpace, dwOffset);
}

void P9080_WriteByteLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr, BYTE data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, P9080_MODE_BYTE, dwLocalAddr);
    P9080_WriteByte(hPlx, hPlx->addrSpace, dwOffset, data);
}

WORD P9080_ReadWordLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, P9080_MODE_WORD, dwLocalAddr);
    return P9080_ReadWord(hPlx, hPlx->addrSpace, dwOffset);
}

void P9080_WriteWordLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr, WORD data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, P9080_MODE_WORD, dwLocalAddr);
    P9080_WriteWord(hPlx, hPlx->addrSpace, dwOffset, data);
}

DWORD P9080_ReadDWordLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, P9080_MODE_DWORD, dwLocalAddr);
    return P9080_ReadDWord(hPlx, hPlx->addrSpace, dwOffset);
}

void P9080_WriteDWordLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr, DWORD data)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, P9080_MODE_DWORD, dwLocalAddr);
    P9080_WriteDWord(hPlx, hPlx->addrSpace, dwOffset, data);
}

void P9080_ReadWriteBlockLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr, 
    PVOID buf, DWORD dwBytes, BOOL fIsRead, P9080_MODE mode)
{
    DWORD dwOffset = hPlx->addrDesc[hPlx->addrSpace].dwMask & dwLocalAddr;
    P9080_SetMode (hPlx, mode, dwLocalAddr);
    P9080_ReadWriteBlock(hPlx, dwOffset, buf, dwBytes, fIsRead, hPlx->addrSpace, mode);
}

void P9080_ReadBlockLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf,
    DWORD dwBytes, P9080_MODE mode)
{
    P9080_ReadWriteBlockLocal (hPlx, dwLocalAddr, buf, dwBytes, TRUE, mode);
}

void P9080_WriteBlockLocal (P9080_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
                     DWORD dwBytes, P9080_MODE mode)
{
    P9080_ReadWriteBlockLocal (hPlx, dwLocalAddr, buf, dwBytes, FALSE, mode);
}

BOOL P9080_IntIsEnabled (P9080_HANDLE hPlx)
{
    if (!hPlx->fUseInt) return FALSE;
    if (!hPlx->Int.hThread) return FALSE;
    return TRUE;
}

VOID P9080_IntHandler (PVOID pData)
{
    P9080_HANDLE hPlx = (P9080_HANDLE) pData;
    P9080_INT_RESULT intResult;
    intResult.dwCounter = hPlx->Int.Int.dwCounter;
    intResult.dwLost = hPlx->Int.Int.dwLost;
    intResult.fStopped = hPlx->Int.Int.fStopped;
    intResult.dwStatusReg = hPlx->Int.Trans[0].Data.Dword;
    hPlx->Int.funcIntHandler(hPlx, &intResult);  
}

BOOL P9080_IntEnable (P9080_HANDLE hPlx, P9080_INT_HANDLER funcIntHandler)
{
    DWORD dwIntStatus;
    DWORD dwAddr;

    if (!hPlx->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hPlx->Int.hThread) return FALSE;

    dwIntStatus = P9080_ReadReg (hPlx, P9080_INTCSR);

    BZERO(hPlx->Int.Trans);
    // This is a samlpe of handling interrupts:
    // Two transfer commands are issued. First the value of the interrrupt control/status
    // register is read. Then, a value of ZERO is written.
    // This will cancel interrupts after the first interrupt occurs.
    // When using interrupts, this section will have to change:
    // you must put transfer commands to CANCEL the source of the interrupt, otherwise, the 
    // PC will hang when an interrupt occurs!
    dwAddr = hPlx->addrDesc[P9080_ADDR_REG].dwAddr + P9080_INTCSR;
    hPlx->Int.Trans[0].cmdTrans = hPlx->addrDesc[P9080_ADDR_REG].fIsMemory ? RM_DWORD : RP_DWORD;
    hPlx->Int.Trans[0].dwPort = dwAddr;
    hPlx->Int.Trans[1].cmdTrans = hPlx->addrDesc[P9080_ADDR_REG].fIsMemory ? WM_DWORD : WP_DWORD;
    hPlx->Int.Trans[1].dwPort = dwAddr;
    hPlx->Int.Trans[1].Data.Dword = dwIntStatus & ~(BIT8|BIT10); // put here the data to write to the control register
    hPlx->Int.Int.dwCmds = 2; 
    hPlx->Int.Int.Cmd = hPlx->Int.Trans;
    hPlx->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;
 
    // this calls WD_IntEnable() and creates an interrupt handler thread
    hPlx->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hPlx->Int.hThread, hPlx->hWD, &hPlx->Int.Int, P9080_IntHandler, (PVOID) hPlx))
        return FALSE;

    // this physically enables interrupts
    P9080_WriteReg (hPlx, P9080_INTCSR, dwIntStatus | (BIT8|BIT10));

    return TRUE;
}

void P9080_IntDisable (P9080_HANDLE hPlx)
{
    DWORD dwIntStatus;

    if (!hPlx->fUseInt) return;
    if (!hPlx->Int.hThread) return;

    // this disables interrupts
    dwIntStatus = P9080_ReadReg (hPlx, P9080_INTCSR);
    P9080_WriteReg (hPlx, P9080_INTCSR, dwIntStatus & ~(BIT8|BIT10));

    // this calls WD_IntDisable()
    InterruptThreadDisable(hPlx->Int.hThread);
    hPlx->Int.hThread = NULL;
}

P9080_DMA_HANDLE P9080_DMAOpen (P9080_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
    DWORD dwBytes, BOOL fIsRead, P9080_MODE mode, P9080_DMA_CHANNEL dmaChannel)
{
    DWORD dwDMAMODE, dwDMADPR, dwDMALADR;
    DWORD dwChannelShift = (dmaChannel==P9080_DMA_CHANNEL_0) ? 0 : P9080_DMA_CHANNEL_SHIFT;
    BOOL fAutoinc = TRUE;
    P9080_DMA_HANDLE hDma;
    
    hDma = malloc (sizeof(P9080_DMA_STRUCT));
    if (hDma==NULL)
    {
        sprintf( P9080_ErrorString, "Failed allocating memory for dma handle!\n");
        goto Exit;
    }
    BZERO (*hDma);
    hDma->dmaChannel = dmaChannel;
    hDma->dma.dwBytes = dwBytes;
    hDma->dma.pUserAddr = buf; 
    hDma->dma.dwOptions = 0; 
    WD_DMALock (hPlx->hWD, &hDma->dma);
    if (!hDma->dma.hDma) 
    {
        sprintf( P9080_ErrorString, "Failed locking the buffer!\n");
        goto Exit;
    }
    if (hDma->dma.dwPages==1)
    {
        //dma of one page ==> direct dma
        dwDMAMODE = 
            (fAutoinc ? 0 : BIT11) 
            | BIT6 
            | (mode==P9080_MODE_BYTE ? 0 : mode==P9080_MODE_WORD ? BIT0 : (BIT1 | BIT0));
        dwDMADPR = BIT0 | (fIsRead ? BIT3 : 0);
        dwDMALADR = dwLocalAddr;

        P9080_WriteReg (hPlx, P9080_DMAMODE + dwChannelShift, dwDMAMODE);
        P9080_WriteReg (hPlx, P9080_DMAPADR + dwChannelShift, (DWORD) hDma->dma.Page[0].pPhysicalAddr);
        P9080_WriteReg (hPlx, P9080_DMALADR + dwChannelShift, dwDMALADR);
        P9080_WriteReg (hPlx, P9080_DMASIZ + dwChannelShift, hDma->dma.Page[0].dwBytes);
        P9080_WriteReg (hPlx, P9080_DMADPR + dwChannelShift, dwDMADPR);
    }
    else
    {
        DWORD dwAlignShift, dwPageNumber, dwMemoryCopied;
        typedef struct {
            DWORD dwPADR;
            DWORD dwLADR;
            DWORD dwSIZ;
            DWORD dwDPR;
        } DMA_LIST;
        DMA_LIST *pList;

        //dma of more then one page ==> chain dma
        hDma->dmaList.dwBytes = hDma->dma.dwPages * sizeof(DMA_LIST) + 0x10; // includes extra 0x10 bytes for quadword alignment
        hDma->dmaList.pUserAddr = NULL;
        hDma->dmaList.dwOptions = DMA_KERNEL_BUFFER_ALLOC;
        WD_DMALock (hPlx->hWD, &hDma->dmaList);
        if (!hDma->dmaList.hDma)
        {
            sprintf (P9080_ErrorString, "Failed locking the chain buffer!\n");
            goto Exit;
        }

        //setting chain of dma pages in the memory
        dwMemoryCopied = 0;
        dwAlignShift = 0x10 - (DWORD) hDma->dmaList.pUserAddr & 0xf;  
        // verifcation that bits 0-3 are zero (quadword aligned)
        pList = (DMA_LIST *) ((DWORD) hDma->dmaList.pUserAddr + dwAlignShift);
        for (dwPageNumber=0; dwPageNumber<hDma->dma.dwPages; dwPageNumber++)
        {
            pList[dwPageNumber].dwPADR = (DWORD) hDma->dma.Page[dwPageNumber].pPhysicalAddr;
            pList[dwPageNumber].dwLADR = dwLocalAddr + (fAutoinc ? dwMemoryCopied : 0);
            pList[dwPageNumber].dwSIZ = hDma->dma.Page[dwPageNumber].dwBytes;
            pList[dwPageNumber].dwDPR = 
                ((DWORD) hDma->dmaList.Page[0].pPhysicalAddr + dwAlignShift + sizeof(DMA_LIST)*(dwPageNumber+1))
                | BIT0 | (fIsRead ? BIT3 : 0);
            dwMemoryCopied += hDma->dma.Page[dwPageNumber].dwBytes;
        }
        pList[dwPageNumber - 1].dwDPR |= BIT1; // mark end of chain
    
        dwDMAMODE = (fAutoinc ? 0 : BIT11) 
                    | BIT6
                    | BIT9        // chain transfer
                    | (mode==P9080_MODE_BYTE ? 0 : mode==P9080_MODE_WORD ? BIT0 : (BIT1 | BIT0));
        dwDMADPR = ((DWORD)hDma->dmaList.Page[0].pPhysicalAddr + dwAlignShift) | BIT0; 
        // starting the dma
        P9080_WriteReg (hPlx, P9080_DMAMODE + dwChannelShift, dwDMAMODE);
        P9080_WriteReg (hPlx, P9080_DMADPR + dwChannelShift, dwDMADPR);
    }

    return hDma;

Exit:
    if (hDma!=NULL)
        P9080_DMAClose(hPlx,hDma);
    return NULL;
}

void P9080_DMAClose (P9080_HANDLE hPlx, P9080_DMA_HANDLE hDma)
{
    if (hDma->dma.hDma)
        WD_DMAUnlock(hPlx->hWD, &hDma->dma);
    if (hDma->dmaList.hDma)
        WD_DMAUnlock(hPlx->hWD, &hDma->dmaList);
    free (hDma);
}

BOOL P9080_DMAIsDone (P9080_HANDLE hPlx, P9080_DMA_HANDLE hDma)
{
    return (P9080_ReadByte(hPlx, P9080_ADDR_REG, P9080_DMACSR + 
        hDma->dmaChannel) & BIT4) == BIT4;
}

void P9080_DMAStart (P9080_HANDLE hPlx, P9080_DMA_HANDLE hDma, BOOL fBlocking)
{
    P9080_WriteByte(hPlx, P9080_ADDR_REG, P9080_DMACSR + hDma->dmaChannel, 
        BIT0 | BIT1);

    //Busy wait for plx to finish transfer
    if (fBlocking) 
        while (!P9080_DMAIsDone(hPlx, hDma))
            ;
}

BOOL P9080_DMAReadWriteBlock (P9080_HANDLE hPlx, DWORD dwLocalAddr, PVOID buf, 
    DWORD dwBytes, BOOL fIsRead, P9080_MODE mode, P9080_DMA_CHANNEL dmaChannel)
{
    P9080_DMA_HANDLE hDma;
    if (dwBytes==0) 
        return TRUE;

    hDma = P9080_DMAOpen(hPlx, dwLocalAddr, buf, dwBytes, fIsRead, mode, dmaChannel);
    if (!hDma) 
        return FALSE;

    P9080_DMAStart(hPlx, hDma, TRUE);
    P9080_DMAClose(hPlx, hDma);
    return TRUE;
}

void P9080_EEPROMDelay(P9080_HANDLE hPlx)
{
    WD_SLEEP sleep;
    BZERO (sleep);
    sleep.dwMicroSeconds = 500;
    WD_Sleep( hPlx->hWD, &sleep);
}

void P9080_EEPROMChipSelect(P9080_HANDLE hPlx, BOOL fSelect)
{
    DWORD dwCNTRL = P9080_ReadReg(hPlx, P9080_CNTRL);
    if (fSelect)
        dwCNTRL |= BIT25;
    else
        dwCNTRL &= ~BIT25;
    P9080_WriteReg(hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);
}

BOOL P9080_EEPROMValid(P9080_HANDLE hPlx)
{
    return (P9080_ReadReg(hPlx, P9080_CNTRL) & BIT28)==BIT28;
}

void P9080_EEPROMWriteBit(P9080_HANDLE hPlx, BOOL fBit)
{
    DWORD dwCNTRL = P9080_ReadReg(hPlx, P9080_CNTRL);

    dwCNTRL &= ~BIT24;
    if (fBit) // data
        dwCNTRL |= BIT26;
    else
        dwCNTRL &= ~BIT26;
    P9080_WriteReg( hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);

    dwCNTRL |= BIT24; // clock
    P9080_WriteReg( hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);

    dwCNTRL &= ~BIT24;
    P9080_WriteReg( hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);
}

BOOL P9080_EEPROMReadBit(P9080_HANDLE hPlx)
{
    BOOL fRet;
    DWORD dwCNTRL = P9080_ReadReg(hPlx, P9080_CNTRL);

    dwCNTRL &= ~BIT24;
    P9080_WriteReg( hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);

    dwCNTRL |= BIT24; // clock
    P9080_WriteReg( hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);

    dwCNTRL &= ~BIT24;
    P9080_WriteReg( hPlx, P9080_CNTRL, dwCNTRL);
    P9080_EEPROMDelay(hPlx);

    fRet = (P9080_ReadReg( hPlx, P9080_CNTRL) & BIT27)==BIT27;

    return fRet;
}

void P9080_EEPROMWriteEnableDisable(P9080_HANDLE hPlx, BOOL fEnable)
{
    P9080_EEPROMChipSelect(hPlx, TRUE);

    // send a WEN instruction
    P9080_EEPROMWriteBit(hPlx, 1);
    P9080_EEPROMWriteBit(hPlx, 0);
    P9080_EEPROMWriteBit(hPlx, 0);
    P9080_EEPROMWriteBit(hPlx, fEnable ? 1 : 0);
    P9080_EEPROMWriteBit(hPlx, fEnable ? 1 : 0);

    P9080_EEPROMWriteBit(hPlx, 0);
    P9080_EEPROMWriteBit(hPlx, 0);
    P9080_EEPROMWriteBit(hPlx, 0);
    P9080_EEPROMWriteBit(hPlx, 0);

    P9080_EEPROMChipSelect(hPlx, FALSE);
}

BOOL P9080_EEPROMReadWord(P9080_HANDLE hPlx, DWORD dwOffset, PWORD pwData)
{
    DWORD dwAddr = dwOffset >> 1;
    DWORD i;

    *pwData = 0;

    P9080_EEPROMChipSelect(hPlx, TRUE);
    P9080_EEPROMWriteBit(hPlx, 1);
    P9080_EEPROMWriteBit(hPlx, 1);
    P9080_EEPROMWriteBit(hPlx, 0);
    // if it's a CS46 EEPROM send only 5 bit address
    for ( i = hPlx->fUseCS46EEPROM ? BIT5 : BIT7; i; i = i>>1)
    {
        P9080_EEPROMWriteBit(hPlx, (dwAddr & i) == i);
    }
    for (i=BIT15; i; i = i>>1)
    {
        *pwData |= P9080_EEPROMReadBit(hPlx) ? i : 0;
    }

    P9080_EEPROMWriteEnableDisable(hPlx, FALSE);
    
    return TRUE;
}

BOOL P9080_EEPROMWriteWord(P9080_HANDLE hPlx, DWORD dwOffset, WORD wData)
{
    DWORD dwAddr = dwOffset >> 1;
    DWORD i;
    WORD readback;

    P9080_EEPROMWriteEnableDisable(hPlx, TRUE);

    P9080_EEPROMChipSelect(hPlx, TRUE);

    // send a PRWRITE instruction
    P9080_EEPROMWriteBit(hPlx, 1);
    P9080_EEPROMWriteBit(hPlx, 0);
    P9080_EEPROMWriteBit(hPlx, 1);
    // if it's a CS46 EEPROM send only a 6 bit address
    for ( i = hPlx->fUseCS46EEPROM ? BIT5 : BIT7; i; i = i>>1)
    {
        P9080_EEPROMWriteBit(hPlx, (dwAddr & i) == i);
    }
    for (i=BIT15; i; i = i>>1)
    {
        P9080_EEPROMWriteBit(hPlx, (wData & i) == i);
    }

    P9080_EEPROMChipSelect(hPlx, FALSE);
    
    P9080_EEPROMWriteEnableDisable(hPlx, FALSE);

    if (P9080_EEPROMReadWord(hPlx, dwOffset, &readback))
    {
        if (wData != readback)
        {
            sprintf( P9080_ErrorString, "Write 0x%04x, Read 0x%04x\n", wData, readback);
            return FALSE;
        }
    }
    else
        return FALSE;

    return TRUE;
}

BOOL P9080_EEPROMReadDWord(P9080_HANDLE hPlx, DWORD dwOffset, PDWORD pdwData)
{
    WORD wData1, wData2;
    if (dwOffset % 4)
    {
        sprintf( P9080_ErrorString, "The offset is not a multiple of 4\n");
        return FALSE;
    }

    if (!P9080_EEPROMReadWord(hPlx, dwOffset, &wData1))
        return FALSE;
    if (!P9080_EEPROMReadWord(hPlx, dwOffset+2, &wData2))
        return FALSE;

    *pdwData = (DWORD) ((wData1 << 16) + wData2);
    return TRUE;
}

BOOL P9080_EEPROMWriteDWord(P9080_HANDLE hPlx, DWORD dwOffset, DWORD dwData)
{
    WORD wData1, wData2;

    if (dwOffset % 4)
    {
        sprintf( P9080_ErrorString, "The offset is not a multiple of 4\n");
        return FALSE;
    }

    wData1 = (WORD) (dwData >> 16);
    wData2 = (WORD) (dwData & 0xffff);

    if (!P9080_EEPROMWriteWord(hPlx, dwOffset, wData1))
        return FALSE;
    if (!P9080_EEPROMWriteWord(hPlx, dwOffset+2, wData2))
        return FALSE;

    return TRUE;
}
                                      
