//////////////////////////////////////////////////////////////////////
// File - pci_lib.c
//
// Library for accessing a PCI card.
// Code was generated by WinDriver Wizard.
// It accesses hardware via WinDriver functions.
// 
//////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include "pci_lib.h"
#include "../../include/windrvr_int_thread.h"


// this string is set to an error message, if one occurs
CHAR PCI_ErrorString[1024];

// internal function used by PCI_Open()
BOOL PCI_DetectCardElements(PCI_HANDLE hPCI);

DWORD PCI_CountCards (DWORD dwVendorID, DWORD dwDeviceID)
{
    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    HANDLE hWD;

    PCI_ErrorString[0] = '\0';
    hWD = WD_Open();
    // check if handle valid & version OK
    if (hWD==INVALID_HANDLE_VALUE) 
    {
        sprintf( PCI_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        return 0;
    }

    BZERO(ver);
    WD_Version(hWD,&ver);
    if (ver.dwVer<WD_VER) 
    {
        sprintf( PCI_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        WD_Close (hWD);
        return 0;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hWD, &pciScan);
    WD_Close (hWD);
    if (pciScan.dwCards==0)
        sprintf( PCI_ErrorString, "no cards found\n");
    return pciScan.dwCards;
}

BOOL PCI_Open (PCI_HANDLE *phPCI, DWORD dwVendorID, DWORD dwDeviceID, DWORD nCardNum, DWORD options)
{
    PCI_HANDLE hPCI = (PCI_HANDLE) malloc (sizeof (PCI_STRUCT));

    WD_VERSION ver;
    WD_PCI_SCAN_CARDS pciScan;
    WD_PCI_CARD_INFO pciCardInfo;

    *phPCI = NULL;
    PCI_ErrorString[0] = '\0';
    BZERO(*hPCI);

    hPCI->cardReg.hCard = 0;
    hPCI->hWD = WD_Open();

    // check if handle valid & version OK
    if (hPCI->hWD==INVALID_HANDLE_VALUE)
    {
        sprintf( PCI_ErrorString, "Failed opening " WD_PROD_NAME " device\n");
        goto Exit;
    }

    BZERO(ver);
    WD_Version(hPCI->hWD,&ver);
    if (ver.dwVer<WD_VER)
    {
        sprintf( PCI_ErrorString, "Incorrect " WD_PROD_NAME " version\n");
        goto Exit;
    }

    BZERO(pciScan);
    pciScan.searchId.dwVendorId = dwVendorID;
    pciScan.searchId.dwDeviceId = dwDeviceID;
    WD_PciScanCards (hPCI->hWD, &pciScan);
    if (pciScan.dwCards==0) // Found at least one card
    {
        sprintf( PCI_ErrorString, "Could not find PCI card\n");
        goto Exit;
    }
    if (pciScan.dwCards<=nCardNum)
    {
        sprintf( PCI_ErrorString, "Card out of range of available cards\n");
        goto Exit;
    }

    BZERO(pciCardInfo);
    pciCardInfo.pciSlot = pciScan.cardSlot[nCardNum];
    WD_PciGetCardInfo (hPCI->hWD, &pciCardInfo);
    hPCI->pciSlot = pciCardInfo.pciSlot;
    hPCI->cardReg.Card = pciCardInfo.Card;

    hPCI->fUseInt = (options & PCI_OPEN_USE_INT) ? TRUE : FALSE;
    if (!hPCI->fUseInt)
    {
        DWORD i;
        // Remove interrupt item if not needed
        for (i=0; i<hPCI->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPCI->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->item = ITEM_NONE;
        }
    }
    else
    {
        DWORD i;
        // make interrupt resource sharable
        for (i=0; i<hPCI->cardReg.Card.dwItems; i++)
        {
            WD_ITEMS *pItem = &hPCI->cardReg.Card.Item[i];
            if (pItem->item==ITEM_INTERRUPT)
                pItem->fNotSharable = FALSE;
        }
    }

    hPCI->cardReg.fCheckLockOnly = FALSE;
    WD_CardRegister (hPCI->hWD, &hPCI->cardReg);
    if (hPCI->cardReg.hCard==0)
    {
        sprintf ( PCI_ErrorString, "Failed locking device\n");
        goto Exit;
    }

    if (!PCI_DetectCardElements(hPCI))
    {
        sprintf ( PCI_ErrorString, "Card does not have all items expected for PCI\n");
        goto Exit;
    }

    // Open finished OK
    *phPCI = hPCI;
    return TRUE;

Exit:
    // Error during Open
    if (hPCI->cardReg.hCard) 
        WD_CardUnregister(hPCI->hWD, &hPCI->cardReg);
    if (hPCI->hWD!=INVALID_HANDLE_VALUE)
        WD_Close(hPCI->hWD);
    free (hPCI);
    return FALSE;
}

void PCI_Close(PCI_HANDLE hPCI)
{
    // disable interrupts
    if (PCI_IntIsEnabled(hPCI))
        PCI_IntDisable(hPCI);

    // unregister card
    if (hPCI->cardReg.hCard) 
        WD_CardUnregister(hPCI->hWD, &hPCI->cardReg);

    // close WinDriver
    WD_Close(hPCI->hWD);

    free (hPCI);
}

void PCI_WritePCIReg(PCI_HANDLE hPCI, DWORD dwReg, DWORD dwData)
{
    WD_PCI_CONFIG_DUMP pciCnf;

    BZERO(pciCnf);
    pciCnf.pciSlot = hPCI->pciSlot;
    pciCnf.pBuffer = &dwData;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = FALSE;
    WD_PciConfigDump(hPCI->hWD,&pciCnf);
}

DWORD PCI_ReadPCIReg(PCI_HANDLE hPCI, DWORD dwReg)
{
    WD_PCI_CONFIG_DUMP pciCnf;
    DWORD dwVal;

    BZERO(pciCnf);
    pciCnf.pciSlot = hPCI->pciSlot;
    pciCnf.pBuffer = &dwVal;
    pciCnf.dwOffset = dwReg;
    pciCnf.dwBytes = 4;
    pciCnf.fIsRead = TRUE;
    WD_PciConfigDump(hPCI->hWD,&pciCnf);
    return dwVal;
}

BOOL PCI_DetectCardElements(PCI_HANDLE hPCI)
{
    DWORD i;
    DWORD ad_sp;

    BZERO(hPCI->Int);
    BZERO(hPCI->addrDesc);

    for (i=0; i<hPCI->cardReg.Card.dwItems; i++)
    {
        WD_ITEMS *pItem = &hPCI->cardReg.Card.Item[i];

        switch (pItem->item)
        {
        case ITEM_MEMORY:
        case ITEM_IO:
            {
                DWORD dwBytes;
                DWORD dwPhysAddr;
                BOOL fIsMemory;
                if (pItem->item==ITEM_MEMORY)
                {
                    dwBytes = pItem->I.Mem.dwBytes;
                    dwPhysAddr = pItem->I.Mem.dwPhysicalAddr;
                    fIsMemory = TRUE;
                }
                else 
                {
                    dwBytes = pItem->I.IO.dwBytes;
                    dwPhysAddr = pItem->I.IO.dwAddr;
                    fIsMemory = FALSE;
                }

                for (ad_sp=0; ad_sp<PCI_ITEMS; ad_sp++)
                {
                    DWORD dwPCIAddr;
                    DWORD dwPCIReg;

                    if (PCI_IsAddrSpaceActive(hPCI, ad_sp)) continue;
                    if (ad_sp<PCI_AD_EPROM) dwPCIReg = PCI_BAR0 + 4*ad_sp;
                    else dwPCIReg = PCI_ERBAR;
                    dwPCIAddr = PCI_ReadPCIReg(hPCI, dwPCIReg);
                    if (dwPCIAddr & 1)
                    {
                        if (fIsMemory) continue;
                        dwPCIAddr &= ~0x3;
                    }
                    else
                    {
                        if (!fIsMemory) continue;
                        dwPCIAddr &= ~0xf;
                    }
                    if (dwPCIAddr==dwPhysAddr)
                        break;
                }
                if (ad_sp<PCI_ITEMS)
                {
                    DWORD j;
                    hPCI->addrDesc[ad_sp].fActive = TRUE;
                    hPCI->addrDesc[ad_sp].index = i;
                    hPCI->addrDesc[ad_sp].fIsMemory = fIsMemory;
                    hPCI->addrDesc[ad_sp].dwMask = 0;
                    for (j=1; j<dwBytes && j!=0x80000000; j *= 2)
                    {
                        hPCI->addrDesc[ad_sp].dwMask = 
                            (hPCI->addrDesc[ad_sp].dwMask << 1) | 1;
                    }
                }
            }
            break;
        case ITEM_INTERRUPT:
            if (hPCI->Int.Int.hInterrupt) return FALSE;
            hPCI->Int.Int.hInterrupt = pItem->I.Int.hInterrupt;
            break;
        }
    }

    // check that all the items needed were found
    // check if interrupt found
    if (hPCI->fUseInt && !hPCI->Int.Int.hInterrupt) 
    {
        return FALSE;
    }

    // check that at least one memory space was found
    for (i = 0; i<PCI_ITEMS; i++)
        if (PCI_IsAddrSpaceActive(hPCI, i)) break;
    if (i==PCI_ITEMS) return FALSE;

    return TRUE;
}

BOOL PCI_IsAddrSpaceActive(PCI_HANDLE hPCI, PCI_ADDR addrSpace){
    return hPCI->addrDesc[addrSpace].fActive;
}

// General read/write function
void PCI_ReadWriteBlock(PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset, BOOL fRead, PVOID buf, DWORD dwBytes, PCI_MODE mode)
{
    WD_TRANSFER trans;
    BOOL fMem = hPCI->addrDesc[addrSpace].fIsMemory;
    // safty check: is the address range active
    if (!PCI_IsAddrSpaceActive(hPCI, addrSpace)) return;
    BZERO(trans);
    if (fRead)
    {
        if (mode==PCI_MODE_BYTE) trans.cmdTrans = fMem ? RM_SBYTE : RP_SBYTE;
        else if (mode==PCI_MODE_WORD) trans.cmdTrans = fMem ? RM_SWORD : RP_SWORD;
        else if (mode==PCI_MODE_DWORD) trans.cmdTrans = fMem ? RM_SDWORD : RP_SDWORD;
    }
    else
    {
        if (mode==PCI_MODE_BYTE) trans.cmdTrans = fMem ? WM_SBYTE : WP_SBYTE;
        else if (mode==PCI_MODE_WORD) trans.cmdTrans = fMem ? WM_SWORD : WP_SWORD;
        else if (mode==PCI_MODE_DWORD) trans.cmdTrans = fMem ? WM_SDWORD : WP_SDWORD;
    }
    if (fMem)
        trans.dwPort = hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwTransAddr;
    else 
        trans.dwPort = hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.IO.dwAddr;
    trans.dwPort += dwOffset;

    trans.fAutoinc = TRUE;
    trans.dwBytes = dwBytes;
    trans.dwOptions = 0;
    trans.Data.pBuffer = buf;
    WD_Transfer (hPCI->hWD, &trans);
}

BYTE PCI_ReadByte (PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset)
{
    BYTE data;
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        PBYTE pData = (PBYTE) (hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwUserDirectAddr + dwOffset);
        data = *pData; // read from the memory mapped range directly
    }
    else PCI_ReadWriteBlock( hPCI, addrSpace, dwOffset, TRUE, &data, sizeof (BYTE), PCI_MODE_BYTE);
    return data;
}

WORD PCI_ReadWord (PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset)
{
    WORD data;
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        PWORD pData = (PWORD) (hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwUserDirectAddr + dwOffset);
        data = *pData; // read from the memory mapped range directly
    }
    else PCI_ReadWriteBlock( hPCI, addrSpace, dwOffset, TRUE, &data, sizeof (WORD), PCI_MODE_WORD);
    return data;
}

DWORD PCI_ReadDword (PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset)
{
    DWORD data;
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        PDWORD pData = (PDWORD) (hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwUserDirectAddr + dwOffset);
        data = *pData; // read from the memory mapped range directly
    }
    else PCI_ReadWriteBlock( hPCI, addrSpace, dwOffset, TRUE, &data, sizeof (DWORD), PCI_MODE_DWORD);
    return data;
}

void PCI_WriteByte (PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset, BYTE data)
{
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        PBYTE pData = (PBYTE) (hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwUserDirectAddr + dwOffset);
        *pData = data; // write to the memory mapped range directly
    }
    else PCI_ReadWriteBlock( hPCI, addrSpace, dwOffset, FALSE, &data, sizeof (BYTE), PCI_MODE_BYTE);
}

void PCI_WriteWord (PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset, WORD data)
{
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        PWORD pData = (PWORD) (hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwUserDirectAddr + dwOffset);
        *pData = data; // write to the memory mapped range directly
    }
    else PCI_ReadWriteBlock( hPCI, addrSpace, dwOffset, FALSE, &data, sizeof (WORD), PCI_MODE_WORD);
}

void PCI_WriteDword (PCI_HANDLE hPCI, PCI_ADDR addrSpace, DWORD dwOffset, DWORD data)
{
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        PDWORD pData = (PDWORD) (hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwUserDirectAddr + dwOffset);
        *pData = data; // write to the memory mapped range directly
    }
    else 
        PCI_ReadWriteBlock( hPCI, addrSpace, dwOffset, FALSE, &data, sizeof (DWORD), PCI_MODE_DWORD);
}

BOOL PCI_IntIsEnabled (PCI_HANDLE hPCI)
{
    if (!hPCI->fUseInt) return FALSE;
    if (!hPCI->Int.hThread) return FALSE;
    return TRUE;
}

VOID PCI_IntHandler (PVOID pData)
{
    PCI_HANDLE hPCI = (PCI_HANDLE) pData;
    PCI_INT_RESULT intResult;

    intResult.dwCounter = hPCI->Int.Int.dwCounter;
    intResult.dwLost = hPCI->Int.Int.dwLost;
    intResult.fStopped = hPCI->Int.Int.fStopped;
    hPCI->Int.funcIntHandler(hPCI, &intResult);
}

BOOL PCI_IntEnable (PCI_HANDLE hPCI, PCI_INT_HANDLER funcIntHandler)
{
    PCI_ADDR addrSpace;

    if (!hPCI->fUseInt) return FALSE;
    // check if interrupt is already enabled
    if (hPCI->Int.hThread) return FALSE;

    BZERO(hPCI->Int.Trans);
    // This is a sample of handling interrupts:
    // One transfer commands is issued. You will need to change this code.
    // You must add transfer commands to CANCEL the source of the interrupt, otherwise, the
    // PC will hang when an interrupt occurs!
    addrSpace = PCI_AD_BAR0; // put the address space of the register here
    if (hPCI->addrDesc[addrSpace].fIsMemory)
    {
        hPCI->Int.Trans[0].dwPort = hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.Mem.dwTransAddr;
        hPCI->Int.Trans[0].cmdTrans = WM_DWORD;
    }
    else
    {
        hPCI->Int.Trans[0].dwPort = hPCI->cardReg.Card.Item[hPCI->addrDesc[addrSpace].index].I.IO.dwAddr;
        hPCI->Int.Trans[0].cmdTrans = WP_DWORD;
    }
    hPCI->Int.Trans[0].dwPort += 0; // put the offset of the register from the beginning of the address space here
    hPCI->Int.Trans[0].Data.Dword = 0; // put the data to write to the control register here
    hPCI->Int.Int.dwCmds = 1;
    hPCI->Int.Int.Cmd = hPCI->Int.Trans;
    hPCI->Int.Int.dwOptions |= INTERRUPT_CMD_COPY;

    // this calls WD_IntEnable() and creates an interrupt handler thread
    hPCI->Int.funcIntHandler = funcIntHandler;
    if (!InterruptThreadEnable(&hPCI->Int.hThread, hPCI->hWD, &hPCI->Int.Int, PCI_IntHandler, (PVOID) hPCI))
        return FALSE;

    return TRUE;
}

void PCI_IntDisable (PCI_HANDLE hPCI)
{
    if (!hPCI->fUseInt) return;
    if (!hPCI->Int.hThread) return;

    // this calls WD_IntDisable()
    InterruptThreadDisable(hPCI->Int.hThread);

    hPCI->Int.hThread = NULL;
}

