#include <cassert>
#include <memory>

#include "soundengine.h"
#include "compression.h"
#include "common.h"
#include "logger.h"
#include "vfs.h"

using std::auto_ptr;
using std::string;

SoundEngine::SoundEngine(bool disableSound) : nosound(disableSound), mutesound(false), currentTrack(playlist.begin())
{
    if (nosound)
        return;

    if (SDL_BuildAudioCVT(&eightbitconv, AUDIO_U8, 1, 22050, SOUND_FORMAT, SOUND_CHANNELS, SOUND_FREQUENCY) < 0) {
        logger->error("Could not build 8bit->16bit conversion filter\n");
        nosound = true;
        return;
    }

    if (SDL_BuildAudioCVT(&monoconv, AUDIO_S16, 1, 22050, SOUND_FORMAT, SOUND_CHANNELS, SOUND_FREQUENCY) < 0) {
        logger->error("Could not build mono->stereo conversion filter\n");
        nosound = true;
        return;
    }

    if (Mix_OpenAudio(SOUND_FREQUENCY, SOUND_FORMAT, SOUND_CHANNELS, 4096) < 0) {
        logger->error("Unable to open sound: %s\n", Mix_GetError());
        nosound = true;
    }
}

SoundEngine::~SoundEngine()
{
    if (nosound)
        return;

    // Clean up sound cache
    for (SoundCache::iterator it = soundCache.begin(); it != soundCache.end(); ++it) {
        Mix_FreeChunk(it->second->chunk);
        delete it->second;
    }

    // Clean up music cache
    for (MusicCache::iterator it = musicCache.begin(); it != musicCache.end(); ++it) {
        delete it->second;
    }

    Mix_CloseAudio();
}

//-----------------------------------------------------------------------------
// Game Sounds
//-----------------------------------------------------------------------------

void SoundEngine::MuteSound(bool mute)
{
    mutesound = mute;
}

void SoundEngine::PlaySound(const std::string& sound)
{
    if (nosound || mutesound)
        return;

    Mix_PlayChannel(-1, LoadSoundImpl(sound)->chunk, 0);
}

int SoundEngine::PlayLoopedSound(const std::string& sound, unsigned int loops)
{
    if (nosound || mutesound)
        return -1;

    return Mix_PlayChannel(-1, LoadSoundImpl(sound)->chunk, static_cast<int>(loops)-1);
}

void SoundEngine::StopLoopedSound(int id)
{
    if (nosound || mutesound)
        return;

    Mix_HaltChannel(id);
}

//-----------------------------------------------------------------------------
// Music Playback
//-----------------------------------------------------------------------------

bool SoundEngine::CreatePlaylist(const std::string& gamename)
{
    // Create the amazing playlist
    playlist.clear();
    playlist.push_back("aoi.aud");
    currentTrack = playlist.begin();

    return true;
}

void SoundEngine::PlayMusic()
{
    if (nosound)
        return;

    if (!Mix_PlayingMusic()) {
        if (Mix_PausedMusic()) {
            Mix_ResumeMusic();
        } else {
            PlayTrack(*currentTrack);
        }
    }
}

void SoundEngine::PauseMusic()
{
    if (nosound)
        return;

    Mix_PauseMusic();
}

void SoundEngine::StopMusic()
{
    if (nosound)
        return;

    Mix_HookMusic(NULL, NULL);
}

void SoundEngine::PlayTrack(const std::string& sound)
{
    if (nosound)
        return;

    StopMusic();

    if (sound == "No theme")
        PlayMusic();

    MusicCache::iterator cachedTrack = musicCache.find(sound);
    if (cachedTrack != musicCache.end()) {
        musicStream.buffer = cachedTrack->second;
    } else {
        SampleBuffer* buffer = new SampleBuffer();
        if (Decode(sound, *buffer)) {
            musicCache.insert(MusicCache::value_type(sound, buffer));
            musicStream.buffer = buffer;
        } else {
            delete buffer;
            return;
        }
    }
    musicStream.pos = 0;
    Mix_HookMusic(MusicHook, reinterpret_cast<void*>(&musicStream));
}

void SoundEngine::NextTrack()
{
    if (nosound)
        return;

    if (++currentTrack == playlist.end())
        currentTrack = playlist.begin();

    PlayTrack(*currentTrack);
}

void SoundEngine::PrevTrack()
{
    if (nosound)
        return;

    if (currentTrack == playlist.begin())
        currentTrack = playlist.end();

    PlayTrack(*(--currentTrack));
}

void SoundEngine::MusicHook(void* userdata, Uint8* stream, int len)
{
    MusicStream* music = reinterpret_cast<MusicStream*>(userdata);
    Uint32 length = ((music->buffer->size() - music->pos) < (Uint32)len)
            ? (music->buffer->size() - music->pos)
            : len;
    if (length > 0) {
        memcpy(stream, &((*music->buffer)[music->pos]), length);
        music->pos += length;
    }
}

//-----------------------------------------------------------------------------
// Sound loading, caching, decoding
//-----------------------------------------------------------------------------

void SoundEngine::LoadSound(const std::string& sound)
{
    LoadSoundImpl(sound);
}

SoundBuffer* SoundEngine::LoadSoundImpl(const std::string& sound)
{
    // Check if sound is already loaded and cached
    SoundCache::iterator cachedSound = soundCache.find(sound);
    if (cachedSound != soundCache.end()) {
        return cachedSound->second;
    } else {
        // Load and cache sound
        SoundBuffer* buffer = new SoundBuffer();
        if (Decode(sound, buffer->data)) {
            buffer->chunk = Mix_QuickLoad_RAW(&buffer->data[0], static_cast<Uint32>(buffer->data.size()));
            soundCache.insert(SoundCache::value_type(sound, buffer));
            return buffer;
        } else {
            delete buffer;
        }
    }
    return 0;
}

bool SoundEngine::Decode(const std::string& filename, SampleBuffer& buffer)
{
    // Use auto_ptr to ensure file is closed when exiting this function
    auto_ptr<VFile> sndfile(VFS_Open(filename.c_str()));
    if (0 == sndfile.get()) {
        logger->error("Unable to load soundfile: %s\n", filename.c_str());
        return false;
    }
    Uint16 frequency;
    Uint32 comp_size, uncomp_size;
    Uint8 flags, type;

    // Note: Headers are identical for TS and TD auds
    sndfile->readWord(&frequency,1);
    sndfile->readDWord(&comp_size,1);
    sndfile->readDWord(&uncomp_size,1);
    sndfile->readByte(&flags,1);
    sndfile->readByte(&type,1);

    SDL_AudioCVT* conv;
    // check that format is recognised
    if (1 == type) {
        conv = &eightbitconv;
    } else if (99 == type) {
        conv = &monoconv;
    } else {
        logger->error("%s has a corrupt header (Unknown type: %i)\n", filename.c_str(), type);
        return false;
    }

    if (frequency != 22050) {
        logger->warning("\"%s\" needs converting from %iHz (instaed of 22050Hz)\n", filename.c_str(), frequency);
    }

    assert(buffer.empty());
    buffer.resize(uncomp_size*conv->len_mult);

    Uint16 comp_sample_size, uncomp_sample_size;
    Uint32 ID;
    static Uint8 chunk[MAX_CHUNK_SIZE];
    static Uint8 tmpbuff[MAX_UNCOMPRESSED_SIZE * 4];

    Uint32 len = 0;
    Uint32 offset = 12; // Main header is 12 bytes
    sndfile->seekSet(offset);

    // set sample and index to 0, these are used in the decompression alogorithm
    Sint32 sample = 0;
    Sint32 index = 0;

    while ((offset+8) < sndfile->fileSize()) {
        // Each sample has a header
        sndfile->readWord(&comp_sample_size, 1);
        sndfile->readWord(&uncomp_sample_size, 1);
        sndfile->readDWord(&ID, 1);

        if (comp_sample_size > (MAX_CHUNK_SIZE)) {
            logger->warning("Size data for current sample too large\n");
            return false;
        }

        // abort if id was wrong */
        if (ID != 0xDEAF) {
            logger->warning("Sample had wrong ID: %x\n", ID);
            return false;
        }

        // compressed data follows header
        sndfile->readByte(chunk, comp_sample_size);

        if (type == 1) {
            Compression::WSADPCM_Decode(tmpbuff, chunk, comp_sample_size, uncomp_sample_size);
        } else {
            Compression::IMADecode(tmpbuff, chunk, comp_sample_size, &index, &sample);
        }
        conv->buf = tmpbuff;
        conv->len = uncomp_sample_size;
        if (SDL_ConvertAudio(conv) < 0) {
            logger->warning("Could not run conversion filter: %s\n", SDL_GetError());
            return false;
        }
        memcpy(&buffer[len], tmpbuff, uncomp_sample_size*conv->len_mult);

        len += uncomp_sample_size*conv->len_mult;
        offset += comp_sample_size + 8;
    }

    return true;
}
