#include <functional>
#include "buildqueue.h"
#include "common.h"
#include "dispatcher.h"
#include "logger.h"
#include "playerpool.h"
#include "unitandstructurepool.h"
#include "unitorstructure.h"

namespace BuildQueue {
using std::make_pair; using std::equal_to;

struct IsType : std::unary_function<buildorder, bool> {
    IsType(UnitOrStructureType* type) : type_(type) {}
    bool operator()(const buildorder& param) {
        return type_ == param.first;
    }
    UnitOrStructureType* type_;
};

/// @TODO These values should be moved to a configuration file
// I don't think buildspeed needs to be changed once this code is stable.
const Uint8 buildspeed = 1;
//const Uint8 buildspeed = 5;
const Uint8 maxbuild = 99;


/// @TODO When passing buildspeed to ActionEvent the delay is about 30 times longer as it should be -- nap
// I have no idea what he means by this -- zx64
BQEvent::BQEvent(Player *p) : ActionEvent(1), player(p), status(BQ_EMPTY)
{}

BQEvent::~BQEvent() {}


bool BQEvent::add(UnitOrStructureType *type) {
    switch (status) {
    case BQ_INVALID:
        logger->error("Queue %p in invalid state\n", this);
        break;
    case BQ_EMPTY:
        if (0 == (left = type->getCost())) {
            logger->error("Type \"%s\" has no cost\n", type->getTName());
        }
        last = p::aequeue->getCurtick();
        queue.push_back(std::make_pair(type, 1));
        status = BQ_RUNNING;
        p::aequeue->scheduleEvent(this);
        return true;
        break;
    case BQ_PAUSED:
        if (getCurrentType() == type) {
            status = BQ_RUNNING;
            last = p::aequeue->getCurtick();
            p::aequeue->scheduleEvent(this);
        }
        return true;
        break;
    case BQ_READY:
        // Can't enqueue, waiting for placement
        if (!type->isStructure()) {
            if (p::dispatcher->unitSpawn((UnitType*)type, player->getPlayerNum())) {
                placed();
            } else {
                logger->debug("Didn't spawn %s...\n", type->getTName());
            }
        }
        return false;
        break;
    case BQ_RUNNING:
        // First try to requeue another of the type
        switch (requeue(type)) {
        case RQ_DONE:
            return true;
        case RQ_MAXED:
            return false;
        case RQ_NEW:
            // This type is new to the queue
            if (0 == type->getCost()) {
                // We divide by cost, so must not be zero.
                logger->error("Type \"%s\" has no cost\n", type->getTName());
                return false;
            }
            queue.push_back(std::make_pair(type, 1));
            return true;
        }
        return false;
        break;
    default:
        logger->error("Queue %p in /really/ invalid state (%i)\n", this, status);
        break;
    }
    return false;
}

ConStatus BQEvent::pausecancel(UnitOrStructureType* type) {
    if (BQ_EMPTY == status) {
        return BQ_EMPTY;
    }
    // find instance of type
    Queue::iterator it = find_if(queue.begin(), queue.end(), IsType(type));
    if (queue.end() == it) {
        // Not queued this type at all
        // UI decision: do we behave as if we had clicked on the type currently
        // being built?
        return BQ_EMPTY;
    }
    if (queue.begin() == it) {
        switch (status) {
        case BQ_RUNNING:
            status = BQ_PAUSED;
            p::ppool->updateSidebar();
            return status;
            break;
        case BQ_READY:
            // Fallthrough
        case BQ_PAUSED:
            if (it->second > 1) {
                --it->second;
            } else {
                // Refund what we've spent so far
                player->changeMoney(getCurrentType()->getCost()-left);
                // Remain in paused state
                next();
            }
            break;
        default:
            logger->error("Unhandled state in pausecancel: %i\n", status);
            break;
        }
    } else {
        if (it->second > 1) {
            --it->second;
        } else {
            queue.erase(it);
        }
    }

    p::ppool->updateSidebar();
    return BQ_CANCELLED;
}

void BQEvent::run() {
    if (BQ_RUNNING != status) {
        return;
    }
    Uint8 delta = min((p::aequeue->getCurtick()-last)/buildspeed, left);

    if (delta == 0) {
        return;
    }
    last += delta*buildspeed;
    /// @TODO Play "tink" sound
    if (!player->changeMoney(-delta)) {
        /// @TODO Play "insufficient funds" sound
        // Note: C&C didn't put build on hold when you initially run out, so you
        // could leave something partially built whilst waiting for the
        // harvester to return

        // Reschedule to keep checking for money
        p::aequeue->scheduleEvent(this);
        return;
    }
    left -= delta;

    if (0 == left) {
        UnitOrStructureType* type = queue.front().first;
        // For structures and blocked unit production, we wait for user
        // interaction.
        status = BQ_READY;
        if (!type->isStructure()) {
            UnitType* utype = (UnitType*)type;
            if (p::dispatcher->unitSpawn(utype, player->getPlayerNum())) {
                /// @TODO Play "unit ready" sound
                // If we were able to spawn the unit, move onto the next
                // item
                status = BQ_RUNNING;
                next();
            }
        } else {
            /// @TODO Play "construction complete" sound
        }
    } else {
        p::aequeue->scheduleEvent(this);
    }
    p::ppool->updateSidebar();
}

ConStatus BQEvent::getStatus(UnitOrStructureType* type, Uint8* quantity, Uint8* progress) const
{
    *quantity = 0;
    *progress = 100; // Default to not grey'd out.
    if (BQ_EMPTY == status || BQ_INVALID == status) {
        // Fast exit for states with nothing to report
        return status;
    }
    if (getCurrentType() == type) {
        *quantity = queue.front().second;
        *progress = 100*(getCurrentType()->getCost()-left)/getCurrentType()->getCost();
    } else {
        Queue::const_iterator it;

        *progress = 0;
        it = find_if(queue.begin(), queue.end(), IsType(type));
        if (queue.end() != it) {
            *quantity = it->second;
        }
    }
    return status;
}

void BQEvent::next()
{
    if (queue.front().second <= 1) {
        queue.erase(queue.begin());
        if (queue.empty()) {
            status = BQ_EMPTY;
            return;
        } else {
            // None left of the current type of thing being built, so move onto
            // the next item in the queue and start building
            status = BQ_RUNNING;
        }
    } else {
        --queue.front().second;
    }
    left = queue.front().first->getCost();
    last = p::aequeue->getCurtick();
    p::aequeue->scheduleEvent(this);
    p::ppool->updateSidebar();
}

BQEvent::RQstate BQEvent::requeue(UnitOrStructureType* type)
{
    Queue::iterator it;

    it = find_if(queue.begin(), queue.end(), IsType(type));
    if (queue.end() == it) {
        return RQ_NEW;
    }
    if (it->second > maxbuild) {
        return RQ_MAXED;
    }
    it->second++;
    return RQ_DONE;
}

/// Used for other bits of code to notify the buildqueue that they can continue
// building having placed what was waiting for a position
void BQEvent::placed()
{
    p::ppool->updateSidebar();
    status = BQ_RUNNING;
    next();
}

} /* namespace BuildQueue */
