#ifndef SOUNDENGINE_H
#define SOUNDENGINE_H

#include <map>
#include <string>
#include <vector>
#include <stdexcept>

#include "SDL_audio.h"
#include "SDL_mixer.h"

#define SOUND_FORMAT          AUDIO_S16SYS
#define SOUND_CHANNELS        2
#define SOUND_FREQUENCY       22050

#define MAX_CHUNK_SIZE        16384
#define MAX_UNCOMPRESSED_SIZE MAX_CHUNK_SIZE << 2
#define MAX_COMPRESSED_SIZE   MAX_CHUNK_SIZE

typedef std::vector<Uint8> SampleBuffer;
struct SoundBuffer {
    SampleBuffer data;
    Mix_Chunk* chunk;
};

struct MusicStream {
    Uint32 pos;
    SampleBuffer* buffer;
};

class SoundEngine
{
public:
    SoundEngine(bool disableSound = false);
    ~SoundEngine();

    void LoadSound(const std::string& sound);

    // Game sounds
    void MuteSound(bool mute);
    void PlaySound(const std::string& sound);
    int  PlayLoopedSound(const std::string& sound, unsigned int loops);
    void StopLoopedSound(int id);

    // Music
    bool CreatePlaylist(const std::string& gamename);
    void PlayMusic(); // Starts|Resumes playing
    void PauseMusic();
    void StopMusic();

    void PlayTrack(const std::string& sound); // Plays a specific track
    void NextTrack(); // Selects the next track in the playlist
    void PrevTrack(); // Selects the previous track in the playlist
    
    static void MusicHook(void* userdata, Uint8* stream, int len);

private:
    typedef std::map<std::string, SoundBuffer*>  SoundCache;
    typedef std::map<std::string, SampleBuffer*> MusicCache;
    typedef std::vector<std::string>             Playlist;

    SoundBuffer* LoadSoundImpl(const std::string& sound);
    bool Decode(const std::string& filename, SampleBuffer& buffer);

    bool nosound;
    bool mutesound;

    SoundCache  soundCache;
    MusicCache  musicCache;
    MusicStream musicStream;
    
    Playlist playlist;
    Playlist::iterator currentTrack;

    SDL_AudioCVT eightbitconv;
    SDL_AudioCVT monoconv;
};

#endif
