//-------------------------------------------------------------------------
// This program demonstrates some of the capabilities of the slider
// control.  It doesn't do everything you want to see, but it is an
// interesting application nonetheless.
//
// Copyright (c) 1996 by Larry Salomon Jr.
// All rights reserved.
//-------------------------------------------------------------------------
#define INCL_GPIPRIMITIVES
#define INCL_WINDIALOGS
#define INCL_WINFRAMEMGR
#define INCL_WINRECTANGLES
#define INCL_WINSTDSLIDER
#define INCL_WINSYS
#define INCL_WINWINDOWMGR
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include "introrc.h"

#define CLS_CLIENT               "ClientClass"

typedef struct {
   ULONG ulSzStruct;             // Size of the structure
   HAB habAnchor;                // Anchor block handle
   HWND hwFrame;                 // Frame window handle
   HWND hwClient;                // Client window handle
   HWND hwDlg;                   // Dialog window handle
   LONG alPoints[3];             // Current slider positions
   ULONG aulDetentes[3];         // Current detente identifiers
   PFNWP pfnOldStatic;           // Old static window procedure
} INSTDATA, *PINSTDATA;

MRESULT EXPENTRY newStaticProc(HWND hwWnd,
                               ULONG ulMsg,
                               MPARAM mpParm1,
                               MPARAM mpParm2)
//-------------------------------------------------------------------------
// This is the new window procedure for the static control which displays
// the fillet based on the slider positions.
//
// Parameters:  (standard)
//-------------------------------------------------------------------------
{
   HWND hwParent;
   PINSTDATA pidData;

   hwParent=WinQueryWindow(hwWnd,QW_PARENT);
   hwParent=WinQueryWindow(hwParent,QW_PARENT);
   pidData=(PINSTDATA)WinQueryWindowPtr(hwParent,0);

   switch (ulMsg) {
   case WM_PAINT:
      {
         HPS hpsPaint;
         RECTL rclPaint;
         RECTL rclWnd;
         RECTL rclClip;
         LONG lDx;
         LONG lDy;
         ULONG ulIndex;
         POINTL aptlPoints[3];

         hpsPaint=WinBeginPaint(hwWnd,NULLHANDLE,&rclPaint);

         //----------------------------------------------------------------
         // We have to do this because I cannot figure out the proper
         // incantation of flags to clip the window to itself.  I've
         // never been able to figure this out, so if anyone else knows,
         // I'd appreciate an email.
         //----------------------------------------------------------------
         WinQueryWindowRect(hwWnd,&rclWnd);
         WinIntersectRect(pidData->habAnchor,&rclClip,&rclWnd,&rclPaint);

         //----------------------------------------------------------------
         // Fill ourselves with white.
         //----------------------------------------------------------------
         WinFillRect(hpsPaint,&rclClip,CLR_WHITE);

         //----------------------------------------------------------------
         // Calculate the deltas needed.  To understand how this works,
         // refer to the diagram below:
         //
         // +-----------+
         // |           |
         // X     X     X
         // |           |
         // +-----------+
         //
         // For 3 points, DX should be CX/2.  This is because we start at
         // position 0 and not position DX.  In general, for N points,
         // DX and DY are CX/(N-1) and CY/(N-1).
         //
         // However, we want a margin around the points, so assume there
         // are N+1 points and deflate the rectange should be DX/2 and
         // DY/2.
         //
         // The N+1 statement explains why we devide by 3 (sliders) and
         // 10 (points per slider).
         //----------------------------------------------------------------
         lDx=(rclWnd.xRight-rclWnd.xLeft)/3;
         lDy=(rclWnd.yTop-rclWnd.yBottom)/10;

         WinInflateRect(pidData->habAnchor,&rclWnd,-lDx/2,-lDy/2);

         //----------------------------------------------------------------
         // Calculate the points first so that we can draw the fillet
         // before the markers.  This allows the markers to overlay the
         // fillet when necessary.
         //----------------------------------------------------------------
         for (ulIndex=0; ulIndex<3; ulIndex++) {
            aptlPoints[ulIndex].x=rclWnd.xLeft+ulIndex*lDx;
            aptlPoints[ulIndex].y=
               rclWnd.yBottom+pidData->alPoints[ulIndex]*lDy;
         } /* endfor */

         //----------------------------------------------------------------
         // Draw the fillet.
         //----------------------------------------------------------------
         GpiSetColor(hpsPaint,CLR_BLACK);
         GpiMove(hpsPaint,&aptlPoints[0]);
         GpiPolyFillet(hpsPaint,2,&aptlPoints[1]);

         //----------------------------------------------------------------
         // Now draw the markers.
         //----------------------------------------------------------------
         GpiSetColor(hpsPaint,CLR_BLUE);

         for (ulIndex=0; ulIndex<3; ulIndex++) {
            aptlPoints[ulIndex].x-=2;
            aptlPoints[ulIndex].y-=2;
            GpiMove(hpsPaint,&aptlPoints[ulIndex]);

            aptlPoints[ulIndex].x+=4;
            aptlPoints[ulIndex].y+=4;
            GpiBox(hpsPaint,DRO_FILL,&aptlPoints[ulIndex],0,0);
         } /* endfor */

         WinEndPaint(hpsPaint);
      }
      break;
   default:
      return (*pidData->pfnOldStatic)(hwWnd,ulMsg,mpParm1,mpParm2);
   } /* endswitch */

   return MRFROMSHORT(FALSE);
}

MRESULT EXPENTRY mainDlgProc(HWND hwWnd,
                             ULONG ulMsg,
                             MPARAM mpParm1,
                             MPARAM mpParm2)
//-------------------------------------------------------------------------
// This is the dialog procedure.
//
// Parameters:  (standard)
//-------------------------------------------------------------------------
{
   PINSTDATA pidData;

   pidData=(PINSTDATA)WinQueryWindowPtr(WinQueryWindow(hwWnd,QW_PARENT),0);

   switch (ulMsg) {
   case WM_INITDLG:
      //-------------------------------------------------------------------
      // Set the tick size on all of the ticks, since all ticks are
      // initially invisible.
      //-------------------------------------------------------------------
      WinSendDlgItemMsg(hwWnd,
                        DMN_SLD_SLIDER1,
                        SLM_SETTICKSIZE,
                        MPFROM2SHORT(SMA_SETALLTICKS,5),
                        0);
      WinSendDlgItemMsg(hwWnd,
                        DMN_SLD_SLIDER2,
                        SLM_SETTICKSIZE,
                        MPFROM2SHORT(SMA_SETALLTICKS,5),
                        0);
      WinSendDlgItemMsg(hwWnd,
                        DMN_SLD_SLIDER3,
                        SLM_SETTICKSIZE,
                        MPFROM2SHORT(SMA_SETALLTICKS,5),
                        0);

      //-------------------------------------------------------------------
      // Subclass the static control.
      //-------------------------------------------------------------------
      pidData->pfnOldStatic=
         WinSubclassWindow(WinWindowFromID(hwWnd,DMN_ST_GRAPH),
                           newStaticProc);
      break;
   case WM_CONTROL:
      switch (SHORT1FROMMP(mpParm1)) {
      case DMN_SLD_SLIDER1:
      case DMN_SLD_SLIDER2:
      case DMN_SLD_SLIDER3:
         switch (SHORT2FROMMP(mpParm1)) {
         case SLN_CHANGE:
            {
               ULONG ulIndex;

               ulIndex=SHORT1FROMMP(mpParm1)-DMN_SLD_SLIDER1;

               //----------------------------------------------------------
               // Get the slider position and repaint.
               //----------------------------------------------------------
               pidData->alPoints[ulIndex]=
                  LONGFROMMR(WinSendDlgItemMsg(hwWnd,
                                               SHORT1FROMMP(mpParm1),
                                               SLM_QUERYSLIDERINFO,
                                               MPFROM2SHORT(SMA_SLIDERARMPOSITION,
                                                            SMA_INCREMENTVALUE),
                                               0));

               if (pidData->aulDetentes[ulIndex]!=0) {
                  WinSendDlgItemMsg(hwWnd,
                                    SHORT1FROMMP(mpParm1),
                                    SLM_REMOVEDETENT,
                                    MPFROMLONG(pidData->aulDetentes[ulIndex]),
                                    0);
               } /* endif */

               pidData->aulDetentes[ulIndex]=
                  LONGFROMMR(WinSendDlgItemMsg(hwWnd,
                                               SHORT1FROMMP(mpParm1),
                                               SLM_ADDDETENT,
                                               mpParm2,
                                               0));

               WinInvalidateRect(WinWindowFromID(hwWnd,DMN_ST_GRAPH),
                                 NULL,
                                 FALSE);
               WinUpdateWindow(WinWindowFromID(hwWnd,DMN_ST_GRAPH));
            }
            break;
         default:
            return WinDefDlgProc(hwWnd,ulMsg,mpParm1,mpParm2);
         } /* endswitch */
         break;
      default:
         return WinDefDlgProc(hwWnd,ulMsg,mpParm1,mpParm2);
      } /* endswitch */
      break;
   default:
      return WinDefDlgProc(hwWnd,ulMsg,mpParm1,mpParm2);
   } /* endswitch */

   return MRFROMSHORT(FALSE);
}

MRESULT EXPENTRY mainWndProc(HWND hwWnd,
                             ULONG ulMsg,
                             MPARAM mpParm1,
                             MPARAM mpParm2)
//-------------------------------------------------------------------------
// This is the client window procedure.
//
// Parameters:  (standard)
//-------------------------------------------------------------------------
{
   PINSTDATA pidData;

   pidData=(PINSTDATA)WinQueryWindowPtr(hwWnd,0);

   switch (ulMsg) {
   case WM_CREATE:
      //-------------------------------------------------------------------
      // Allocate memory for an initialize the instance data.
      //-------------------------------------------------------------------
      pidData=calloc(1,sizeof(INSTDATA));
      if (pidData==NULL) {
         WinAlarm(HWND_DESKTOP,WA_ERROR);
         return MRFROMSHORT(TRUE);
      } /* endif */

      WinSetWindowPtr(hwWnd,0,pidData);

      pidData->ulSzStruct=sizeof(INSTDATA);
      pidData->habAnchor=WinQueryAnchorBlock(hwWnd);
      pidData->hwFrame=WinQueryWindow(hwWnd,QW_PARENT);
      pidData->hwClient=hwWnd;

      //-------------------------------------------------------------------
      // Note the trick below.  Instead of calling WinCreateWindow()
      // for all of the controls needed, we create a modeless dialog
      // with us as the parent.  There are problems with this, however.
      // See the column for more information.
      //-------------------------------------------------------------------
      pidData->hwDlg=WinLoadDlg(hwWnd,
                                hwWnd,
                                mainDlgProc,
                                NULLHANDLE,
                                DLG_MAIN,
                                NULL);
      if (pidData->hwDlg==NULLHANDLE) {
         free(pidData);
         WinAlarm(HWND_DESKTOP,WA_ERROR);
         return MRFROMSHORT(TRUE);
      } /* endif */
      break;
   case WM_DESTROY:
      WinDestroyWindow(pidData->hwDlg);
      free(pidData);
      break;
   case WM_ERASEBACKGROUND:
      {
         HPS hpsPaint;
         PRECTL prclPaint;

         hpsPaint=(HPS)LONGFROMMP(mpParm1);
         prclPaint=(PRECTL)PVOIDFROMMP(mpParm2);

         WinFillRect(hpsPaint,prclPaint,SYSCLR_DIALOGBACKGROUND);
      }
      break;
   default:
      return WinDefWindowProc(hwWnd,ulMsg,mpParm1,mpParm2);
   } /* endswitch */

   return MRFROMSHORT(FALSE);
}

INT main(VOID)
//-------------------------------------------------------------------------
// This is the "standard" main().
//-------------------------------------------------------------------------
{
   HAB habAnchor;
   HMQ hmqQueue;
   ULONG ulCreate;
   HWND hwFrame;
   HWND hwClient;
   BOOL bLoop;
   QMSG qmMsg;

   habAnchor=WinInitialize(0);
   hmqQueue=WinCreateMsgQueue(habAnchor,0);

   ulCreate=FCF_SYSMENU | FCF_TITLEBAR | FCF_MINMAX | FCF_SIZEBORDER |
               FCF_ICON | FCF_TASKLIST | FCF_SHELLPOSITION;

   WinRegisterClass(habAnchor,CLS_CLIENT,mainWndProc,0,sizeof(PVOID));

   hwFrame=WinCreateStdWindow(HWND_DESKTOP,
                              0,
                              &ulCreate,
                              CLS_CLIENT,
                              "Slider Sample",
                              0,
                              NULLHANDLE,
                              RES_CLIENT,
                              &hwClient);
   if (hwFrame!=NULLHANDLE) {
      WinSetWindowPos(hwFrame,
                      NULLHANDLE,
                      0,
                      0,
                      400,
                      400,
                      SWP_SIZE|SWP_SHOW);

      bLoop=WinGetMsg(habAnchor,&qmMsg,NULLHANDLE,0,0);
      while (bLoop) {
         WinDispatchMsg(habAnchor,&qmMsg);
         bLoop=WinGetMsg(habAnchor,&qmMsg,NULLHANDLE,0,0);
      } /* endwhile */

      WinDestroyWindow(hwFrame);
   } /* endif */

   WinDestroyMsgQueue(hmqQueue);
   WinTerminate(habAnchor);
   return 0;
}
