/*****************************************************************************
 * $Id: bigpipe.c,v 2.8 1996/06/12 16:50:04 ak Exp $
 *****************************************************************************
 * $Log: bigpipe.c,v $
 * Revision 2.8  1996/06/12  16:50:04  ak
 * Consistent arguments.
 *
 * Revision 2.7  1996/06/12  16:43:24  ak
 * Added priority control.
 *
 * Revision 2.6  1995/07/11  00:50:19  ak
 * Bugfix pipe inheritance.
 *
 * Revision 2.5  1995/07/04  11:01:12  ak
 * Bugfix.
 *
 * Revision 2.4  1995/07/02  17:03:04  ak
 * Renamed to bigpipe.
 *
 * Revision 2.3  1995/07/02  16:55:01  ak
 * Thread closes its file handle on termination. Fixes.
 *
 * Revision 2.2  1995/07/02  16:12:16  ak
 * Pass parameters as argv-list. Run in background.
 *
 * Revision 2.1  1995/06/29  21:50:29  ak
 * Tagged as version 2 to distinguish from 16bit buffer utility.
 *
 * Revision 1.2  1995/06/29  21:41:28  ak
 * Changed freelist allocation to reduce working set in case of low
 * buffer usage.
 *
 * Revision 1.1  1995/06/29  19:35:05  ak
 * Initial revision
 *
 *****************************************************************************/

#if defined(OS2) && OS2 >= 2 && (defined(__MT__) || defined(__MULTI__))

static char *rcsid = "$Id: bigpipe.c,v 2.8 1996/06/12 16:50:04 ak Exp $";

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <math.h>
#include <process.h>
#include <signal.h>
#include <string.h>
#define INCL_BASE
#include <os2.h>

#include <libx.h>

#define BUFFERSIZE	(2048 * 1024)	/* default buffer size */
#define BLOCKSIZE	65536	     	/* max default blocksize */

typedef struct Block  Block;
typedef struct Handle Handle;

struct Block {
    Block *   volatile	next;		/* next used block */
    int       volatile	nbytes;		/* number of bytes in block buffer */
    char *		buffer;		/* block buffer */
};

struct Handle {
    Block *		blocks;		/* array of block descriptors */
    char *		buffer;		/* block buffers */

    int			nblocks;	/* number of blocks */
    int			blocksize;	/* size of a block */
    int			reblock;	/* reblock to a multiple of .. */
    
    HMTX		mutex;		/* protection semaphore */
    Block *   volatile	first;		/* first in-use block */
    Block *   volatile	last;		/* last in-use block */
    int       volatile	nused;		/* number of used blocks */

    int			rlevel;		/* reader threshold levels */
    int       volatile	rtid;		/* reader thread */
    int       volatile	rerrno;		/* reader error code */
    int       volatile	rsleep;		/* reader asleep */
    HEV			revent;		/* reader wakeup event */
    int			rprio[2];	/* reader priority */

    int			wlevel;		/* writer threshold levels */
    int       volatile	wtid;		/* writer thread */
    int       volatile	werrno;		/* writer error code */
    int       volatile	wsleep;		/* writer asleep */
    HEV			wevent;		/* writer wakeup event */
    int			wprio[2];	/* writer priority */

    int       volatile	eof;		/* end-of-file */

    int			pipes[4];	/* the whole pipe, 2=reader, 1=writer */
};

double
now(void)
{
    ULONG t;
    DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT, (PBYTE)&t, sizeof t);
    return (double)t / 1000.0;
}

static void
reader(void *arg)
{
    Handle *	buf = arg;
    Block *	blk;
    int		nr, nbytes, i;
    ULONG	n;

    DosSetPriority(PRTYS_THREAD, buf->rprio[0], buf->rprio[1], 0);

#ifdef __EMX__
    signal(SIGPIPE, SIG_IGN);
#endif

    while (!buf->eof) {

	/* check reader threshold */
	if (buf->wtid && buf->nused >= buf->rlevel) {
#ifdef DEBUG
	    DosEnterCritSec();
	    fprintf(stderr, "%.3f reader sleeping\n", now());
	    DosExitCritSec();
#endif
	    buf->rsleep = 1;
	    DosWaitEventSem(buf->revent, SEM_INDEFINITE_WAIT);
	    DosResetEventSem(buf->revent, &n);
	    buf->rsleep = 0;
	    continue;
	}

	/* fill blocks until buffer full */
	while (buf->nused < buf->nblocks && !buf->eof) {

	    DosRequestMutexSem(buf->mutex, SEM_INDEFINITE_WAIT);

	    /* get lowest numbered available block */
	    for (i = 0; i < buf->nblocks; ++i)
		if (buf->blocks[i].nbytes == 0)
		    break;
	    blk = &buf->blocks[i];

	    DosReleaseMutexSem(buf->mutex);

	    assert(i < buf->nblocks);

#ifdef DEBUG
	    DosEnterCritSec();
	    fprintf(stderr, "%.3f reader aquired block %d\n", now(), i);
	    DosExitCritSec();
#endif

	    /* fill block */
	    nbytes = 0;
	    do {
		nr = read(buf->pipes[2], blk->buffer + nbytes, buf->blocksize - nbytes);
		if (nr > 0)
		    nbytes += nr;
		if (nr < 0) {
		    buf->rerrno = errno;
		    buf->eof = 1;
		    break;
		}
	    } while (nr && nbytes < buf->blocksize);

#ifdef DEBUG
	    DosEnterCritSec();
	    fprintf(stderr, "%.3f reader read %d bytes\n", now(), nbytes);
	    DosExitCritSec();
#endif

	    if (nbytes) {
		/* need reblocking */
		if (buf->reblock && (nr = nbytes % buf->reblock) != 0) {
		    memset(blk->buffer + nbytes, 0, buf->reblock - nr);
		    nbytes += buf->reblock - nr;
		    buf->eof = 1;
		}
		blk->nbytes = nbytes;

		DosRequestMutexSem(buf->mutex, SEM_INDEFINITE_WAIT);

		/* append block to list of used blocks */
		blk->next = 0;
		if (buf->first)
		    buf->last->next = blk;
		else
		    buf->first = buf->last = blk;
		buf->last = blk;
		buf->nused += 1;

		/* wakeup writer */
		if (buf->wsleep && buf->nused >= buf->wlevel)
		    DosPostEventSem(buf->wevent);

		DosReleaseMutexSem(buf->mutex);

		DosSleep(0);
	    } else {
		/* we're done */
		buf->eof = 1;
	    }
	}
    }
    close(buf->pipes[2]);
    buf->pipes[2] = -1;
    buf->rtid = 0;
    DosPostEventSem(buf->wevent);
#ifdef DEBUG
    DosEnterCritSec();
    fprintf(stderr, "%.3f reader terminated\n", now());
    DosExitCritSec();
#endif
}

static void
writer(void *arg)
{
    Handle *	buf = arg;
    Block *	blk;
    int		nbytes, nw;
    ULONG	n;

    DosSetPriority(PRTYS_THREAD, buf->wprio[0], buf->wprio[1], 0);

#ifdef __EMX__
    signal(SIGPIPE, SIG_IGN);
#endif

    while (!buf->eof || buf->nused) {

	/* check writer threshold */
	if (buf->rtid && buf->nused < buf->wlevel) {
#ifdef DEBUG
	    DosEnterCritSec();
	    fprintf(stderr, "%.3f writer sleeping\n", now());
	    DosExitCritSec();
#endif
	    buf->wsleep = 1;
	    DosWaitEventSem(buf->wevent, SEM_INDEFINITE_WAIT);
	    DosResetEventSem(buf->wevent, &n);
	    buf->wsleep = 0;
	    continue;
	}

	/* flush used blocks */
	while (buf->nused) {

	    DosRequestMutexSem(buf->mutex, SEM_INDEFINITE_WAIT);

	    /* get next used block */
	    blk = buf->first;
	    buf->first = blk->next;

	    DosReleaseMutexSem(buf->mutex);

#ifdef DEBUG
	    DosEnterCritSec();
	    fprintf(stderr, "%.3f writer aquired block %d\n", now(), blk - buf->blocks);
	    DosExitCritSec();
#endif

	    /* flush block */
	    nbytes = 0;
	    do {
		nw = write(buf->pipes[1], blk->buffer + nbytes, blk->nbytes - nbytes);
		if (nw > 0)
		    nbytes += nw;
		if (nw < 0) {
		    buf->werrno = errno;
		    buf->eof = 1;
		    break;
		}
	    } while (nw && nw < blk->nbytes);

#ifdef DEBUG
	    DosEnterCritSec();
	    fprintf(stderr, "%.3f writer wrote %d bytes\n", now(), nw);
	    DosExitCritSec();
#endif

	    DosRequestMutexSem(buf->mutex, SEM_INDEFINITE_WAIT);

	    blk->nbytes = 0;
	    buf->nused -= 1;

	    /* wakeup reader */
	    if (buf->rsleep && buf->nused < buf->rlevel)
		DosPostEventSem(buf->revent);

	    DosReleaseMutexSem(buf->mutex);

	    DosSleep(0);
	}
    }
    close(buf->pipes[1]);
    buf->pipes[1] = -1;
    buf->wtid = 0;
    DosPostEventSem(buf->revent);
#ifdef DEBUG
    DosEnterCritSec();
    fprintf(stderr, "%.3f writer terminated\n", now());
    DosExitCritSec();
#endif
}

static void *
terminate(int rc, Handle *buf, int closef)
{
    int i;

#ifdef DEBUG
    DosEnterCritSec();
    fprintf(stderr, "%.3f terminating\n", now());
    DosExitCritSec();
#endif

    if (buf->mutex)
	DosCloseMutexSem(buf->mutex);
    if (buf->revent)
	DosCloseEventSem(buf->revent);
    if (buf->wevent)
	DosCloseEventSem(buf->wevent);
    if (closef)
	for (i = 0; i <= 3; ++i)
	    if (buf->pipes[i] >= 0)
		close(buf->pipes[i]);
    free(buf->buffer);
    free(buf->blocks);
    free(buf);
    errno = rc;
    return 0;
}

static void
priority(int prio[2], char *arg)
{
    switch (*arg++) {
    case 'i':
	prio[0] = PRTYC_IDLETIME;
	break;
    case 'r':
	prio[0] = PRTYC_REGULAR;
	break;
    case 'f':
	prio[0] = PRTYC_FOREGROUNDSERVER;
	break;
    case 't':
	prio[0] = PRTYC_TIMECRITICAL;
	break;
    default:
    	prio[0] = PRTYC_NOCHANGE;
	--arg;
    }

    prio[1] = isdigit(*arg) ? strtol(arg, NULL, 0) : 0;
}

static Handle *
initialize(Handle *buf, char **argv)
{
    int		i;
    char *	cp;
    long	buffersize = BUFFERSIZE;
    int		blocksize = 0, reblock = 0;
    int		nblocks;
    double	rlevel, wlevel;

    if ((cp = getenv("BUFFER")) != NULL)
	buffersize = strtoul(cp, NULL, 0) * 1024;
    rlevel = wlevel = 1.0;

    for (i = 0; argv[i] && argv[i][0] == '-'; ++i) {
	switch (argv[i][1]) {
	case 's':
	    buffersize = strtoul(argv[i]+2, NULL, 0) * 1024L;
	    continue;
	case 'b':
	    blocksize = reblock = strtoul(argv[i]+2, NULL, 0);
	    continue;
	case 'i':
	    rlevel = atof(argv[i]+2) / 100.0;
	    continue;
	case 'o':
	    wlevel = atof(argv[i]+2) / 100.0;
	    continue;
	case 'p':
	    if (isdigit(argv[i][2])) {
		priority(buf->rprio, argv[i]+2);
		priority(buf->wprio, argv[i]+2);
	    } else {
		priority(buf->rprio, "r31");
		priority(buf->wprio, "r31");
	    }
	    continue;
	}
	break;
    }

    if (blocksize <= 0) {
	blocksize = buffersize / 16;
	if (blocksize > BLOCKSIZE)
	    blocksize = BLOCKSIZE;
    }
    nblocks = buffersize / blocksize;
    if (nblocks < 2 || reblock > blocksize) {
	errno = EINVAL;
	free(buf);
	return 0;
    }

    buf->blocks = calloc(nblocks, sizeof(Block));
    if (!buf->blocks) {
	free(buf);
	errno = ENOMEM;
	return 0;
    }

    buf->buffer = malloc(nblocks * blocksize);
    if (!buf->buffer) {
	free(buf->blocks);
	free(buf);
	errno = ENOMEM;
	return 0;
    }

    buf->nblocks   = nblocks;
    buf->blocksize = blocksize;
    buf->reblock   = reblock;

    buf->rlevel = (1.0 - rlevel) * nblocks;
    buf->wlevel = wlevel * nblocks;

    if (buf->rlevel < 1)
	buf->rlevel = 1;
    else if (buf->rlevel > nblocks)
	buf->rlevel = nblocks;

    if (buf->wlevel < 1)
	buf->wlevel = 1;
    else if (buf->wlevel > nblocks)
	buf->wlevel = nblocks;
    for (i = 0; i < nblocks; ++i)
	buf->blocks[i].buffer = buf->buffer + i * blocksize;

    if (DosCreateMutexSem(0, &buf->mutex, 0, 0) == 0
     && DosCreateEventSem(0, &buf->revent, 0, 0) == 0
     && DosCreateEventSem(0, &buf->wevent, 0, 0) == 0)
	return buf;

    return terminate(ENOMEM, buf, 1);
}

/*****************************************************************************/

static Handle *
start_buffer(Handle *buf, char **argv)
{
    int tid;

#ifdef DEBUG
    fprintf(stderr, "pipes: %d -> %d -> bigpipe -> %d -> %d\n",
	    buf->pipes[3], buf->pipes[2], buf->pipes[1], buf->pipes[0]);
#endif

    if (!initialize(buf, argv))
	return 0;

    tid = _beginthread(reader, 0, 16384, buf);
    if (tid < 0)
	return terminate(errno, buf, 1);
    buf->rtid = tid;

    tid = _beginthread(writer, 0, 16384, buf);
    if (tid < 0) {
	int saveerrno = errno;
	buf->eof = 1;
	while (buf->rtid) {
	    DosPostEventSem(buf->revent);
	    DosSleep(1);
	}
	return terminate(saveerrno, buf, 1);
    }
    buf->wtid = tid;

    return buf;
}
	
void *
pipe_buffer(int rfd, int wfd, char **argv)
{
    Handle *	buf;

    buf = calloc(1, sizeof(Handle));
    if (!buf) {
	errno = ENOMEM;
	return 0;
    }
    buf->pipes[3] = -1;
    buf->pipes[2] = rfd;
    buf->pipes[1] = wfd;
    buf->pipes[0] = -1;

    return start_buffer(buf, argv);
}

void *
pipe_start(int fds[2], char **argv)
{
    Handle *	buf;
    int		i;

    buf = calloc(1, sizeof(Handle));
    if (!buf) {
	errno = ENOMEM;
	return 0;
    }
    for (i = 0; i < 4; ++i)
	buf->pipes[i] = -1;

    if (pipe(buf->pipes+0) < 0 || pipe(buf->pipes+2) < 0)
	return terminate(errno, buf, 1);

    fds[1] = buf->pipes[3];
    fds[0] = buf->pipes[0];

    noinherit(buf->pipes[2]);
    noinherit(buf->pipes[1]);

    return start_buffer(buf, argv);
}

int
pipe_end(void *bufp, int abortf)
{
    Handle *	buf = bufp;
    int		rc;
    TID		tid;

    if (abortf)
	buf->eof = 1;

    if (tid = buf->rtid) {
#ifdef DEBUG
	fprintf(stderr, "%.3f waiting for reader\n", now());
#endif
	DosPostEventSem(buf->revent);
	DosWaitThread(&tid, DCWW_WAIT);
    }

    if (tid = buf->wtid) {
#ifdef DEBUG
	fprintf(stderr, "%.3f waiting for writer\n", now());
#endif
	DosPostEventSem(buf->wevent);
	DosWaitThread(&tid, DCWW_WAIT);
    }

    rc = 0;
    if (buf->rerrno)
	rc = buf->rerrno;
    else if (buf->werrno)
	rc = buf->werrno;

    terminate(rc, buf, 0);
    return rc;
}

/*****************************************************************************/
#ifdef MAIN

#include <getopt.h>

void
setFail(int fd)
{
    ULONG state;

    if (DosQueryFHState(fd, &state) == 0) {
	state &=~OPEN_FLAGS_RANDOMSEQUENTIAL;
	state |= OPEN_FLAGS_SEQUENTIAL | OPEN_FLAGS_NO_CACHE | OPEN_FLAGS_FAIL_ON_ERROR;
	DosSetFHState(fd, state);
    }
}

void
usage(void)
{
    fprintf(stderr,
"buffer [options]\n"
" -s<n>		buffer size in KB, default=%u\n"
" -b<n>		block size for reblocking (bytes)\n"
" -i<n>		read when buffer <n>%% free, default=100\n"
" -o<n>		write when buffer <n>%% full, default=100\n",
" -p[irft][<n>]	set priority\n",
	    BUFFERSIZE / 1024);
    exit(2);
}

int
main(int argc, char **argv)
{
    int		i;
    int		rfd = 0, wfd = 1;
    void *	handle;

    setFail(0);
    setFail(1);

    for (i = 1; i < argc && argv[i][0] == '-'; ++i)
	if (!strchr("sbiop", argv[i][1]))
	    usage();

    if (i < argc) {
	if (strcmp(argv[i], "-") != 0) {
	    rfd = open(argv[i], O_RDONLY|O_BINARY, 0);
	    if (rfd < 0) {
		perror(argv[i]);
		exit(1);
	    }
	}
	++i;
    }
    if (i < argc) {
	if (strcmp(argv[i], "-") != 0) {
	    wfd = open(argv[i], O_CREAT|O_TRUNC|O_WRONLY|O_BINARY, 0777);
	    if (wfd < 0) {
		perror(argv[i]);
		exit(1);
	    }
	}
	++i;
    }

    setmode(rfd, O_BINARY);
    setmode(wfd, O_BINARY);

    handle = pipe_buffer(rfd, wfd, argv+1);
    if (!handle) {
	perror(argv[0]);
	exit(1);
    }
    i = pipe_end(handle, 0);

    if (rfd != 0)
	close(rfd);
    if (wfd != 1)
	close(wfd);

    return (i < 0) ? 1 : 0;
}

#endif /* MAIN */

#endif /* OS2... */

